/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.api.task;

/** for advanced task customization.
 * 
 * The default implementation can be found in {@link DefaultTaskHandler}.
 * 
 * @author Tom Baeyens
 */
public class TaskHandler {

  /** called after the task is created according 
   * to the task definition.  This method allows to 
   * create subtasks.
   * @return indicates if the process execution should wait. */
  public boolean executionCreateTask(OpenTask task) {
    task.historyTaskStart();
    return true;
  }
  
  /** called when the related execution gets a 
   * signal.  this method must either 
   * {@link OpenTask#setSignalling(boolean) remove signalling} from this task
   * or {@link OpenTask#cancel(String) cancel} it. */
  public void executionSignal(OpenTask task) {
    task.setSignalling(false);
  }
  
  /** called when the given task is assigned to an actor. */
  public void taskAssign(OpenTask task, String userId) {
    task.historyTaskAssign(userId);
  }

  /** called when the given task is cancelled. */ 
  public void taskCancel(OpenTask task, String reason) {
    task.historyTaskCancel(reason);
  }
  
  /** called when the given task completes.  The default behaviour 
   * will send a signal to the execution if this task is still signalling. */
  public void taskComplete(OpenTask task, String outcome) {
    task.historyTaskComplete(outcome);
    
    if (task.isSignalling()) {
      task.getExecution().signal(outcome);
    }
  }

  /** is called when a subtask completes.  this can be used to 
   * prematurely signal the execution. e.g. for "5 out of 7" scenarios.*/
  public void taskSubTaskComplete(OpenTask task, OpenTask subTask, String outcome) {
  }
  
  /** is called when a variable is updated.  This can be used to 
   * propagate the execution based on availability of variables. */
  public void taskVariableUpdate(OpenTask task, String key, Object value) {
  }
}
