/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.bpmn.flownodes;

import java.util.ArrayList;
import java.util.List;

import org.jbpm.api.Execution;
import org.jbpm.api.activity.ActivityBehaviour;
import org.jbpm.bpmn.parser.BindingsParser;
import org.jbpm.internal.log.Log;
import org.jbpm.pvm.internal.model.Activity;
import org.jbpm.pvm.internal.model.Condition;
import org.jbpm.pvm.internal.model.ExecutionImpl;
import org.jbpm.pvm.internal.model.Transition;

/**
 * Basic activity for BPMN activities (tasks, gateways and event)
 * 
 * @author bernd.ruecker@camunda.com
 * @author Ronald van Kuijk (kukeltje)
 */
public abstract class BpmnActivity implements ActivityBehaviour {

  private static final Log log = Log.getLog(BpmnActivity.class.getName());

  private static final long serialVersionUID = 1L;

  protected static final boolean FORK_ALLOWED = true;
  protected static final boolean FORK_DISALLOWED = !FORK_ALLOWED;
  protected static final boolean CONDITIONS_CHECKED = true;
  protected static final boolean CONDITIONS_IGNORED = !CONDITIONS_CHECKED;

  protected List<ActivityResource> activityResources = new ArrayList<ActivityResource>();

  /**
   * In BPMN multiple outgoing sequence flows behave like a fork.
   * 
   * Code copied basically from jPDL fork.
   */
  protected void proceed(ExecutionImpl execution, List<Transition> transitions) {
	if (log.isDebugEnabled()) {		
		log.debug("Proceeding from execution " + execution.getActivityName());
	}

    Activity activity = execution.getActivity();

    // if no outgoing transitions should be forked,
    if (transitions.size() == 0) {
      execution.end();
    }

    // if there is exactly 1 transition to be taken, just use the incoming
    // execution
    else if (transitions.size() == 1) {
      execution.take(transitions.get(0));

      // if there are more transitions
    } else {
      ExecutionImpl concurrentRoot = null;
      if (Execution.STATE_ACTIVE_ROOT.equals(execution.getState())) {
        concurrentRoot = execution;
        execution.setState(Execution.STATE_INACTIVE_CONCURRENT_ROOT);
        execution.setActivity(null);
      } else if (Execution.STATE_ACTIVE_CONCURRENT.equals(execution.getState())) {
        concurrentRoot = execution.getParent();
      }

      for (Transition transition : transitions) {
        // launch a concurrent path of execution
        String childExecutionName = transition.getName();
        ExecutionImpl concurrentExecution = concurrentRoot.createExecution(childExecutionName);
        concurrentExecution.setActivity(activity);
        concurrentExecution.setState(Execution.STATE_ACTIVE_CONCURRENT);
        concurrentExecution.take(transition);

        if (concurrentRoot.isEnded()) {
          break;
        }
      }
    }
  }

  protected List<Transition> findTransitions(ExecutionImpl execution, boolean checkConditions) {
    Activity activity = execution.getActivity();
    // evaluate the conditions and find the transitions that should be forked
    List<Transition> forkingTransitions = new ArrayList<Transition>();
    List<Transition> outgoingTransitions = activity.getOutgoingTransitions();
    for (Transition transition : outgoingTransitions) {
      Condition condition = transition.getCondition();
      // also ignore the default transition of the exclusive gateway
      if ( ( (condition == null) 
             || (!checkConditions) 
             || (condition.evaluate(execution))
           ) 
           && (activity.getDefaultOutgoingTransition() != transition)
         ) {
        forkingTransitions.add(transition);
      }
    }
    if (log.isDebugEnabled()) {
    	log.debug(forkingTransitions.size() + " out of " + outgoingTransitions.size() + " selected for " + activity.getName());
    }
    return forkingTransitions;
  }

  public void addActivityResource(ActivityResource activityResource) {
    this.activityResources.add(activityResource);
  }

}