/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.bpmn.parser;

import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

import javax.xml.namespace.QName;

import org.jbpm.api.JbpmException;
import org.jbpm.api.activity.ActivityBehaviour;
import org.jbpm.bpmn.common.Resource;
import org.jbpm.bpmn.common.ResourceParameter;
import org.jbpm.bpmn.flownodes.BpmnBinding;
import org.jbpm.bpmn.flownodes.ExclusiveGatewayActivity;
import org.jbpm.bpmn.model.BpmnProcessDefinition;
import org.jbpm.internal.log.Log;
import org.jbpm.pvm.internal.model.ActivityImpl;
import org.jbpm.pvm.internal.model.CompositeElementImpl;
import org.jbpm.pvm.internal.model.ExpressionCondition;
import org.jbpm.pvm.internal.model.ProcessDefinitionImpl;
import org.jbpm.pvm.internal.model.ScopeElementImpl;
import org.jbpm.pvm.internal.model.TransitionImpl;
import org.jbpm.pvm.internal.model.VariableDefinitionImpl;
import org.jbpm.pvm.internal.task.TaskDefinitionImpl;
import org.jbpm.pvm.internal.util.TagBinding;
import org.jbpm.pvm.internal.util.XmlUtil;
import org.jbpm.pvm.internal.wire.binding.ObjectBinding;
import org.jbpm.pvm.internal.wire.xml.WireParser;
import org.jbpm.pvm.internal.xml.Bindings;
import org.jbpm.pvm.internal.xml.Parse;
import org.jbpm.pvm.internal.xml.Parser;
import org.w3c.dom.Element;

/**
 * @author Tom Baeyens
 * @author Bernd Ruecker (bernd.ruecker@camunda.com)
 */
public class BpmnParser extends Parser {

  private static final Log log = Log.getLog(BpmnParser.class.getName());

  static ObjectBinding objectBinding = new ObjectBinding();
  public static final WireParser wireParser = WireParser.getInstance();

  static final String[] DEFAULT_ACTIVITIES_RESOURCES = { "jbpm.bpmn.flownodes.xml" };
  static final String CATEGORY_ACTIVITY = "activity";

  static BindingsParser bindingsParser = new BindingsParser();

  public BpmnParser() {
    
    initialize(); // initialises underlying SAX parser
    parseBindings(); // initialises bindings
    
    // Setting BPMN2 xsd schema
    List<String> schemaResources = new ArrayList<String>();
    schemaResources.add("BPMN20.xsd");
    setSchemaResources(schemaResources);
    
  }

  public Object parseDocumentElement(Element documentElement, Parse parse) {
    List<ProcessDefinitionImpl> processDefinitions = new ArrayList<ProcessDefinitionImpl>();

    parseDefinition(documentElement, parse);

    for (Element processElement : XmlUtil.elements(documentElement, "process")) {
      ProcessDefinitionImpl processDefinition = parseProcess(processElement, parse);
      processDefinitions.add(processDefinition);
    }

    return processDefinitions;
  }

  public ProcessDefinitionImpl parseProcess(Element processElement, Parse parse) {
    BpmnProcessDefinition processDefinition = new BpmnProcessDefinition();

    parse.contextStackPush(processDefinition);
    try {

      // process attribues
      String name = XmlUtil.attribute(processElement, "name", true, parse);
      processDefinition.setName(name);

      String key = XmlUtil.attribute(processElement, "id", false, parse);
      if (key != null) {
        processDefinition.setKey(key);
      }

      Element descriptionElement = XmlUtil.element(processElement, "documentation");
      if (descriptionElement != null) {
        String description = XmlUtil.getContentText(descriptionElement);
        processDefinition.setDescription(description);
      }

      // TODO: should be done in a different way? On a different level?
      parseResources((Element)processElement.getParentNode(), parse, processDefinition);
      
      parseInterfaces((Element)processElement.getParentNode(), parse, processDefinition);
      
      parseItemDefinitions((Element)processElement.getParentNode(), parse, processDefinition);

      parseMessages((Element)processElement.getParentNode(), parse, processDefinition);

      

      parseDataObjects(processElement, parse, processDefinition);

      // activities
      parseActivities(processElement, parse, processDefinition);

      // bind activities to their destinations
      parseSequenceFlows(processElement, parse, processDefinition);

    } finally {
      parse.contextStackPop();
    }

    return processDefinition;
  }

  // /////////////////////////////////////////////////////////////////////////////////////////

  protected void parseBindings() {
    Bindings bindings = new Bindings();
    setBindings(bindings);

    for (String activityResource : DEFAULT_ACTIVITIES_RESOURCES) {
      Enumeration<URL> resourceUrls = getResources(activityResource);
      if (resourceUrls.hasMoreElements()) {
        while (resourceUrls.hasMoreElements()) {
          URL resourceUrl = resourceUrls.nextElement();
          log.trace("loading bpmn activities from resource: " + resourceUrl);
          List<BpmnBinding> activityBindings = (List<BpmnBinding>) bindingsParser.createParse().setUrl(resourceUrl).execute().checkErrors(
                  "bpmn activities from " + resourceUrl.toString()).getDocumentObject();

          for (TagBinding binding : activityBindings) {
            binding.setCategory(CATEGORY_ACTIVITY);
            bindings.addBinding(binding);
          }
        }
      } else {
        log.trace("skipping unavailable activities resource: " + activityResource);
      }
    }
  }

  protected Enumeration<URL> getResources(String resourceName) {
    ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
    Enumeration<URL> resourceUrls;
    try {
      resourceUrls = classLoader.getResources(resourceName);
    } catch (Exception e) {
      throw new JbpmException("couldn't get resource urls for "+resourceName, e);
    }
    return resourceUrls;
  }

  private void parseDataObjects(Element element, Parse parse, BpmnProcessDefinition processDefinition) {
    List<VariableDefinitionImpl> variableDefinitions = new ArrayList<VariableDefinitionImpl>();

    for (Element dataElement : XmlUtil.elements(element, "dataObject")) {
      VariableDefinitionImpl variableDefinition = new VariableDefinitionImpl();

      String name = XmlUtil.attribute(dataElement, "id", true, parse);
      variableDefinition.setName(name);

      String typeRef = XmlUtil.attribute(dataElement, "itemSubjectRef");
      variableDefinition.setTypeName(processDefinition.getType(typeRef));

      variableDefinitions.add(variableDefinition);
    }

    processDefinition.setVariableDefinition(variableDefinitions);
  }

  public void parseActivities(Element element, Parse parse, CompositeElementImpl compositeElement) {
    List<Element> elements = XmlUtil.elements(element);
    for (Element nestedElement : elements) {
      String tagName = XmlUtil.getTagLocalName(nestedElement);
      String name = XmlUtil.attribute(nestedElement, "name", false, parse);
      String id = XmlUtil.attribute(nestedElement, "id", true, parse);

      TagBinding activityBinding = (TagBinding) getBinding(nestedElement, CATEGORY_ACTIVITY);
      if (activityBinding == null) {
        if (!"sequenceFlow".equals(tagName)) {
          log.debug("unrecognized activity: " + tagName);
        }
        continue;
      }

      ActivityImpl activity = compositeElement.createActivity();
      parse.contextStackPush(activity);
      try {
        activity.setType(activityBinding.getTagName());
        activity.setName(id);
        activity.setDescription(name);
        log.debug("Parse Activity: " + name);
        ActivityBehaviour activityBehaviour = (ActivityBehaviour) activityBinding.parse(nestedElement, parse, this);
        activity.setActivityBehaviour(activityBehaviour);
      } finally {
        parse.contextStackPop();
      }
    }
  }

  public void parseSequenceFlows(Element element, Parse parse, CompositeElementImpl compositeElement) {
    List<Element> transitionElements = XmlUtil.elements(element, "sequenceFlow");
    for (Element transitionElement : transitionElements) {
      String transitionName = XmlUtil.attribute(transitionElement, "name", false, parse);
      String transitionId = XmlUtil.attribute(transitionElement, "id", false, parse); // id is not required
      String sourceRef = XmlUtil.attribute(transitionElement, "sourceRef", true, parse);
      String targetRef = XmlUtil.attribute(transitionElement, "targetRef", true, parse);

      log.trace(transitionId + ": " + sourceRef + " -> " + targetRef);
      Element conditionElement = XmlUtil.element(transitionElement, "conditionExpression");
      log.trace("    with " + ((conditionElement == null) ? "0" : "1") + " conditionExpression");

      TransitionImpl transition = compositeElement.findActivity(sourceRef).createOutgoingTransition();
      

      try {
        // If something went wrong parsing the activity, there is no behaviour and an exception is thrown in .getBehaviour()
        ActivityBehaviour a = compositeElement.findActivity(sourceRef).getActivityBehaviour();
        if (a instanceof ExclusiveGatewayActivity) {
          if (transitionId.equals(((ExclusiveGatewayActivity) a).getDefault())) {
            compositeElement.findActivity(sourceRef).setDefaultOutgoingTransition(transition);
          }
        } else {
          // Other flownodes do not have default sequenceFlows, so set it to null
          compositeElement.findActivity(sourceRef).setDefaultOutgoingTransition(null);
        }
      } catch (JbpmException je) {
        // catch it and only re-throw if not this specific exception.
        if (!je.getMessage().contains("no behaviour on")) {
          throw je;
        }
      }

      if (conditionElement != null) {
        String type = conditionElement.getAttributeNS("http://www.w3.org/2001/XMLSchema-instance", "type");
        if ("bpmn:tFormalExpression".equals(type) || "tFormalExpression".equals(type)) {

          String expr = conditionElement.getTextContent();
          String lang = XmlUtil.attribute(conditionElement, "language");
          // TODO: add looking up the default language in the document under definitions if lang  is null.
          
          ExpressionCondition expressionCondition = new ExpressionCondition();
          expressionCondition.setExpression(expr);
          expressionCondition.setLanguage(lang);
          transition.setCondition(expressionCondition);

        } else {
          parse.addProblem("Type of the conditionExpression on sequenceFlow with id=" + transitionId + " is of onsupported type 'bpmn:tExpression'",
                  transitionElement);
        }
      }

      compositeElement.findActivity(targetRef).addIncomingTransition(transition);

      transition.setName(transitionId);
      transition.setDescription(transitionName);
    }
  }
  public void parseDefinition(Element documentElement, Parse parse) {
    parseImports(documentElement, parse);
    ///parseItemDefinitions(documentElement, parse);
    //parseMessages(documentElement, parse);
    // parseInterfaces(documentElement, parse);
    // parseResources(documentElement, parse);
  }

  // public static void parseAssignmentAttributes(Element element,
  // AssignableDefinitionImpl assignableDefinition, Parse parse) {
  // Element descriptionElement = XmlUtil.element(element, "description");
  // if (descriptionElement!=null) {
  // String description = XmlUtil.getContentText(descriptionElement);
  // assignableDefinition.setDescription(description);
  // }
  //  
  // Element assignmentHandlerElement = XmlUtil.element(element,
  // "assignment-handler");
  // if (assignmentHandlerElement!=null) {
  // ObjectDescriptor objectDescriptor =
  // parseObjectDescriptor(assignmentHandlerElement, parse);
  // assignableDefinition.setAssignmentHandlerDescriptor(objectDescriptor);
  // }
  //  
  // String assigneeExpression = XmlUtil.attribute(element, "assignee");
  // assignableDefinition.setAssigneeExpression(assigneeExpression);
  //    
  // String assigneeExpressionLanguage = XmlUtil.attribute(element,
  // "assignee-lang");
  // assignableDefinition.setAssigneeExpressionLanguage(assigneeExpressionLanguage);
  //    
  // String candidateUsersExpression = XmlUtil.attribute(element,
  // "candidate-users");
  // assignableDefinition.setCandidateUsersExpression(candidateUsersExpression);
  //    
  // String candidateUsersExpressionLanguage = XmlUtil.attribute(element,
  // "candidate-users-lang");
  // assignableDefinition.setCandidateUsersExpressionLanguage(candidateUsersExpressionLanguage);
  //    
  // String candidateGroupsExpression = XmlUtil.attribute(element,
  // "candidate-groups");
  // assignableDefinition.setCandidateGroupsExpression(candidateGroupsExpression);
  //    
  // String candidateGroupsExpressionLanguage = XmlUtil.attribute(element,
  // "candidate-groups-lang");
  // assignableDefinition.setCandidateGroupsExpressionLanguage(candidateGroupsExpressionLanguage);
  // }

  public static TaskDefinitionImpl parseTaskDefinition(Element element, Parse parse, ScopeElementImpl scopeElement) {
    TaskDefinitionImpl taskDefinition = new TaskDefinitionImpl();

    String taskName = XmlUtil.attribute(element, "name");
    taskDefinition.setName(taskName);

    BpmnProcessDefinition processDefinition = parse.contextStackFind(BpmnProcessDefinition.class);
    if (processDefinition.getTaskDefinition(taskName) != null) {
      parse.addProblem("duplicate task name " + taskName, element);
    } else {
      processDefinition.addTaskDefinitionImpl(taskDefinition);
    }

    return taskDefinition;
  }


  private void parseResources(Element documentElement, Parse parse, BpmnProcessDefinition processDefinition) {

    for (Element resourceElement : XmlUtil.elements(documentElement, "resource")) {

      Resource resource = new Resource();

      resource.setId(XmlUtil.attribute(resourceElement, "id"));
      resource.setName(XmlUtil.attribute(resourceElement, "name"));

      for (Element resourceParameterElement : XmlUtil.elements(documentElement, "resourceParameter")) {

        ResourceParameter resourceParameter = new ResourceParameter();
        resourceParameter.setId(XmlUtil.attribute(resourceParameterElement, "id"));
        resourceParameter.setName(XmlUtil.attribute(resourceParameterElement, "name"));
        resourceParameter.setType(QName.valueOf(XmlUtil.attribute(resourceParameterElement, "name")));

        resource.getParameters().put(XmlUtil.attribute(resourceParameterElement, "name"), resourceParameter);
      }

      processDefinition.getResources().put(resource.getName(), resource);
    }

  }

  private void parseInterfaces(Element documentElement, Parse parse, BpmnProcessDefinition processDefinition) {
    
    for (Element interfaceElement : XmlUtil.elements(documentElement, "interface")) {
      for (Element operationElement : XmlUtil.elements(interfaceElement, "operation")) {
        processDefinition.getOperations().put(XmlUtil.attribute(operationElement, "id"), operationElement);
      }
      processDefinition.getInterfaces().put(XmlUtil.attribute(interfaceElement, "id"), interfaceElement);
    }
  }

  private void parseMessages(Element documentElement, Parse parse, BpmnProcessDefinition processDefinition) {

    for (Element messageElement : XmlUtil.elements(documentElement, "message")) {
      processDefinition.getMessages().put(XmlUtil.attribute(messageElement, "id"), messageElement);
    }
  }

  private void parseItemDefinitions(Element documentElement, Parse parse, BpmnProcessDefinition processDefinition) {

    for (Element itemDefinitionElement : XmlUtil.elements(documentElement, "itemDefinition")) {
      processDefinition.getItemDefinitions().put(XmlUtil.attribute(itemDefinitionElement, "id"), itemDefinitionElement);
    }

  }

  private void parseImports(Element documentElement, Parse parse) {

  }
  
}
