/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.integration.console.forms;

import org.jboss.bpm.console.server.plugin.TaskDispatcherPlugin;
import org.jbpm.api.*;
import org.jbpm.api.model.Transition;
import org.jbpm.api.model.OpenExecution;
import org.jbpm.api.task.Task;
import org.jbpm.api.env.EnvironmentFactory;
import org.jbpm.api.env.Environment;
import org.jbpm.integration.spi.mgmt.ServerConfig;
import org.jbpm.integration.spi.mgmt.ServerConfigFactory;
import org.jbpm.pvm.internal.task.TaskImpl;
import org.jbpm.pvm.internal.model.ExecutionImpl;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.naming.InitialContext;
import java.net.MalformedURLException;
import java.net.URL;
import java.io.*;
import java.util.Set;
import java.util.Map;
import java.util.HashMap;
import java.util.List;

import freemarker.template.Template;
import freemarker.template.DefaultObjectWrapper;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class TaskDispatcherPluginImpl implements TaskDispatcherPlugin
{
  private ProcessEngine processEngine;

  private ServerConfig serverConfig = null;

  public TaskDispatcherPluginImpl()
  {
    initializeProcessEngine();
  }

  public URL getDispatchUrl(long taskId)
  {
    StringBuilder spec = new StringBuilder();
    spec.append("http://");
    spec.append(getServerConfig().getWebServiceHost());
    spec.append(":").append(getServerConfig().getWebServicePort());
    spec.append("/gwt-console-server/rs/task/");
    spec.append( taskId );
    spec.append("/render");

    try
    {
      return new URL(spec.toString());
    }
    catch (MalformedURLException e)
    {
      throw new RuntimeException("Failed to resolve task dispatch url", e);
    }
  }


  private ServerConfig getServerConfig()
  {
    if(null==serverConfig)
    {
      serverConfig = ServerConfigFactory.getServerConfig();
    }
    return serverConfig;
  }

  public DataHandler provideTaskUI(long taskId)
  {
    Environment env = ((EnvironmentFactory)processEngine).openEnvironment();

    DataHandler result = null;

    try
    {
      TaskService taskService = processEngine.getTaskService();
      Task task = taskService.getTask(taskId);

      // access the processdefition
      TaskImpl cast = ((TaskImpl) task);
      ExecutionImpl processInstance = cast.getProcessInstance();
      String processInstanceId =  processInstance.getId();
      String processId =  processInstance.getProcessDefinition().getId();

      RepositoryService repoService = processEngine.getRepositoryService();
      ProcessDefinitionQuery query = repoService.createProcessDefinitionQuery();
      query.processDefinitionId(processId);
      ProcessDefinition procDef = query.uniqueResult();

      // check if a template exists
      String name = task.getName() + ".ftl";
      InputStream template = repoService.getResourceAsStream(
          procDef.getDeploymentDbid(), name
      );

      // merge template with process variables
      if(template!=null)
      {
        ExecutionService execService = processEngine.getExecutionService();

        Set<String> varNames = execService.getVariableNames(processInstanceId);
        if(varNames!=null)
        {
          Map<String, Object> processContext = execService.getVariables(processInstanceId, varNames);

          // plugin context

          StringBuilder action = new StringBuilder();
          action.append("http://");
          action.append(getServerConfig().getWebServiceHost());
          action.append(":").append(getServerConfig().getWebServicePort());
          action.append("/gwt-console-server/rs/task/");
          action.append( taskId );
          action.append("/process");

          Map<String, Object> renderContext = new HashMap<String,Object>();

          // form directive
          FormDirective formDirective = new FormDirective();
          formDirective.setAction( action.toString() );
          renderContext.put("form", formDirective);

          // outcome directive
          // TODO: Fix when https://jira.jboss.org/jira/browse/JBPM-2220 is done
          OutcomeDirective outcomeDirective = new OutcomeDirective();
          List<Transition> transitions =
              ((ExecutionImpl) processInstance).getActivity().getOutgoingTransitions();
          for(Transition t : transitions)
          {
            String outcomeName = t.getName()!=null ? t.getName() : "to_"+t.getDestination().getName();
            outcomeDirective.getValues().add(outcomeName);
          }
          renderContext.put("outcome",outcomeDirective);          

          // process variables
          renderContext.putAll(processContext);


          result = processTemplate(name, template, renderContext);
        }
      }

      return result;
    }
    finally{
      env.close();
    }
  }

  private DataHandler processTemplate(
      final String name, InputStream src,
      Map<String, Object> renderContext
      )
  {
    DataHandler merged = null;

    try
    {
      freemarker.template.Configuration cfg = new freemarker.template.Configuration();      
      cfg.setObjectWrapper(new DefaultObjectWrapper());
      cfg.setTemplateUpdateDelay(0);

      Template temp = new Template(name, new InputStreamReader(src), cfg);
      temp.dump(System.out);

      final ByteArrayOutputStream bout = new ByteArrayOutputStream();
      Writer out = new OutputStreamWriter(bout);
      temp.process(renderContext, out);
      out.flush();
       
      merged = new DataHandler(

          new DataSource()
          {

            public InputStream getInputStream() throws IOException
            {
              return new ByteArrayInputStream(bout.toByteArray());
            }

            public OutputStream getOutputStream() throws IOException
            {
              return bout;
            }

            public String getContentType()
            {
              return "*/*";
            }

            public String getName()
            {
              return name + "_DataSource";
            }
          }
      );

    }
    catch (Exception e)
    {
      throw new RuntimeException("Failed to process task template", e);
    }

    return merged;
  }

  public void processCompletion(long taskId, String outcome, InputStream payload)
  {

  }

  protected void initializeProcessEngine()
  {
    try
    {
      InitialContext ctx = new InitialContext();
      this.processEngine = (ProcessEngine)ctx.lookup("java:/ProcessEngine");
    }
    catch (Exception e)
    {
      throw new RuntimeException("Failed to lookup process engine", e);
    }
  }

}
