/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.integration.jboss4;

import org.jboss.deployment.DeploymentException;
import org.jboss.deployment.DeploymentInfo;
import org.jboss.deployment.SubDeployer;
import org.jboss.deployment.SubDeployerSupport;
import org.jbpm.integration.spi.DeploymentAdaptor;
import org.jbpm.integration.spi.JBPMDeploymentMetaData;
import org.jbpm.integration.spi.DeploymentRef;
import org.jbpm.internal.log.Log;

import java.util.List;
import java.util.Iterator;
import java.util.ArrayList;

/**
 * An AS 4.2.x deployer for jBPM4 process archives.
 * Delegates to {@link org.jbpm.integration.spi.DeploymentAdaptor}.
 * The deployer retains a {@link org.jbpm.integration.spi.DeploymentRef} 
 * for subsequent undeployment calls.
 * 
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class JBPMDeployer extends SubDeployerSupport
    implements SubDeployer, JBPMDeployerMBean
{

  private static final Log log = Log.getLog(JBPMDeployer.class.getName());

  private DeploymentAdaptor adaptor;
  private static final String ARCHIVE_SUFFIX = ".bar";
  private static final String JPDL_FILE_SUFFIX = ".jpdl.xml";
  private static final String CONTEXT_REFERENCE = "jbpm.deployer.reference";

  /**
   * Default CTOR used to set default values to the Suffixes and RelativeOrder
   * attributes. Those are read at subdeployer registration time by the MainDeployer
   * to alter its SuffixOrder.
   */

  public JBPMDeployer()
  {
    initializeMainDeployer();
    this.adaptor = new DeploymentAdaptor();   
  }

  /**
   * Set the suffixes and relative order attributes.
   *
   * Those are read at subdeployer registration time by the MainDeployer
   * to update its SuffixOrder list.
   */
  protected void initializeMainDeployer()
  {
    setSuffixes(new String[]{ARCHIVE_SUFFIX, JPDL_FILE_SUFFIX});
    setRelativeOrder(1000); // Make sure it kicks in after the other deployers
  }

  /**
   * Returns true if this deployer can deploy the given DeploymentInfo.
   *
   * @return True if this deployer can deploy the given DeploymentInfo.   
   */
  public boolean accepts(DeploymentInfo di)
  {
    String urlStr = di.url.toString();
    return urlStr.endsWith(ARCHIVE_SUFFIX) || urlStr.endsWith(JPDL_FILE_SUFFIX);
  }


  public void init(DeploymentInfo deploymentInfo) throws DeploymentException
  {
    super.init(deploymentInfo);
  }

  public void create(DeploymentInfo deploymentInfo) throws DeploymentException
  {
    super.create(deploymentInfo);
  }

  public void start(DeploymentInfo rootDeployment) throws DeploymentException
  {
    if(rootDeployment.url.getFile().endsWith(ARCHIVE_SUFFIX))
    {
      log.info("Deploy " + rootDeployment.url);

      List<DeploymentRef> deploymentRefs = new ArrayList<DeploymentRef>();

      Iterator iterator = rootDeployment.subDeployments.iterator();
      while(iterator.hasNext())
      {
        DeploymentInfo subDeployment = (DeploymentInfo)iterator.next();
        JBPMDeploymentMetaData md = new JBPMDeploymentMetaData();
        md.setClassloader(subDeployment.localCl);
        md.setWatch(rootDeployment.watch);
        md.setProcessDescriptor(subDeployment.url);

        DeploymentRef deploymentRef;

        try
        {
          deploymentRef = adaptor.deploy(md);
          deploymentRefs.add(deploymentRef);
        }
        catch (Throwable t)
        {          
          throw new DeploymentException(t);
        }
      }

      rootDeployment.context.put(CONTEXT_REFERENCE, deploymentRefs);
    }
  }

  public void destroy(DeploymentInfo deploymentInfo)
  {
    try
    {
      super.destroy(deploymentInfo);
    }
    catch (DeploymentException e)
    {
      throw new RuntimeException(e);
    }
  }

  public void stop(DeploymentInfo rootDeployment)
  {
    if(rootDeployment.url.getFile().endsWith(ARCHIVE_SUFFIX))
    {
      log.info("Undeploy " + rootDeployment.url);
      
      List<DeploymentRef> deploymentRefs = (List<DeploymentRef>)
          rootDeployment.context.get(CONTEXT_REFERENCE);

      if(null==deploymentRefs)
      {
        log.info("Failed to retrieve process reference information. " +
            "Ignore undepoyment call: "+ rootDeployment.url);
        return;
      }

      adaptor.undeploy(deploymentRefs);
    }
  }
}
