/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.jpdl.internal.activity;

import org.w3c.dom.Element;

import org.jbpm.jpdl.internal.xml.JpdlParser;
import org.jbpm.pvm.internal.el.Expression;
import org.jbpm.pvm.internal.model.ActivityImpl;
import org.jbpm.pvm.internal.model.ExpressionCondition;
import org.jbpm.pvm.internal.model.TransitionImpl;
import org.jbpm.pvm.internal.util.XmlUtil;
import org.jbpm.pvm.internal.wire.usercode.UserCodeCondition;
import org.jbpm.pvm.internal.wire.usercode.UserCodeReference;
import org.jbpm.pvm.internal.xml.Parse;

/**
 * @author Alejandro Guizar
 */
public class ForEachBinding extends JpdlBinding {

  private static final String VARIABLE = "var";
  private static final String COLLECTION = "in";

  public ForEachBinding() {
    super("foreach");
  }

  @Override
  public Object parseJpdl(Element element, Parse parse, JpdlParser parser) {
    ForEachActivity activity = new ForEachActivity();

    if (element.hasAttribute(VARIABLE)) {
      activity.setVariable(element.getAttribute(VARIABLE));
    }
    else {
      parse.addProblem(VARIABLE + " attribute missing", element);
    }

    if (element.hasAttribute(COLLECTION)) {
      Expression collection = Expression.create(element.getAttribute(COLLECTION), Expression.LANGUAGE_UEL_VALUE);
      activity.setCollection(collection);
    }
    else {
      parse.addProblem(COLLECTION + " attribute missing", element);
    }

    // process transition elements
    Element transitionElement = XmlUtil.element(element, "transition");

    if (transitionElement == null) {
      parse.addProblem("outgoing transition expected", element);
    }
    else {
      ActivityImpl activityFromStack = parse.contextStackFind(ActivityImpl.class);
      TransitionImpl transition = activityFromStack.getDefaultOutgoingTransition();

      Element conditionElement = XmlUtil.element(transitionElement, "condition");
      if (conditionElement != null) {
        if (conditionElement.hasAttribute("expr")) {
          ExpressionCondition condition = new ExpressionCondition();
          condition.setExpression(conditionElement.getAttribute("expr"));
          condition.setLanguage(XmlUtil.attribute(conditionElement, "lang"));
          transition.setCondition(condition);
        }
        else {
          Element handlerElement = XmlUtil.element(conditionElement, "handler");
          if (handlerElement != null) {
            UserCodeReference conditionReference = parser.parseUserCodeReference(handlerElement, parse);
            UserCodeCondition condition = new UserCodeCondition();
            condition.setConditionReference(conditionReference);
            transition.setCondition(condition);
          }
        }
      }
    }

    return activity;
  }

}
