/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.model.impl;

//$Id: ActivityImpl.java 1989 2008-08-22 20:13:51Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.InvalidProcessException;
import org.jboss.bpm.NotImplementedException;
import org.jboss.bpm.model.Activity;
import org.jboss.bpm.model.ConnectingObject;
import org.jboss.bpm.model.Expression;
import org.jboss.bpm.model.InputSet;
import org.jboss.bpm.model.MutablePropertySupport;
import org.jboss.bpm.model.OutputSet;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.Property;
import org.jboss.bpm.model.SequenceFlow;
import org.jboss.bpm.ri.runtime.MutableToken;
import org.jboss.bpm.runtime.ExecutionContext;
import org.jboss.bpm.runtime.Token;
import org.jboss.bpm.runtime.TokenExecutor;

/**
 * An activity is a generic term for work that a company or organization performs via business processes. An activity
 * can be atomic or non-atomic (compound). The types of activities that are a part of a Process Model are: Process,
 * Sub-Process, and 
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
@SuppressWarnings("serial")
public abstract class ActivityImpl extends FlowObjectImpl implements Activity, MutablePropertySupport,
    SingleInFlowSetterSupport, SingleOutFlowSetterSupport
{
  // provide logging
  private static final Log log = LogFactory.getLog(ActivityImpl.class);
  
  private List<InputSet> inputSets = new ArrayList<InputSet>();
  private List<OutputSet> outputSets = new ArrayList<OutputSet>();
  private List<Expression> ioRules = new ArrayList<Expression>();
  private List<Property> props = new ArrayList<Property>();
  private SequenceFlow inFlow;
  private SequenceFlow outFlow;

  public ActivityImpl(String name)
  {
    super(name);
  }

  public int getStartQuantity()
  {
    throw new NotImplementedException("JBPM-1631", "Activity startQuantity");
  }

  public int getCompletionQuantity()
  {
    throw new NotImplementedException("JBPM-1632", "Activity completionQuantity");
  }

  public List<Expression> getIORules()
  {
    return Collections.unmodifiableList(ioRules);
  }

  public void addIORule(Expression expr)
  {
    ioRules.add(expr);
  }

  public List<InputSet> getInputSets()
  {
    return Collections.unmodifiableList(inputSets);
  }

  public void addInputSet(InputSet inputSet)
  {
    inputSets.add(inputSet);
  }

  public List<OutputSet> getOutputSets()
  {
    return Collections.unmodifiableList(outputSets);
  }

  public void addOutputSet(OutputSet outputSet)
  {
    outputSets.add(outputSet);
  }

  public LoopType getLoopType()
  {
    throw new NotImplementedException("JBPM-1633", "Activity loopType");
  }

  public List<String> getPerformers()
  {
    throw new NotImplementedException("JBPM-1634", "Activity Performers");
  }

  public Property getProperty(String name)
  {
    for (Property prop : props)
    {
      if (prop.getName().equals(name))
        return prop;
    }
    return null;
  }

  public Object getPropertyValue(String name)
  {
    Property prop = getProperty(name);
    return prop != null ? prop.getValue() : null;
  }

  public <T> T getPropertyValue(Class<T> clazz, String name)
  {
    Property prop = getProperty(name);
    return prop != null ? prop.getValue(clazz) : null;
  }
  
  public List<Property> getProperties()
  {
    return Collections.unmodifiableList(props);
  }

  public List<String> getPropertyNames()
  {
    List<String> names = new ArrayList<String>();
    for (Property prop : props)
    {
      names.add(prop.getName());
    }
    return names;
  }

  public void addProperty(Property prop)
  {
    props.add(prop);
  }

  public ConnectingObject getInFlow()
  {
    return inFlow;
  }

  public void setInFlow(SequenceFlow inFlow)
  {
    this.inFlow = inFlow;
  }

  public SequenceFlow getOutFlow()
  {
    return outFlow;
  }

  public void setOutFlow(SequenceFlow flow)
  {
    this.outFlow = flow;
  }

  @Override
  public void execute(Token token)
  {
    MutableToken mutableToken = (MutableToken)token;
    mutableToken.setOutputSet(getActiveOutputSet());
    mutableToken.setInputSet(getActiveInputSet(token));
    super.execute(token);
    processOutputSet(token);
    postProcessInputSet(token);
  }

  /**
   * Select and validate active inputSet
   */
  protected InputSet getActiveInputSet(Token token)
  {
    InputSetImpl inputSet = null;
    ExecutionContext exContext = token.getExecutionContext();

    // Find the InputSet that matches the data in the Token
    if (inputSets.size() > 0)
    {
      for (InputSet auxSet : inputSets)
      {
        boolean allInputPropsInContext = true;
        for (Property prop : auxSet.getProperties())
        {
          Object att = exContext.getAttachment(prop.getName());
          allInputPropsInContext &= (att != null);
        }
        if (allInputPropsInContext == true)
        {
          inputSet = new InputSetImpl();
          for (Property prop : auxSet.getProperties())
          {
            String name = prop.getName();
            Object value = exContext.getAttachment(name);
            inputSet.addProperty(new PropertyImpl(name, new ExpressionImpl(value)));
          }
          break;
        }
      }
      if (inputSet == null)
      {
        log.warn("InputSets: " + inputSets);
        throw new IllegalStateException("Cannot find matching inputSet for " + exContext + " in Activity: " + getName());
      }
    }

    // Create an empty input set
    if (inputSet == null)
      inputSet = new InputSetImpl();

    return inputSet;
  }

  /**
   * Select the active outputSet
   */
  protected OutputSet getActiveOutputSet()
  {
    OutputSet outputSet = null;
    if (outputSets.size() > 0)
    {
      if (outputSets.size() > 1)
        throw new NotImplementedException("JBPM-1635", "IORules and multiple outputSets");

      outputSet = new OutputSetImpl();
      for (Property prop : outputSets.get(0).getProperties())
      {
        outputSet.addProperty(prop);
      }
    }

    // Create an empty output set
    if (outputSet == null)
      outputSet = new OutputSetImpl();

    return outputSet;
  }

  /**
   * Transfer data from outputSet to Token
   */
  protected void processOutputSet(Token token)
  {
    ExecutionContext exContext = token.getExecutionContext();

    // Add the outputSet properties to the Token
    OutputSet outputSet = token.getOutputSet();
    for (Property prop : getActiveOutputSet().getProperties())
    {
      Property outProp = outputSet.getProperty(prop.getName());
      if (outProp == null)
        throw new IllegalStateException("Cannot find outputSet property '" + prop.getName() + "' in Activity: "
            + getName());

      String name = outProp.getName();
      Object value = outProp.getValue();
      exContext.addAttachment(name, value);
    }
  }

  /**
   * Remove the inputSet properties
   */
  protected void postProcessInputSet(Token token)
  {
    // InputSet inputSet = token.getInputSet();
    // ExecutionContext exContext = token.getExecutionContext();
    // for (Property prop : inputSet.getProperties())
    // {
    // // TODO: define proper scope for token data
    // exContext.removeAttachment(prop.getName());
    // }
  }

  @Override
  protected void defaultFlowHandler(TokenExecutor tokenExecutor, Token token)
  {
    tokenExecutor.move(token, getOutFlow());
  }

  @Override
  protected void create(Process proc)
  {
    super.create(proc);

    // Validate InputSets
    for (InputSet inSet : inputSets)
    {
      int artSize = inSet.getArtifactInputs().size();
      int propSize = inSet.getProperties().size();
      if (artSize == 0 && propSize == 0)
      {
        throw new InvalidProcessException(
            "For the combination of ArtifactInputs and PropertyInputs, there MUST be at least one item defined for the InputSet");
      }
    }

    // Validate OutputSets
    for (OutputSet outSet : outputSets)
    {
      int artSize = outSet.getArtifactOutputs().size();
      int propSize = outSet.getProperties().size();
      if (artSize == 0 && propSize == 0)
      {
        throw new InvalidProcessException(
            "For the combination of ArtifactOutputs and PropertyOututs, there MUST be at least one item defined for the OutputSet");
      }
    }
  }
}