/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.model.impl;

//$Id: EndEventImpl.java 1982 2008-08-22 10:09:27Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.management.ObjectName;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.client.SignalManager;
import org.jboss.bpm.model.Constants;
import org.jboss.bpm.model.EndEvent;
import org.jboss.bpm.model.EventDetail;
import org.jboss.bpm.model.MessageEventDetail;
import org.jboss.bpm.model.ObjectNameFactory;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.SequenceFlow;
import org.jboss.bpm.model.Signal;
import org.jboss.bpm.model.EventDetail.EventDetailType;
import org.jboss.bpm.runtime.ExecutionContext;
import org.jboss.bpm.runtime.SignalHandler;
import org.jboss.bpm.runtime.Token;
import org.jboss.bpm.runtime.TokenExecutor;
import org.jboss.util.id.UID;

/**
 * As the name implies, the End Event indicates where a Process will end.
 * 
 * In terms of Sequence Flow, the End Event ends the flow of the Process, and thus, will not have any outgoing Sequence Flow. An End Event can have a specific Result
 * that will appear as a marker within the center of the End Event shape. End Event Results are Message, Error, Compensation, Link, and Multiple.
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
@SuppressWarnings("serial")
public class EndEventImpl extends EventImpl implements EndEvent, SingleInFlowSetterSupport
{
  // provide logging
  private static final Log log = LogFactory.getLog(EndEventImpl.class);

  private List<EventDetail> resultSet = new ArrayList<EventDetail>();
  private SequenceFlow inFlow;

  public EndEventImpl(String name)
  {
    super(name);
  }

  @Override
  public ObjectName getID()
  {
    if (id == null)
    {
      StringBuilder str = new StringBuilder(Constants.ID_DOMAIN + ":");
      str.append("type=EndEvent,name=" + getName() + ",id=" + new UID());
      id = ObjectNameFactory.create(str.toString());
    }
    return id;
  }
  
  public SequenceFlow getInFlow()
  {
    return inFlow;
  }

  /**
   * Set the in flow
   */
  public void setInFlow(SequenceFlow inFlow)
  {
    this.inFlow = inFlow;
  }

  public void addResult(EventDetail result)
  {
    resultSet.add(result);
  }

  public List<EventDetail> getResult()
  {
    return Collections.unmodifiableList(resultSet);
  }

  @Override
  protected void defaultExecution(Token token)
  {
    super.defaultExecution(token);
    
    for (EventDetail eventDetail : getResult())
    {
      if (eventDetail.getEventDetailType() == EventDetailType.Message)
      {
        MessageEventDetail msgEventDetail = (MessageEventDetail)eventDetail;
        MessageSender messageSender = new MessageSender(this, msgEventDetail.getMessageRef());
        messageSender.sendMessage(token);
      }
    }
  }

  @Override
  protected void defaultFlowHandler(TokenExecutor tokenExecutor, Token token)
  {
    log.debug("End reached in: " + getName());
    ExecutionContext exContext = token.getExecutionContext();
    exContext.addAttachment(EndSignalCallback.class, new EndSignalCallback(tokenExecutor));
  }

  public SignalHandler getSignalHandler()
  {
    SignalHandler handler = super.getSignalHandler();
    if (handler == null)
    {
      handler = new SignalHandler()
      {
        SignalManager signalManager = SignalManager.locateSignalManager();
        public void throwEnterSignal(Token token)
        {
          Signal signal = new Signal(getID(), Signal.SignalType.SYSTEM_END_EVENT_ENTER);
          signalManager.throwSignal(signal);
        }

        public void throwExitSignal(Token token)
        {
          Signal signal = new Signal(getID(), Signal.SignalType.SYSTEM_END_EVENT_EXIT);
          signalManager.throwSignal(signal);
          
          // Destroy the token
          ExecutionContext exContext = token.getExecutionContext();
          EndSignalCallback callback = exContext.getAttachment(EndSignalCallback.class);
          callback.destroyToken(token);
        }
      };
    }
    return handler;
  }

  @Override
  protected void create(Process proc)
  {
    super.create(proc);
    
    // Initialize Results
    for (EventDetail result : getResult())
    {
      EventDetailImpl resultImpl = (EventDetailImpl)result;
      resultImpl.initialize(this);
    }
  }

  public String toString()
  {
    return "EndEvent[" + getName() + "]";
  }

  /**
   * The callback that destroys the token AFTER the SYSTEM_END_EVENT_EXIT signal
   */
  static class EndSignalCallback
  {
    TokenExecutor tokenExecutor;
    public EndSignalCallback(TokenExecutor tokenExecutor)
    {
      this.tokenExecutor = tokenExecutor;
    }

    void destroyToken(Token token)
    {
      tokenExecutor.destroy(token);
    }
  }
}