/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.model.impl;

//$Id: GatewayImpl.java 1982 2008-08-22 10:09:27Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.management.ObjectName;

import org.jboss.bpm.client.SignalManager;
import org.jboss.bpm.model.ConnectingObject;
import org.jboss.bpm.model.Constants;
import org.jboss.bpm.model.Gate;
import org.jboss.bpm.model.Gateway;
import org.jboss.bpm.model.ObjectNameFactory;
import org.jboss.bpm.model.SequenceFlow;
import org.jboss.bpm.model.Signal;
import org.jboss.bpm.model.SequenceFlow.ConditionType;
import org.jboss.bpm.runtime.ExecutionHandler;
import org.jboss.bpm.runtime.SignalHandler;
import org.jboss.bpm.runtime.Token;
import org.jboss.util.id.UID;

/**
 * Gateways are modelling elements that are used to control how Sequence Flow interact as they converge and diverge
 * within a Process. If the flow does not need to be controlled, then a Gateway is not needed.
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
@SuppressWarnings("serial")
public abstract class GatewayImpl extends FlowObjectImpl implements Gateway, MultipleInFlowSetterSupport
{
  // The list of incomming flows
  protected List<ConnectingObject> inFlows = new ArrayList<ConnectingObject>();
  // The list of outgoing gates
  private Map<String, Gate> gates = new LinkedHashMap<String, Gate>();
  // The list of flows from which a token is expected
  protected List<ConnectingObject> expectedFlows;
  // The list of received tokens
  protected List<Token> receivedTokens;

  public GatewayImpl(String name)
  {
    super(name);
  }

  @Override
  public ObjectName getID()
  {
    if (id == null)
    {
      StringBuilder str = new StringBuilder(Constants.ID_DOMAIN + ":");
      str.append("type=" + getGatewayType() + "Gateway,name=" + getName() + ",id=" + new UID());
      id = ObjectNameFactory.create(str.toString());
    }
    return id;
  }
  
  public List<Gate> getGates()
  {
    return Collections.unmodifiableList(new ArrayList<Gate>(gates.values()));
  }

  public void addGate(Gate gate)
  {
    String targetName = gate.getOutgoingSequenceFlow().getTargetName();
    gates.put(targetName, gate);
  }

  public List<ConnectingObject> getInFlows()
  {
    return Collections.unmodifiableList(inFlows);
  }

  public void addInFlow(ConnectingObject inFlow)
  {
    this.inFlows.add(inFlow);
  }

  public Gate getDefaultGate()
  {
    Gate gate = null;
    for (Gate aux : gates.values())
    {
      SequenceFlow seqFlow = aux.getOutgoingSequenceFlow();
      if (seqFlow.getConditionType() == ConditionType.Default)
      {
        gate = aux;
        break;
      }
    }
    return gate;
  }

  public Gate getGateByTargetName(String targetName)
  {
    Gate gate = gates.get(targetName);
    return gate;
  }

  @Override
  public void defaultExecution(Token token)
  {
    // Initialize the gateway
    if (expectedFlows == null)
    {
      expectedFlows = new ArrayList<ConnectingObject>(inFlows);
      receivedTokens = new ArrayList<Token>();
    }
    
    // Check that token from flow is valid
    ConnectingObject flow = token.getFlow();
    if (expectedFlows.contains(flow) == false)
      throw new IllegalStateException("Unexpected token from: " + flow);
    
    // Call execution handler
    ExecutionHandler exHandler = getExecutionHandler();
    if (exHandler != null)
      exHandler.execute(token);

    // Remove the flow from the expected list  
    expectedFlows.remove(flow);
    
    // Store the received token for processing in the FlowHandler
    receivedTokens.add(token);
  }

  @Override
  public void reset()
  {
    super.reset();
    expectedFlows = null;
    receivedTokens = null;
  }
  
  public SignalHandler getSignalHandler()
  {
    SignalHandler handler = super.getSignalHandler();
    if (handler == null)
    {
      handler = new SignalHandler()
      {
        SignalManager signalManager = SignalManager.locateSignalManager();
        public void throwEnterSignal(Token token)
        {
          Signal signal = new Signal(getID(), Signal.SignalType.SYSTEM_GATEWAY_ENTER);
          signalManager.throwSignal(signal);
        }

        public void throwExitSignal(Token token)
        {
          Signal signal = new Signal(getID(), Signal.SignalType.SYSTEM_GATEWAY_EXIT);
          signalManager.throwSignal(signal);
        }
      };
    }
    return handler;
  }
}