/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.model.impl;

//$Id: InclusiveGatewayImpl.java 1982 2008-08-22 10:09:27Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.model.Expression;
import org.jboss.bpm.model.FlowObject;
import org.jboss.bpm.model.Gate;
import org.jboss.bpm.model.InclusiveGateway;
import org.jboss.bpm.model.SequenceFlow;
import org.jboss.bpm.model.Expression.ExpressionLanguage;
import org.jboss.bpm.model.SequenceFlow.ConditionType;
import org.jboss.bpm.runtime.ExecutionContext;
import org.jboss.bpm.runtime.Token;
import org.jboss.bpm.runtime.TokenExecutor;
import org.jboss.bpm.runtime.Attachments.Key;
import org.mvel.MVEL;

/**
 * This Decision represents a branching point where Alternatives are based on conditional expressions contained within
 * outgoing Sequence Flow. However, in this case, the True evaluation of one condition expression does not exclude the
 * evaluation of other condition expressions. All Sequence Flow with a True evaluation will be traversed by a Token.
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
@SuppressWarnings("serial")
public class InclusiveGatewayImpl extends GatewayImpl implements InclusiveGateway
{
  // provide logging
  private static final Log log = LogFactory.getLog(InclusiveGatewayImpl.class);

  public InclusiveGatewayImpl(String name)
  {
    super(name);
  }

  public GatewayType getGatewayType()
  {
    return GatewayType.Inclusive;
  }

  @Override
  protected void defaultFlowHandler(TokenExecutor tokenExecutor, Token token)
  {
    FlowObject sourceRef = token.getFlow().getSourceRef();
    log.debug("Propagate token comming from: " + sourceRef);

    // Get the applicable gates
    List<Gate> applicableGates = getApplicableGates(token);
    log.debug("applicableGates: " + applicableGates);

    if (applicableGates.size() == 1)
    {
      SequenceFlow outFlow = applicableGates.get(0).getOutgoingSequenceFlow();
      tokenExecutor.move(token, outFlow);
    }
    else
    {
      List<Token> outTokens = new ArrayList<Token>();
      for (Gate aux : applicableGates)
      {
        SequenceFlow outFlow = aux.getOutgoingSequenceFlow();
        Token outToken = token.copyToken();
        tokenExecutor.create(outToken, outFlow);
        outTokens.add(outToken);
      }
      for (Token outToken : outTokens)
      {
        tokenExecutor.start(outToken);
      }
    }

    // Destroy the incomming token if there are
    // more than one applicable gates
    if (applicableGates.size() > 1)
      tokenExecutor.destroy(token);
  }

  // Get applicable gates which' condition evaluates to TRUE
  // Fall back to the default gate if there is one
  // Choke if there is no applicable gate
  private List<Gate> getApplicableGates(Token token)
  {
    List<Gate> applicableGates = new ArrayList<Gate>();
    for (Gate auxGate : getGates())
    {
      SequenceFlow seqFlow = auxGate.getOutgoingSequenceFlow();
      if (seqFlow.getConditionType() == ConditionType.Expression)
      {
        Expression expr = seqFlow.getConditionExpression();
        if (expr.getExpressionLanguage() == ExpressionLanguage.MVEL)
        {
          String mvel = expr.getExpressionBody();
          ExecutionContext exContext = token.getExecutionContext();
          Map<String, Object> vars = new HashMap<String, Object>();
          for (Key key : exContext.getAttachmentKeys())
          {
            String name = key.getNamePart();
            Object value = exContext.getAttachment(name);
            vars.put(name, value);
          }
          Boolean result = (Boolean)MVEL.eval(mvel, vars);
          if (result == true)
          {
            applicableGates.add(auxGate);
          }
        }
        else
        {
          throw new IllegalStateException("Unsupported expression language: " + expr.getExpressionLanguage());
        }
      }
    }

    // Use to the default gate if there is one
    if (applicableGates.size() == 0)
    {
      for (Gate auxGate : getGates())
      {
        SequenceFlow seqFlow = auxGate.getOutgoingSequenceFlow();
        if (seqFlow.getConditionType() == ConditionType.Default)
        {
          applicableGates.add(auxGate);
        }
      }
    }

    // Fallback to the single outgoing gate that is not conditional
    if (applicableGates.size() == 0 && getGates().size() == 1)
    {
      Gate auxGate = getGates().get(0);
      SequenceFlow seqFlow = auxGate.getOutgoingSequenceFlow();
      if (seqFlow.getConditionType() == ConditionType.None)
      {
        applicableGates.add(auxGate);
      }
    }

    if (applicableGates.size() == 0)
      throw new IllegalStateException("Cannot select applicable gate in: " + this);

    return applicableGates;
  }

  public String toString()
  {
    return "InclusiveGateway[" + getName() + "]";
  }
}