/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.model.impl;

//$Id: ParallelGatewayImpl.java 1949 2008-08-21 08:28:04Z thomas.diesler@jboss.com $

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.model.ConnectingObject;
import org.jboss.bpm.model.Gate;
import org.jboss.bpm.model.ParallelGateway;
import org.jboss.bpm.model.SequenceFlow;
import org.jboss.bpm.ri.runtime.TokenImpl;
import org.jboss.bpm.runtime.Token;
import org.jboss.bpm.runtime.TokenExecutor;

/**
 * Parallel Gateway is required when two or more Activities need to be executed in parallel.
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
@SuppressWarnings("serial")
public class ParallelGatewayImpl extends GatewayImpl implements ParallelGateway
{
  // provide logging
  private static final Log log = LogFactory.getLog(ParallelGatewayImpl.class);

  // Sync management
  private Set<ConnectingObject> outstandingFlows;
  private Set<Token> mergeTokens;

  public ParallelGatewayImpl(String name)
  {
    super(name);
  }

  public GatewayType getGatewayType()
  {
    return GatewayType.Parallel;
  }

  @Override
  public void defaultExecution(Token token)
  {
    super.defaultExecution(token);
    
    // Multiple incomingFlows must be synchronized
    if (outstandingFlows == null)
    {
      outstandingFlows = new HashSet<ConnectingObject>(inFlows);
      mergeTokens = new HashSet<Token>();
    }

    ConnectingObject flow = token.getFlow();
    outstandingFlows.remove(flow);
    mergeTokens.add(token);
  }

  @Override
  protected void defaultFlowHandler(TokenExecutor tokenExecutor, Token token)
  {
    // In any case, the incomming token is not propagated
    tokenExecutor.suspend(token);
    
    // If the gateway has a single incomming flow the outgoing token is the incomming token 
    Token outToken = (getInFlows().size() == 1 ? token : null);
    
    // The outgoing token is the merge of all incomming tokens
    if (outToken == null)
    {
      // If there are no more outstanding flows
      if (outstandingFlows.size() == 0)
      {
        // Merge the tokens together and submit the merged tokens
        outToken = getMergedTokens();
      }

      // There are outstanding flows
      else if (outstandingFlows.size() > 0)
      {
        log.debug("Waiting for " + outstandingFlows + " in gateway: " + this);
      }
    }
    
    // Schedule the outgoing token
    if (outToken != null)
    {
      // Start a copy of the outgoing token for every gate
      List<Token> outTokens = new ArrayList<Token>();
      for(Gate gate : getGates())
      {
        SequenceFlow outFlow = gate.getOutgoingSequenceFlow();
        outToken = outToken.copyToken();
        tokenExecutor.create(outToken, outFlow);
        outTokens.add(outToken);
      }
      for (Token auxToken : outTokens)
      {
        tokenExecutor.start(auxToken);
      }
      
      // Destroy the received tokens
      for (Token auxToken : receivedTokens)
      {
        tokenExecutor.destroy(auxToken);
      }
      
      // Reset the gateway
      reset();
    }
  }
  
  @Override
  public void reset()
  {
    super.reset();
    outstandingFlows = null;
    mergeTokens = null;
  }
  
  private Token getMergedTokens()
  {
    TokenImpl mergedToken = new TokenImpl(null);
    for (Token auxToken : mergeTokens)
    {
      log.debug("mergeToken: " + auxToken);
      mergedToken.mergeToken(auxToken);
    }
    return mergedToken;
  }

  public String toString()
  {
    return "ParallelGateway[" + getName() + "]";
  }
}