/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.model.impl;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.InvalidProcessException;
import org.jboss.bpm.NotImplementedException;
import org.jboss.bpm.client.MessageListener;
import org.jboss.bpm.model.Message;
import org.jboss.bpm.model.Process;
import org.jboss.bpm.model.Property;
import org.jboss.bpm.model.ReceiveTask;
import org.jboss.bpm.model.Message.Implementation;
import org.jboss.bpm.runtime.ExecutionContext;
import org.jboss.bpm.runtime.Token;
import org.jboss.bpm.runtime.TokenExecutor;

//$Id: ReceiveTaskImpl.java 1982 2008-08-22 10:09:27Z thomas.diesler@jboss.com $

/**
 * Task that corresponds to the TaskType.None
 * 
 * @author thomas.diesler@jboss.com
 * @since 08-Jul-2008
 */
@SuppressWarnings("serial")
public class ReceiveTaskImpl extends TaskImpl implements ReceiveTask, MessageListener
{
  // provide logging
  private static final Log log = LogFactory.getLog(ReceiveTaskImpl.class);

  // A Web service is the default technology
  private Implementation implementation = Implementation.WebService;
  private Message messageRef;

  private List<Message> receivedMessages = new ArrayList<Message>();
  private List<Token> suspendedTokens = new ArrayList<Token>();

  public ReceiveTaskImpl(String name)
  {
    super(name);
  }

  @Override
  public TaskType getTaskType()
  {
    return TaskType.Receive;
  }

  public Message.Implementation getImplementation()
  {
    return implementation;
  }

  public Message getMessageRef()
  {
    return messageRef;
  }

  public void setMessageRef(Message message)
  {
    this.messageRef = message;
  }

  public boolean isInstantiate()
  {
    throw new NotImplementedException("JBPM-1648", "ReceiveTask Instantiate");
  }

  public synchronized void catchMessage(Message message)
  {
    String msgName = message.getName();
    if (messageRef.getName().equals(msgName))
    {
      log.debug("catchMessage in " + this + " => " + message);
      for (Property prop : messageRef.getProperties())
      {
        String name = prop.getName();
        if (message.getProperty(name) == null)
          throw new IllegalArgumentException("Received message does not contain expected property: " + name);
      }
      receivedMessages.add(message);
      
      if (suspendedTokens.size() == 0)
      {
        log.debug("Suspend message: " + message);
      }
      else
      {
        Token token = suspendedTokens.remove(0);
        ExecutionContext exContext = token.getExecutionContext();
        TokenExecutor tokenExecutor = exContext.removeAttachment(TokenExecutor.class);
        tokenExecutor.activate(token.getTokenID());
      }
    }
    else
    {
      log.debug("Ignore unexpected message: " + message);
    }
  }

  @Override
  public synchronized void defaultExecution(Token token)
  {
    if (receivedMessages.size() > 0)
    {
      // Copy the expected properties from the
      // received message to the execution context
      Message msg = receivedMessages.get(0);
      ExecutionContext exContext = token.getExecutionContext();
      for (Property prop : messageRef.getProperties())
      {
        String key = prop.getName();
        Object value = msg.getPropertyValue(key);
        exContext.addAttachment(key, value);
      }

      // Call default execute
      super.defaultExecution(token);
    }
  }

  @Override
  protected void endTimeAssignments(Token token)
  {
    if (receivedMessages.size() > 0)
    {
      super.endTimeAssignments(token);
    }
  }

  @Override
  protected synchronized void defaultFlowHandler(TokenExecutor tokenExecutor, Token token)
  {
    if (receivedMessages.size() > 0)
    {
      receivedMessages.remove(0);
      tokenExecutor.move(token, getOutFlow());
    }
    else
    {
      tokenExecutor.suspend(token);
      ExecutionContext exContext = token.getExecutionContext();
      exContext.addAttachment(TokenExecutor.class, tokenExecutor);
      suspendedTokens.add(token);
    }
  }

  @Override
  protected void create(Process proc)
  {
    super.create(proc);

    if (messageRef == null)
      throw new InvalidProcessException("A Message for the MessageRef attribute MUST be entered");

    ProcessImpl procImpl = (ProcessImpl)proc;
    procImpl.initializeMessageRef(messageRef);
  }
}