/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.ri.runtime;

//$Id: TokenImpl.java 1958 2008-08-21 13:28:11Z thomas.diesler@jboss.com $

import org.jboss.bpm.model.InputSet;
import org.jboss.bpm.model.OutputSet;
import org.jboss.bpm.model.SequenceFlow;
import org.jboss.bpm.runtime.Attachments;
import org.jboss.bpm.runtime.BasicExecutionContext;
import org.jboss.bpm.runtime.ExecutionContext;
import org.jboss.bpm.runtime.Token;
import org.jboss.bpm.runtime.Attachments.Key;
import org.jboss.util.id.UID;

/**
 * A Token is a descriptive construct used to describe how the flow of a Process will proceed at runtime.
 * 
 * By tracking how the Token traverses the Flow Objects, gets diverted through alternative paths, 
 * and gets split into parallel paths, the normal Sequence Flow should be completely definable.
 * 
 * A Token will have a unique identity that can be used to separate multiple Tokens that may exist because of 
 * concurrent process instances or the splitting of the Token for parallel processing within a single process instance.
 * 
 * @author Thomas.Diesler@jboss.com
 * @since 20-Apr-2007
 */
public class TokenImpl implements MutableToken
{
  private String id;
  private SequenceFlow flow;
  private ExecutionContext context;
  private InputSet inputSet;
  private OutputSet outputSet;
  private TokenStatus status;

  /**
   * Construct a Token with given {@link Attachments}
   */
  public TokenImpl(Attachments att)
  {
    this.context = new BasicExecutionContext(att);
    this.id = new UID().toString();
    this.status = TokenStatus.Created;
  }

  public String getTokenID()
  {
    return id;
  }

  public TokenStatus getTokenStatus()
  {
    return status;
  }

  public void setTokenStatus(TokenStatus status)
  {
    this.status = status;
  }
  
  public ExecutionContext getExecutionContext()
  {
    return context;
  }
  
  public SequenceFlow getFlow()
  {
    return flow;
  }

  public void setFlow(SequenceFlow flow)
  {
    this.flow = flow;
  }

  public InputSet getInputSet()
  {
    return inputSet;
  }

  public void setInputSet(InputSet inputSet)
  {
    this.inputSet = inputSet;
  }

  public OutputSet getOutputSet()
  {
    return outputSet;
  }

  public void setOutputSet(OutputSet outputSet)
  {
    this.outputSet = outputSet;
  }

  /**
   * Create a schallow copy of this Token.
   * <p/>
   * The content in the {@link ExecutionContext} will be copied by reference. 
   */
  public Token copyToken()
  {
    return new TokenImpl(context);
  }
  
  /**
   * Merge this Token with another token.
   */
  public void mergeToken(Token token)
  {
    ExecutionContext mergeContext = token.getExecutionContext();
    for(Key key : mergeContext.getAttachmentKeys())
    {
      Object mergeValue = mergeContext.getAttachment(key.getClassPart(), key.getNamePart());
      Object existValue = context.getAttachment(key.getClassPart(), key.getNamePart());
      if (existValue != null && existValue.equals(mergeValue) == false)
        throw new IllegalStateException("Cannot merge the same key with different values: " + key);
      
      context.addAttachment(key.getClassPart(), key.getNamePart(), mergeValue);
    }
  }
  
  public String toString()
  {
    return "[sf=" + getFlow() + ",status=" + getTokenStatus() + ",ctx=" + getExecutionContext() + "]";
  }
}