/*
 * Copyright 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose.material

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalDensity
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp

/**
 * <a href="https://material.io/components/dividers" class="external" target="_blank">Material
 * Design divider</a>.
 *
 * A divider is a thin line that groups content in lists and layouts.
 *
 * ![Dividers
 * image](https://developer.android.com/images/reference/androidx/compose/material/dividers.png)
 *
 * @param modifier Modifier to be applied to the divider line
 * @param color color of the divider line
 * @param thickness thickness of the divider line, 1 dp is used by default. Using [Dp.Hairline] will
 *   produce a single pixel divider regardless of screen density.
 * @param startIndent start offset of this line, no offset by default
 */
@Composable
fun Divider(
    modifier: Modifier = Modifier,
    color: Color = MaterialTheme.colors.onSurface.copy(alpha = DividerAlpha),
    thickness: Dp = 1.dp,
    startIndent: Dp = 0.dp
) {
    val indentMod =
        if (startIndent.value != 0f) {
            Modifier.padding(start = startIndent)
        } else {
            Modifier
        }
    val targetThickness =
        if (thickness == Dp.Hairline) {
            (1f / LocalDensity.current.density).dp
        } else {
            thickness
        }
    Box(modifier.then(indentMod).fillMaxWidth().height(targetThickness).background(color = color))
}

private const val DividerAlpha = 0.12f
