package org.jetbrains.exposed.v1.jdbc.statements.jdbc

import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import org.jetbrains.exposed.v1.core.statements.api.ResultApi
import org.jetbrains.exposed.v1.core.statements.api.RowApi
import java.sql.ResultSet

/**
 * Class responsible for wrapping a [ResultSet] generated by executing a statement that queries a JDBC database.
 *
 * @property result The actual [ResultSet] returned by the database after statement execution.
 */
class JdbcResult(
    val result: ResultSet
) : ResultApi, RowApi {
    private var consumed = false

    override fun <T> mapRows(block: (RowApi) -> T?): Flow<T?> {
        if (consumed) error("Result is already consumed")
        consumed = true

        return flow {
            while (next()) {
                emit(block(this@JdbcResult))
            }
        }
    }

    override fun toString(): String = "JdbcResult(resultSet = $result)"

    override fun getObject(index: Int): Any? = result.getObject(index)

    override fun <T> getObject(index: Int, type: Class<T>): T? = result.getObject(index, type)

    override fun getObject(name: String): Any? = result.getObject(name)

    override fun <T> getObject(name: String, type: Class<T>): T? = result.getObject(name, type)

    override fun getString(index: Int): String? = result.getString(index)

    /**
     * Moves from the current position in the [ResultSet] to the next row and returns `true`,
     * or returns `false` if there are no more rows to move forward to.
     */
    fun next(): Boolean = result.next()

    override fun close() {
        releaseResult()
    }

    /** Releases the underlying statement object, and any database or driver resources, as well as its [ResultSet]. */
    fun releaseResult() {
        val statement = result.statement
        statement?.close()
    }
}

/** Returns the actual underlying [ResultSet] at the current position in this result [RowApi]. */
val RowApi.origin: ResultSet get() = (this as JdbcResult).result
