/*
 * Copyright 2000-2024 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.rt.coverage.instrumentation.filters.lines;

import com.intellij.rt.coverage.instrumentation.data.InstrumentationData;
import org.jetbrains.coverage.org.objectweb.asm.Label;
import org.jetbrains.coverage.org.objectweb.asm.Opcodes;

import java.util.HashSet;
import java.util.Set;

/**
 * Ignore lines generated by compiler for try-finally block.
 * 3 different types of code are ignored by this filter.
 *
 * @see TryFinallyLineFilter.State
 */
public class TryFinallyLineFilter extends BaseLineFilter {
  private final Set<Label> myCatchLabels = new HashSet<Label>();
  private State myState = State.INITIAL;
  private boolean myCatchLabelVisited = false;

  private enum State {
    // No code on a line
    INITIAL,
    // Scenario 1: exception is stored into variable after a catch statement
    F1_ASTORE,
    // Scenario 2: no-throw execution is redirected before a catch block
    F2_GOTO,
    // Scenario 3: exception is rethrown inside finally block
    F3_ALOAD, F3_THROW,
  }

  @Override
  public boolean isApplicable(InstrumentationData context) {
    return true;
  }

  @Override
  protected boolean shouldRemoveLine() {
    return myState == State.F1_ASTORE
        || (myState == State.F2_GOTO && myCatchLabelVisited)
        || myState == State.F3_THROW;
  }

  @Override
  public void visitTryCatchBlock(Label start, Label end, Label handler, String type) {
    super.visitTryCatchBlock(start, end, handler, type);
    myCatchLabels.add(handler);
  }

  @Override
  public void visitLabel(Label label) {
    super.visitLabel(label);
    myCatchLabelVisited = myCatchLabels.contains(label);
  }

  @Override
  public void visitVarInsn(int opcode, int varIndex) {
    mv.visitVarInsn(opcode, varIndex);
    if (opcode == Opcodes.ASTORE && myCatchLabelVisited) {
      myState = State.F1_ASTORE;
    } else if (opcode == Opcodes.ALOAD && myState == State.INITIAL) {
      myState = State.F3_ALOAD;
    } else {
      setHasInstructions();
    }
  }

  @Override
  public void visitJumpInsn(int opcode, Label label) {
    mv.visitJumpInsn(opcode, label);
    if (opcode != Opcodes.GOTO) {
      setHasInstructions();
      return;
    }
    if (myState == State.INITIAL) {
      myState = State.F2_GOTO;
    } else {
      setHasInstructions();
    }
  }

  @Override
  public void visitInsn(int opcode) {
    mv.visitInsn(opcode);
    if (opcode == Opcodes.ATHROW && myState == State.F3_ALOAD) {
      myState = State.F3_THROW;
    } else {
      setHasInstructions();
    }
  }

  @Override
  public void visitLineNumber(int line, Label start) {
    super.visitLineNumber(line, start);
    myState = State.INITIAL;
  }
}
