/*
 * Copyright 2000-2024 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.rt.coverage.instrumentation.filters.lines;

import com.intellij.rt.coverage.instrumentation.data.InstrumentationData;
import org.jetbrains.coverage.gnu.trove.TIntIntHashMap;
import org.jetbrains.coverage.org.objectweb.asm.Handle;
import org.jetbrains.coverage.org.objectweb.asm.Label;
import org.jetbrains.coverage.org.objectweb.asm.Opcodes;

import java.util.HashSet;
import java.util.Set;

/**
 * This is an additional filter for lines generated by java compiler for older versions of java.
 */
public class TryWithResourcesJava8LineFilter extends BaseLineFilter {
  private final TIntIntHashMap myCandidates = new TIntIntHashMap();
  private final Set<Label> myCatchStarts = new HashSet<Label>();
  private int myState = 1;


  @Override
  public boolean isApplicable(InstrumentationData context) {
    return true;
  }

  @Override
  protected boolean shouldRemoveLine() {
    return myState == 9;
  }

  @Override
  protected boolean wasLineSeenBefore() {
    int previousState = myCandidates.get(getCurrentLine());
    return previousState == -1;
  }

  @Override
  public void visitLineNumber(int line, Label start) {
    myCandidates.put(getCurrentLine(), hasInstructions() ? -1 : myState);
    super.visitLineNumber(line, start);
    int previousState = myCandidates.get(line);
    myState = previousState == 0 ? 1 : previousState;
  }

  @Override
  public void visitTryCatchBlock(Label start, Label end, Label handler, String type) {
    super.visitTryCatchBlock(start, end, handler, type);
    myCatchStarts.add(start);
  }

  @Override
  public void visitLabel(Label label) {
    super.visitLabel(label);
    if (myCatchStarts.contains(label) && myState == 3) {
      myState = 4;
    }
  }

  @Override
  public void visitVarInsn(int opcode, int var) {
    mv.visitVarInsn(opcode, var);
    if ((myState == 2 || myState == 4 || myState == 6) && opcode == Opcodes.ASTORE) {
      myState++;
    } else if (myState == 9 && opcode == Opcodes.ASTORE) {
      myState = 5;
    } else if ((myState == 5 || myState == 7) && opcode == Opcodes.ALOAD) {
      myState++;
    } else {
      setHasInstructions();
    }
  }

  @Override
  public void visitInsn(int opcode) {
    mv.visitInsn(opcode);
    if ((myState == 1 || myState == 4) && opcode == Opcodes.ACONST_NULL) {
      myState = 2;
    } else if (myState == 8 && opcode == Opcodes.ATHROW) {
      myState = 9;
    } else {
      setHasInstructions();
    }
  }
}
