/*
 * Copyright 2000-2022 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.rt.coverage.util;

import com.intellij.rt.coverage.data.ClassData;
import com.intellij.rt.coverage.data.CoverageData;
import com.intellij.rt.coverage.data.FileMapData;
import com.intellij.rt.coverage.data.LineMapData;

/**
 * This class performs line mapping for JSR45 support
 */
public abstract class LineMapper<T extends CoverageData> {
  protected abstract T createNewLine(T targetLine, int line);

  protected abstract T[] createArray(int size);

  protected abstract T[] getLines(ClassData classData);

  /**
   * Apply line mappings: move hits from original line in bytecode to the mapped line.
   *
   * @param linesMap        line mappings from target class to source class
   * @param sourceClassData the class to which the mapped lines are moved
   * @param targetClassData the class which initially contains the mapped lines,
   *                        at the end of this method all mapped lines in this class are set to null
   * @return mapped lines array
   */
  public T[] mapLines(LineMapData[] linesMap, ClassData sourceClassData, ClassData targetClassData) {
    final T[] oldLines = getLines(sourceClassData);
    if (linesMap == null) return oldLines;
    if (sourceClassData == targetClassData && oldLines == null) return null;
    try {
      final T[] sourceLines = getSourceLinesArray(linesMap, sourceClassData, targetClassData);
      final T[] targetLines = getLines(targetClassData);
      for (final LineMapData mapData : linesMap) {
        for (int index = 0; index < mapData.getCount(); index++) {
          final int sourceLineNumber = mapData.getSourceLine(index);
          if (ArrayUtil.safeLoad(sourceLines, sourceLineNumber) == null) {
            final T targetLineData = ArrayUtil.safeLoad(targetLines, mapData.getMappingStart(index));
            if (targetLineData != null) {
              final T source = createNewLine(targetLineData, sourceLineNumber);
              ArrayUtil.safeStore(sourceLines, sourceLineNumber, source);
            }
          }
          for (int targetLineNumber = mapData.getMappingStart(index); targetLineNumber < mapData.getMappingEnd(index); targetLineNumber++) {
            // if this is a mapping inside one class, there is no need to map the line to itself
            if (sourceClassData == targetClassData && sourceLineNumber == targetLineNumber) continue;
            final T source = ArrayUtil.safeLoad(sourceLines, sourceLineNumber);
            final T target = ArrayUtil.safeLoad(targetLines, targetLineNumber);
            if (target == null) continue;
            if (source != null) {
              source.merge(target);
            }
            targetLines[targetLineNumber] = null;
          }
        }
      }
      return sourceLines;
    } catch (Throwable e) {
      ErrorReporter.warn("Error creating line mappings for " + targetClassData.getName(), e);
      return oldLines;
    }
  }

  /**
   * Return lines array of <code>sourceClassData</code> if it fits mapping, or enlarge it.
   */
  private T[] getSourceLinesArray(LineMapData[] linesMap, ClassData sourceClassData, ClassData targetClassData) {
    final int maxMappedSourceLineNumber = maxSourceLineNumber(linesMap);
    final T[] currentLines = getLines(sourceClassData);
    if (targetClassData == sourceClassData) {
      return currentLines;
    } else if (currentLines == null) {
      return createArray(1 + maxMappedSourceLineNumber);
    } else if (currentLines.length >= 1 + maxMappedSourceLineNumber) {
      return currentLines;
    } else {
      final T[] sourceLines = createArray(1 + maxMappedSourceLineNumber);
      System.arraycopy(currentLines, 0, sourceLines, 0, currentLines.length);
      return sourceLines;
    }
  }

  /**
   * Remove all lines that are generated by inline.
   * Do not touch lines that are mapped to itself.
   */
  public static <T> void dropMappedLines(FileMapData[] mappings, T[] lines, String className) {
    for (FileMapData mapData : mappings) {
      final boolean isThisClass = className.equals(mapData.getClassName());
      for (LineMapData lineMapData : mapData.getLines()) {
        for (int index = 0; index < lineMapData.getCount(); index++) {
          final int sourceLineNumber = lineMapData.getSourceLine(index);
          for (int i = lineMapData.getMappingStart(index); i < lineMapData.getMappingEnd(index) && i < lines.length; i++) {
            if (isThisClass && i == sourceLineNumber) continue;
            lines[i] = null;
          }
        }
      }
    }
  }

  private static int maxSourceLineNumber(LineMapData[] linesMap) {
    int max = 0;
    for (final LineMapData mapData : linesMap) {
      for (int index = 0; index < mapData.getCount(); index++) {
        max = Math.max(max, mapData.getSourceLine(index));
      }
    }
    return max;
  }

}
