/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package org.jetbrains.kotlin.analysis.api.components

import com.intellij.openapi.vfs.VirtualFile
import com.intellij.psi.PsiElement
import org.jetbrains.kotlin.analysis.api.KaExperimentalApi
import org.jetbrains.kotlin.analysis.api.resolve.extensions.KaResolveExtension
import org.jetbrains.kotlin.analysis.api.scopes.KaScope
import org.jetbrains.kotlin.psi.KtElement

@KaExperimentalApi
public interface KaResolveExtensionInfoProvider : KaSessionComponent {
    /**
     * Returns the [KaScope] containing top-level callable declarations generated by [KaResolveExtension]s.
     *
     * @see KaResolveExtension
     */
    @KaExperimentalApi
    public val resolveExtensionScopeWithTopLevelDeclarations: KaScope

    /**
     * Whether the given [VirtualFile] is provided by a [KaResolveExtension].
     *
     * @see KaResolveExtension
     */
    @KaExperimentalApi
    public val VirtualFile.isResolveExtensionFile: Boolean

    /**
     * Whether the given [KtElement] is provided by a [KaResolveExtension].
     *
     * @see KaResolveExtension
     */
    @KaExperimentalApi
    public val KtElement.isFromResolveExtension: Boolean

    /**
     * A list of [PsiElement]s which are used as navigation targets in place of this [KtElement] provided by a [KaResolveExtension].
     *
     * These [PsiElement]s will typically be the source item(s) that caused the given [KtElement] to be generated by the
     * [KaResolveExtension]. For example, for a [KtElement] generated by a resource compiler, the navigation elements will typically be the
     * [PsiElement]s of the resource items in the corresponding resource file.
     *
     * @see KaResolveExtension
     * @see org.jetbrains.kotlin.analysis.api.resolve.extensions.KaResolveExtensionNavigationTargetsProvider
     */
    @KaExperimentalApi
    public val KtElement.resolveExtensionNavigationElements: Collection<PsiElement>
}
