/*
 * Copyright 2010-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jetbrains.jet.j2k.ast

import java.util.ArrayList

class MemberComments(elements: List<Element>) : WhiteSpaceSeparatedElementList(elements, WhiteSpace.NoSpace) {
    class object {
        val Empty = MemberComments(ArrayList())
    }
}

abstract class Member(val comments: MemberComments, val modifiers: Set<Modifier>) : Element {
    fun isStatic(): Boolean = modifiers.contains(Modifier.STATIC)
    fun commentsToKotlin(): String = comments.toKotlin()
}

//member itself and all the elements before it in the code (comments, whitespaces)
class MemberHolder(val member: Member, val elements: List<Element>)

class MemberList(elements: List<Element>) : WhiteSpaceSeparatedElementList(elements, WhiteSpace.NewLine) {
    val members: List<Member>
        get() = elements.filter { it is Member }.map { it as Member }
}

class ClassMembers private(
        val primaryConstructor: PrimaryConstructor?,
        val secondaryConstructors: MemberList,
        val allMembers: MemberList,
        val staticMembers: MemberList,
        val nonStaticMembers: MemberList) {
    class object {
        public fun fromBodyElements(elements: List<Element>): ClassMembers {
            val groups = splitInGroups(elements)
            val constructors = groups.filter { it.member is Constructor }
            val primaryConstructor = constructors.map { it.member }.filterIsInstance(javaClass<PrimaryConstructor>()).firstOrNull()
            val secondaryConstructors = constructors.filter { it.member is SecondaryConstructor }
            val nonConstructors = groups.filter { it.member !is Constructor }
            val staticMembers = nonConstructors.filter { it.member.isStatic() }
            val nonStaticMembers = nonConstructors.filter { !it.member.isStatic() }
            return ClassMembers(primaryConstructor,
                                secondaryConstructors.toMemberList(),
                                nonConstructors.toMemberList(),
                                staticMembers.toMemberList(),
                                nonStaticMembers.toMemberList())
        }
    }
}

private fun List<MemberHolder>.toMemberList() = MemberList(flatMap { it.elements })

private fun splitInGroups(elements: List<Element>): List<MemberHolder> {
    val result = ArrayList<Pair<Member, MutableList<Element>>>()
    var currentGroup = ArrayList<Element>()
    for (element in elements) {
        currentGroup.add(element)
        if (element is Member) {
            result.add(element to currentGroup)
            currentGroup = ArrayList()
        }
    }
    if (result.isNotEmpty()) {
        result.last!!.second.addAll(currentGroup)
    }
    return result map { MemberHolder(it.first, it.second) }
}