/*
 * Copyright 2010-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jetbrains.jet.lang.resolve.scopes.receivers

import org.jetbrains.jet.lang.descriptors.*
import org.jetbrains.jet.lang.types.JetType
import org.jetbrains.jet.lang.resolve.scopes.JetScope
import org.jetbrains.jet.lang.psi.JetSimpleNameExpression
import org.jetbrains.jet.lang.resolve.DescriptorUtils
import org.jetbrains.jet.lang.resolve.DescriptorUtils.getFqName
import org.jetbrains.jet.lang.resolve.name.Name
import org.jetbrains.jet.lang.resolve.scopes.ChainedScope
import java.util.ArrayList
import org.jetbrains.jet.utils.addIfNotNull
import org.jetbrains.jet.lang.resolve.BindingContext.*
import org.jetbrains.jet.lang.diagnostics.Errors.*
import org.jetbrains.jet.lang.types.expressions.ExpressionTypingContext
import org.jetbrains.jet.lang.psi.psiUtil.getTopmostParentQualifiedExpressionForSelector
import org.jetbrains.jet.lang.resolve.descriptorUtil.getClassObjectReferenceTarget
import org.jetbrains.jet.lang.psi.JetExpression
import org.jetbrains.jet.lang.resolve.bindingContextUtil.recordScopeAndDataFlowInfo
import kotlin.properties.Delegates

public trait Qualifier {

    public val expression: JetExpression

    public val packageView: PackageViewDescriptor?

    public val classifier: ClassifierDescriptor?

    public val name: Name
        get() = classifier?.getName() ?: packageView!!.getName()

    // package, classifier or class object descriptor
    public val resultingDescriptor: DeclarationDescriptor

    public val scope: JetScope
}

class QualifierReceiver (
        val referenceExpression: JetSimpleNameExpression,
        override val packageView: PackageViewDescriptor?,
        override val classifier: ClassifierDescriptor?
) : Qualifier, ReceiverValue {

    override val expression: JetExpression = referenceExpression.getTopmostParentQualifiedExpressionForSelector() ?: referenceExpression

    val descriptor: DeclarationDescriptor
        get() = classifier ?: packageView ?: throw AssertionError("PackageView and classifier both are null")

    override var resultingDescriptor: DeclarationDescriptor by Delegates.notNull()

    override val scope: JetScope get() {
        val scopes = listOf(classifier?.getClassObjectType()?.getMemberScope(), getNestedClassesAndPackageMembersScope()).filterNotNull().copyToArray()
        return ChainedScope(descriptor, "Member scope for " + name + " as package or class or object", *scopes as Array<JetScope?>)
    }

    fun getClassObjectReceiver(): ReceiverValue =
            classifier?.getClassObjectType()?.let { ExpressionReceiver(referenceExpression, it) } ?: ReceiverValue.NO_RECEIVER

    fun getNestedClassesAndPackageMembersScope(): JetScope {
        val scopes = ArrayList<JetScope>(4)

        scopes.addIfNotNull(packageView?.getMemberScope())

        if (classifier is ClassDescriptor) {
            scopes.add(classifier.getStaticScope())

            val classObjectDescriptor = classifier.getClassObjectDescriptor()
            if (classObjectDescriptor != null) {
                // non-static members are resolved through class object receiver
                scopes.add(DescriptorUtils.getStaticNestedClassesScope(classObjectDescriptor))
            }

            if (classifier.getKind() != ClassKind.ENUM_ENTRY) {
                scopes.add(DescriptorUtils.getStaticNestedClassesScope(classifier))
            }
        }

        return ChainedScope(descriptor, "Static scope for " + name + " as package or class or object", *scopes.copyToArray())
    }

    override fun getType(): JetType = throw IllegalStateException("No type corresponds to QualifierReceiver '$this'")

    override fun exists() = true

    override fun toString() = "Package{$packageView} OR Class{$classifier}"
}

fun createQualifier(
        expression: JetSimpleNameExpression,
        receiver: ReceiverValue,
        context: ExpressionTypingContext
): QualifierReceiver? {
    val receiverScope = when {
        !receiver.exists() -> context.scope
        receiver is QualifierReceiver -> receiver.scope
        else -> receiver.getType().getMemberScope()
    }

    val name = expression.getReferencedNameAsName()
    val packageViewDescriptor = receiverScope.getPackage(name)
    val classifierDescriptor = receiverScope.getClassifier(name)

    if (packageViewDescriptor == null && classifierDescriptor == null) return null

    context.recordScopeAndDataFlowInfo(expression)

    val qualifier = QualifierReceiver(expression, packageViewDescriptor, classifierDescriptor)
    context.trace.record(QUALIFIER, qualifier.expression, qualifier)
    return qualifier
}

private fun QualifierReceiver.resolveAsStandaloneExpression(context: ExpressionTypingContext): JetType? {
    resolveAndRecordReferenceTarget(context, selector = null)
    if (classifier is TypeParameterDescriptor) {
        context.trace.report(TYPE_PARAMETER_IS_NOT_AN_EXPRESSION.on(referenceExpression, classifier))
    }
    else if (classifier is ClassDescriptor && classifier.getClassObjectType() == null) {
        context.trace.report(NO_CLASS_OBJECT.on(referenceExpression, classifier))
    }
    else if (packageView != null) {
        context.trace.report(EXPRESSION_EXPECTED_PACKAGE_FOUND.on(referenceExpression))
    }
    return null
}

private fun QualifierReceiver.resolveAsReceiverInQualifiedExpression(context: ExpressionTypingContext, selector: DeclarationDescriptor?) {
    resolveAndRecordReferenceTarget(context, selector)
    if (classifier is TypeParameterDescriptor) {
        context.trace.report(TYPE_PARAMETER_ON_LHS_OF_DOT.on(referenceExpression, classifier as TypeParameterDescriptor))
    }
    else if (classifier is ClassDescriptor && classifier.getClassObjectDescriptor() != null) {
        context.trace.record(EXPRESSION_TYPE, referenceExpression, classifier.getClassObjectType())
    }
}

private fun QualifierReceiver.resolveAndRecordReferenceTarget(context: ExpressionTypingContext, selector: DeclarationDescriptor?) {
    resultingDescriptor = resolveReferenceTarget(selector)
    context.trace.record(REFERENCE_TARGET, referenceExpression, resultingDescriptor)
}

private fun QualifierReceiver.resolveReferenceTarget(selector: DeclarationDescriptor?): DeclarationDescriptor {
    if (classifier is TypeParameterDescriptor) {
        return classifier
    }

    val containingDeclaration = when {
        selector is ConstructorDescriptor -> selector.getContainingDeclaration().getContainingDeclaration()
        else -> selector?.getContainingDeclaration()
    }

    if (packageView != null && (containingDeclaration is PackageFragmentDescriptor || containingDeclaration is PackageViewDescriptor)
            && getFqName(packageView) == getFqName(containingDeclaration)) {
        return packageView
    }

    if (classifier != null && containingDeclaration is ClassDescriptor && classifier == containingDeclaration) {
        return classifier
    }
    if (classifier is ClassDescriptor && classifier.getClassObjectDescriptor() != null) {
        return classifier.getClassObjectReferenceTarget()
    }

    return descriptor
}
