/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jetbrains.kotlin.resolve.scopes

import org.jetbrains.kotlin.descriptors.*
import org.jetbrains.kotlin.incremental.components.LookupLocation
import org.jetbrains.kotlin.name.Name
import org.jetbrains.kotlin.types.KotlinType
import org.jetbrains.kotlin.utils.Printer

/**
 * Introduces a simple wrapper for internal scope.
 */
public abstract class AbstractScopeAdapter : KtScope {
    protected abstract val workerScope: KtScope

    public fun getActualScope(): KtScope =
            if (workerScope is AbstractScopeAdapter)
                (workerScope as AbstractScopeAdapter).getActualScope()
            else
                workerScope

    override fun getImplicitReceiversHierarchy(): List<ReceiverParameterDescriptor> {
        return workerScope.getImplicitReceiversHierarchy()
    }

    override fun getFunctions(name: Name, location: LookupLocation): Collection<FunctionDescriptor> {
        return workerScope.getFunctions(name, location)
    }

    override fun getPackage(name: Name): PackageViewDescriptor? {
        return workerScope.getPackage(name)
    }

    override fun getClassifier(name: Name, location: LookupLocation): ClassifierDescriptor? {
        return workerScope.getClassifier(name, location)
    }

    override fun getProperties(name: Name, location: LookupLocation): Collection<VariableDescriptor> {
        return workerScope.getProperties(name, location)
    }

    override fun getSyntheticExtensionProperties(receiverTypes: Collection<KotlinType>, name: Name, location: LookupLocation): Collection<PropertyDescriptor> {
        return workerScope.getSyntheticExtensionProperties(receiverTypes, name, location)
    }

    override fun getSyntheticExtensionFunctions(receiverTypes: Collection<KotlinType>, name: Name, location: LookupLocation): Collection<FunctionDescriptor> {
        return workerScope.getSyntheticExtensionFunctions(receiverTypes, name, location)
    }

    override fun getSyntheticExtensionProperties(receiverTypes: Collection<KotlinType>): Collection<PropertyDescriptor> {
        return workerScope.getSyntheticExtensionProperties(receiverTypes)
    }

    override fun getSyntheticExtensionFunctions(receiverTypes: Collection<KotlinType>): Collection<FunctionDescriptor> {
        return workerScope.getSyntheticExtensionFunctions(receiverTypes)
    }

    override fun getLocalVariable(name: Name): VariableDescriptor? {
        return workerScope.getLocalVariable(name)
    }

    override fun getContainingDeclaration(): DeclarationDescriptor {
        return workerScope.getContainingDeclaration()
    }

    override fun getDeclarationsByLabel(labelName: Name): Collection<DeclarationDescriptor> {
        return workerScope.getDeclarationsByLabel(labelName)
    }

    override fun getDescriptors(kindFilter: DescriptorKindFilter,
                                nameFilter: (Name) -> Boolean): Collection<DeclarationDescriptor> {
        return workerScope.getDescriptors(kindFilter, nameFilter)
    }

    override fun getOwnDeclaredDescriptors(): Collection<DeclarationDescriptor> {
        return workerScope.getOwnDeclaredDescriptors()
    }

    override fun printScopeStructure(p: Printer) {
        p.println(javaClass.getSimpleName(), " {")
        p.pushIndent()

        p.print("worker =")
        workerScope.printScopeStructure(p.withholdIndentOnce())

        p.popIndent()
        p.println("}")
    }
}
