/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jetbrains.kotlin.resolve.constants

import org.jetbrains.kotlin.builtins.KotlinBuiltIns
import org.jetbrains.kotlin.descriptors.annotations.Annotations
import org.jetbrains.kotlin.types.*

interface CompileTimeConstant<out T> {
    val isError: Boolean
        get() = false

    val parameters: CompileTimeConstant.Parameters

    fun toConstantValue(expectedType: KotlinType): ConstantValue<T>

    fun getValue(expectedType: KotlinType): T = toConstantValue(expectedType).value

    val canBeUsedInAnnotations: Boolean get() = parameters.canBeUsedInAnnotation

    val usesVariableAsConstant: Boolean get() = parameters.usesVariableAsConstant

    val usesNonConstValAsConstant: Boolean get() = parameters.usesNonConstValAsConstant

    val isPure: Boolean get() = parameters.isPure

    class Parameters(
            val canBeUsedInAnnotation: Boolean,
            val isPure: Boolean,
            val usesVariableAsConstant: Boolean,
            val usesNonConstValAsConstant: Boolean
    )
}

class TypedCompileTimeConstant<out T>(
        val constantValue: ConstantValue<T>,
        override val parameters: CompileTimeConstant.Parameters
) : CompileTimeConstant<T> {
    override val isError: Boolean
        get() = constantValue is ErrorValue

    val type: KotlinType = constantValue.type

    override fun toConstantValue(expectedType: KotlinType): ConstantValue<T> = constantValue
}

class IntegerValueTypeConstant(
        private val value: Number,
        private val builtIns: KotlinBuiltIns,
        override val parameters: CompileTimeConstant.Parameters
) : CompileTimeConstant<Number> {
    private val typeConstructor = IntegerValueTypeConstructor(value.toLong(), builtIns)

    override fun toConstantValue(expectedType: KotlinType): ConstantValue<Number> {
        val factory = ConstantValueFactory(builtIns)
        val type = getType(expectedType)
        return when {
            KotlinBuiltIns.isInt(type) -> {
                factory.createIntValue(value.toInt())
            }
            KotlinBuiltIns.isByte(type) -> {
                factory.createByteValue(value.toByte())
            }
            KotlinBuiltIns.isShort(type) -> {
                factory.createShortValue(value.toShort())
            }
            else -> {
                factory.createLongValue(value.toLong())
            }
        }
    }

    val unknownIntegerType = KotlinTypeImpl.create(
            Annotations.EMPTY, typeConstructor, false, emptyList<TypeProjection>(),
            ErrorUtils.createErrorScope("Scope for number value type (" + typeConstructor.toString() + ")", true)
    )

    fun getType(expectedType: KotlinType): KotlinType = TypeUtils.getPrimitiveNumberType(typeConstructor, expectedType)

    override fun toString() = typeConstructor.toString()
}
