/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jetbrains.kotlin.psi;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;
import com.intellij.openapi.project.Project;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.kotlin.resolve.ImportPath;

import java.util.Collection;
import java.util.Map;

public class KtImportsFactory {
    @NotNull private final Project project;

    private final Map<ImportPath, KtImportDirective> importsCache = Maps.newHashMap();

    public KtImportsFactory(@NotNull Project project) {
        this.project = project;
    }

    @NotNull
    public KtImportDirective createImportDirective(@NotNull ImportPath importPath) {
        KtImportDirective directive = importsCache.get(importPath);
        if (directive != null) {
            return directive;
        }

        KtImportDirective createdDirective = KtPsiFactoryKt.KtPsiFactory(project, false).createImportDirective(importPath);
        importsCache.put(importPath, createdDirective);

        return createdDirective;
    }

    @NotNull
    public Collection<KtImportDirective> createImportDirectives(@NotNull Collection<ImportPath> importPaths) {
        return Collections2.transform(importPaths,
                                      new Function<ImportPath, KtImportDirective>() {
                                          @Override
                                          public KtImportDirective apply(@Nullable ImportPath path) {
                                              assert path != null;
                                              return createImportDirective(path);
                                          }
                                      });
    }
}
