/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.util.io;

import com.intellij.openapi.Forceable;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.util.io.Bits;
import com.intellij.util.io.ByteBufferWrapper;
import com.intellij.util.io.IOStatistics;
import com.intellij.util.io.MappingFailedException;
import com.intellij.util.io.Page;
import com.intellij.util.io.PagedFileStorageCache;
import com.intellij.util.io.PersistentHashMapValueStorage;
import com.intellij.util.io.StorageLock;
import com.intellij.util.io.StorageLockContext;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Arrays;
import org.jetbrains.annotations.Nullable;

public class PagedFileStorage
implements Forceable {
    private static final Logger LOG = Logger.getInstance(PagedFileStorage.class);
    public static final int BUFFER_SIZE = StorageLock.BUFFER_SIZE;
    private static final ByteOrder ourNativeByteOrder = ByteOrder.nativeOrder();
    private static final ThreadLocal<byte[]> ourTypedIOBuffer = ThreadLocal.withInitial(() -> new byte[8]);
    static final StorageLock ourLock = new StorageLock();
    public static final ThreadLocal<StorageLockContext> THREAD_LOCAL_STORAGE_LOCK_CONTEXT = new ThreadLocal();
    private final StorageLockContext myStorageLockContext;
    private final boolean myNativeBytesOrder;
    private int myStorageIndex;
    private final PagedFileStorageCache myLastAccessedBufferCache = new PagedFileStorageCache();
    private final Path myFile;
    private final boolean myReadOnly;
    protected final int myPageSize;
    protected final boolean myValuesAreBufferAligned;
    private volatile boolean isDirty;
    private volatile long mySize = -1L;

    public PagedFileStorage(Path file2, @Nullable StorageLockContext storageLockContext, int pageSize, boolean valuesAreBufferAligned, boolean nativeBytesOrder) {
        this.myFile = file2;
        this.myReadOnly = PersistentHashMapValueStorage.CreationTimeOptions.READONLY.get() == Boolean.TRUE;
        StorageLockContext context = THREAD_LOCAL_STORAGE_LOCK_CONTEXT.get();
        if (context != null) {
            if (storageLockContext != null && storageLockContext != context) {
                throw new IllegalStateException();
            }
            storageLockContext = context;
        }
        this.myStorageLockContext = storageLockContext != null ? storageLockContext : PagedFileStorage.ourLock.myDefaultContext;
        this.myPageSize = Math.max(pageSize > 0 ? pageSize : BUFFER_SIZE, Page.PAGE_SIZE);
        this.myValuesAreBufferAligned = valuesAreBufferAligned;
        this.myStorageIndex = this.myStorageLockContext.getStorageLock().registerPagedFileStorage(this);
        this.myNativeBytesOrder = nativeBytesOrder;
    }

    public int getPageSize() {
        return this.myPageSize;
    }

    public void lockRead() {
        this.myStorageLockContext.lockRead();
    }

    public void unlockRead() {
        this.myStorageLockContext.unlockRead();
    }

    public void lockWrite() {
        this.myStorageLockContext.lockWrite();
    }

    public void unlockWrite() {
        this.myStorageLockContext.unlockWrite();
    }

    public StorageLockContext getStorageLockContext() {
        return this.myStorageLockContext;
    }

    public Path getFile() {
        return this.myFile;
    }

    public void putInt(long addr, int value2) {
        if (this.myValuesAreBufferAligned) {
            long page = addr / (long)this.myPageSize;
            int page_offset = (int)(addr % (long)this.myPageSize);
            this.getBuffer(page).putInt(page_offset, value2);
        } else {
            Bits.putInt(PagedFileStorage.getThreadLocalTypedIOBuffer(), 0, value2);
            this.put(addr, PagedFileStorage.getThreadLocalTypedIOBuffer(), 0, 4);
        }
    }

    public int getInt(long addr) {
        if (this.myValuesAreBufferAligned) {
            long page = addr / (long)this.myPageSize;
            int page_offset = (int)(addr % (long)this.myPageSize);
            return this.getReadOnlyBuffer(page).getInt(page_offset);
        }
        this.get(addr, PagedFileStorage.getThreadLocalTypedIOBuffer(), 0, 4);
        return Bits.getInt(PagedFileStorage.getThreadLocalTypedIOBuffer(), 0);
    }

    public int getOffsetInPage(long addr) {
        return (int)(addr % (long)this.myPageSize);
    }

    public ByteBufferWrapper getByteBuffer(long address, boolean modify) {
        long page = address / (long)this.myPageSize;
        assert (page >= 0L && page <= 65535L) : address + " in " + this.myFile;
        return this.getBufferWrapper(page, modify, this.myReadOnly);
    }

    public void putLong(long addr, long value2) {
        if (this.myValuesAreBufferAligned) {
            long page = addr / (long)this.myPageSize;
            int page_offset = (int)(addr % (long)this.myPageSize);
            this.getBuffer(page).putLong(page_offset, value2);
        } else {
            Bits.putLong(PagedFileStorage.getThreadLocalTypedIOBuffer(), 0, value2);
            this.put(addr, PagedFileStorage.getThreadLocalTypedIOBuffer(), 0, 8);
        }
    }

    public long getLong(long addr) {
        if (this.myValuesAreBufferAligned) {
            long page = addr / (long)this.myPageSize;
            int page_offset = (int)(addr % (long)this.myPageSize);
            return this.getReadOnlyBuffer(page).getLong(page_offset);
        }
        this.get(addr, PagedFileStorage.getThreadLocalTypedIOBuffer(), 0, 8);
        return Bits.getLong(PagedFileStorage.getThreadLocalTypedIOBuffer(), 0);
    }

    public byte get(long index2) {
        long page = index2 / (long)this.myPageSize;
        int offset2 = (int)(index2 % (long)this.myPageSize);
        return this.getReadOnlyBuffer(page).get(offset2);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void get(long index2, byte[] dst, int offset2, int length) {
        long i = index2;
        int o = offset2;
        int l = length;
        while (l > 0) {
            ByteBuffer buffer;
            long page = i / (long)this.myPageSize;
            int page_offset = (int)(i % (long)this.myPageSize);
            int page_len = Math.min(l, this.myPageSize - page_offset);
            ByteBuffer byteBuffer = buffer = this.getReadOnlyBuffer(page);
            synchronized (byteBuffer) {
                try {
                    buffer.position(page_offset);
                }
                catch (IllegalArgumentException iae) {
                    throw new IllegalArgumentException("can't position buffer to offset " + page_offset + ", buffer.limit=" + buffer.limit() + ", page=" + page + ", file=" + this.myFile.getFileName() + ", file.length=" + this.length());
                }
                buffer.get(dst, o, page_len);
            }
            l -= page_len;
            o += page_len;
            i += (long)page_len;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void put(long index2, byte[] src, int offset2, int length) {
        long i = index2;
        int o = offset2;
        int l = length;
        while (l > 0) {
            ByteBuffer buffer;
            long page = i / (long)this.myPageSize;
            int page_offset = (int)(i % (long)this.myPageSize);
            int page_len = Math.min(l, this.myPageSize - page_offset);
            ByteBuffer byteBuffer = buffer = this.getBuffer(page);
            synchronized (byteBuffer) {
                try {
                    buffer.position(page_offset);
                }
                catch (IllegalArgumentException iae) {
                    throw new IllegalArgumentException("can't position buffer to offset " + page_offset);
                }
                buffer.put(src, o, page_len);
            }
            l -= page_len;
            o += page_len;
            i += (long)page_len;
        }
    }

    public void close() {
        try {
            this.force();
        }
        finally {
            this.unmapAll();
            this.myStorageLockContext.getStorageLock().removeStorage(this.myStorageIndex);
            this.myStorageIndex = -1;
        }
    }

    private void unmapAll() {
        this.myStorageLockContext.getStorageLock().unmapBuffersForOwner(this.myStorageIndex, this.myStorageLockContext, false);
        this.myLastAccessedBufferCache.clear();
    }

    public void resize(long newSize) throws IOException {
        long finished;
        long oldSize;
        long l = oldSize = Files.exists(this.myFile, new LinkOption[0]) ? Files.size(this.myFile) : 0L;
        if (oldSize == newSize && oldSize == this.length()) {
            return;
        }
        long started = IOStatistics.DEBUG ? System.currentTimeMillis() : 0L;
        this.myStorageLockContext.getStorageLock().invalidateBuffer(this.myStorageIndex | (int)(oldSize / (long)this.myPageSize));
        long unmapAllFinished = IOStatistics.DEBUG ? System.currentTimeMillis() : 0L;
        this.resizeFile(newSize);
        long delta = newSize - oldSize;
        if (delta > 0L) {
            this.fillWithZeros(oldSize, delta);
        }
        if (IOStatistics.DEBUG && (finished = System.currentTimeMillis()) - started > 100L) {
            IOStatistics.dump("Resized " + this.myFile + " from " + oldSize + " to " + newSize + " for " + (finished - started) + ", unmap all:" + (finished - unmapAllFinished));
        }
    }

    private void resizeFile(long newSize) throws IOException {
        this.mySize = -1L;
        try (RandomAccessFile raf = new RandomAccessFile(this.myFile.toFile(), "rw");){
            raf.setLength(newSize);
        }
        this.mySize = newSize;
    }

    private void fillWithZeros(long from2, long length) {
        byte[] buff = new byte[8192];
        Arrays.fill(buff, (byte)0);
        while (length > 0L) {
            int filled = Math.min((int)length, 8192);
            this.put(from2, buff, 0, filled);
            length -= (long)filled;
            from2 += (long)filled;
        }
    }

    public final long length() {
        long size = this.mySize;
        if (size == -1L) {
            if (Files.exists(this.myFile, new LinkOption[0])) {
                try {
                    this.mySize = size = Files.size(this.myFile);
                }
                catch (IOException e) {
                    LOG.error(e);
                }
            } else {
                size = 0L;
                this.mySize = 0L;
            }
        }
        return size;
    }

    private ByteBuffer getBuffer(long page) {
        return this.getBufferWrapper(page, true, this.myReadOnly).getCachedBuffer();
    }

    private ByteBuffer getReadOnlyBuffer(long page) {
        return this.getBufferWrapper(page, false, this.myReadOnly).getCachedBuffer();
    }

    private ByteBufferWrapper getBufferWrapper(long page, boolean modify, boolean readOnly) {
        ByteBufferWrapper pageFromCache = this.myLastAccessedBufferCache.getPageFromCache(page, this.myStorageLockContext.getStorageLock().getMappingChangeCount(), readOnly);
        if (pageFromCache != null) {
            if (modify) {
                this.markDirty(pageFromCache);
            }
            return pageFromCache;
        }
        try {
            assert (page >= 0L && page <= 65535L) : page;
            if (this.myStorageIndex == -1) {
                throw new IOException("storage is already closed; path " + this.myFile);
            }
            ByteBufferWrapper byteBufferWrapper = this.myStorageLockContext.getStorageLock().get(this.myStorageIndex | (int)page, !modify, readOnly);
            if (modify) {
                this.markDirty(byteBufferWrapper);
            }
            ByteBuffer buf = byteBufferWrapper.getBuffer();
            if (this.myNativeBytesOrder && buf.order() != ourNativeByteOrder) {
                buf.order(ourNativeByteOrder);
            }
            this.myLastAccessedBufferCache.updateCache(page, byteBufferWrapper, this.myStorageLockContext.getStorageLock().getMappingChangeCount());
            return byteBufferWrapper;
        }
        catch (IOException e) {
            throw new MappingFailedException("Cannot map buffer", e);
        }
    }

    private void markDirty(ByteBufferWrapper buffer) {
        if (!this.isDirty) {
            this.isDirty = true;
        }
        buffer.markDirty();
    }

    private static byte[] getThreadLocalTypedIOBuffer() {
        return ourTypedIOBuffer.get();
    }

    @Override
    public void force() {
        long finished;
        long started;
        long l = started = IOStatistics.DEBUG ? System.currentTimeMillis() : 0L;
        if (this.isDirty) {
            this.myStorageLockContext.getStorageLock().flushBuffersForOwner(this.myStorageIndex, this.myStorageLockContext);
            this.isDirty = false;
        }
        if (IOStatistics.DEBUG && (finished = System.currentTimeMillis()) - started > 100L) {
            IOStatistics.dump("Flushed " + this.myFile + " for " + (finished - started));
        }
    }

    public boolean isDirty() {
        return this.isDirty;
    }
}

