/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package org.jetbrains.kotlin.fir.extensions

import org.jetbrains.kotlin.fir.FirSession
import org.jetbrains.kotlin.fir.declarations.FirClassLikeDeclaration
import org.jetbrains.kotlin.fir.declarations.FirRegularClass
import org.jetbrains.kotlin.fir.types.ConeKotlinType
import org.jetbrains.kotlin.fir.types.FirResolvedTypeRef
import org.jetbrains.kotlin.fir.types.FirUserTypeRef
import kotlin.reflect.KClass

abstract class FirSupertypeGenerationExtension(session: FirSession) : FirExtension(session) {
    companion object {
        val NAME: FirExtensionPointName = FirExtensionPointName("SupertypeGenerator")
    }

    final override val name: FirExtensionPointName
        get() = NAME

    final override val extensionType: KClass<out FirExtension> = FirSupertypeGenerationExtension::class

    abstract fun needTransformSupertypes(declaration: FirClassLikeDeclaration): Boolean

    abstract fun computeAdditionalSupertypes(
        classLikeDeclaration: FirClassLikeDeclaration,
        resolvedSupertypes: List<FirResolvedTypeRef>,
        typeResolver: TypeResolveService
    ): List<ConeKotlinType>

    /**
     * This function allows adding new supertypes to some nested class, which was generated by the plugin.
     * The intended use of this function is for the case when the supertype of the generated class depends
     *   on some types from annotation arguments.
     * ```
     * @AddNestedClassesBasedOnArgument(XXX::class)
     * interface Some {
     *     class Generated : XXX()
     * }
     * ```
     *
     * If some new types will be generated, then default `Any` supertype will be automatically removed
     *
     * IMPORTANT: this method works only on one level, which means that it won't be called for the nested class of the generated class.
     *   It also doesn't work for top-level classes
     */
    @ExperimentalSupertypesGenerationApi
    open fun computeAdditionalSupertypesForGeneratedNestedClass(
        klass: FirRegularClass,
        typeResolver: TypeResolveService
    ): List<FirResolvedTypeRef> = emptyList()

    fun interface Factory : FirExtension.Factory<FirSupertypeGenerationExtension>

    abstract class TypeResolveService {
        abstract fun resolveUserType(type: FirUserTypeRef): FirResolvedTypeRef
    }
}

val FirExtensionService.supertypeGenerators: List<FirSupertypeGenerationExtension> by FirExtensionService.registeredExtensions()

@RequiresOptIn("This API is experimental and works for a limited number of cases", level = RequiresOptIn.Level.ERROR)
annotation class ExperimentalSupertypesGenerationApi
