/*
 * Copyright 2010-2022 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package org.jetbrains.kotlin.fir.extensions

import org.jetbrains.kotlin.fir.FirSession
import org.jetbrains.kotlin.fir.SessionAndScopeSessionHolder
import org.jetbrains.kotlin.fir.declarations.FirDeclarationDataKey
import org.jetbrains.kotlin.fir.declarations.FirDeclarationDataRegistry
import org.jetbrains.kotlin.fir.declarations.FirReceiverParameter
import org.jetbrains.kotlin.fir.expressions.FirFunctionCall
import org.jetbrains.kotlin.fir.resolve.calls.ImplicitExtensionReceiverValue
import org.jetbrains.kotlin.fir.symbols.impl.FirCallableSymbol
import org.jetbrains.kotlin.fir.symbols.impl.FirReceiverParameterSymbol
import kotlin.reflect.KClass

abstract class FirExpressionResolutionExtension(session: FirSession) : FirExtension(session) {
    companion object {
        val NAME: FirExtensionPointName = FirExtensionPointName("ExpressionResolutionExtension")
    }

    final override val name: FirExtensionPointName
        get() = NAME

    final override val extensionType: KClass<out FirExtension> = FirExpressionResolutionExtension::class

    abstract fun addNewImplicitReceivers(
        functionCall: FirFunctionCall,
        sessionHolder: SessionAndScopeSessionHolder,
        containingCallableSymbol: FirCallableSymbol<*>,
    ): List<ImplicitExtensionReceiverValue>

    fun interface Factory : FirExtension.Factory<FirExpressionResolutionExtension>
}

val FirExtensionService.expressionResolutionExtensions: List<FirExpressionResolutionExtension> by FirExtensionService.registeredExtensions()

/**
 * Controls whether the CodeFragmentCapturedValueAnalyzer (part of KaFirCompilerFacility) should capture this receiver parameter.
 * Set to false - the debugger won't provide the runtime value of this parameter to compiled code fragment in evaluate expression.
 * This in turn means that the origin plugin must provide IR lowering to generate code that can be evaluated without these values,
 * e.g., through inlining or in-place instance creation.
 *
 * True for receivers not generated by plugin,
 * or when plugin lowering guarantees runtime stack values are available for this parameter.
 */
var FirReceiverParameter.captureValueInAnalyze: Boolean? by FirDeclarationDataRegistry.data(CaptureValueInAnalyzeKey)
val FirReceiverParameterSymbol.captureValueInAnalyze: Boolean get() = fir.captureValueInAnalyze ?: true

private object CaptureValueInAnalyzeKey : FirDeclarationDataKey()