package com.github.ajalt.clikt.sources

import com.github.ajalt.clikt.core.Context
import com.github.ajalt.clikt.parameters.options.*

/**
 * A [ValueSource] that reads values from a map.
 *
 * This implementation will only return a single value for each option. If you use conversions like
 * [pair] or [triple], you'll need to implement a [ValueSource] yourself.
 *
 * @param values The map of key to value for each option
 * @param getKey A function that return the key in [values] for a given option. By default, it joins the
 */
@ExperimentalValueSourceApi
class MapValueSource(
        private val values: Map<String, String>,
        private val getKey: (Context, Option) -> String = defaultKey
) : ValueSource {
    override fun getValues(context: Context, option: Option): List<ValueSource.Invocation> {
        return values[getKey(context, option)]
                ?.let { ValueSource.Invocation.just(it) }.orEmpty()
    }

    companion object {
        /**
         * Return a map key for an option.
         *
         * The key is generated by joining the parents of the option's command (if any) with the
         * value returned by [ValueSource.name].
         *
         * ## Examples
         *
         * - For an option named `--option` in a command with no parents, return "option"
         * - For an option named `--option` in a command that is the subcommand of a command named `root`, return "root.option"
         */
        val defaultKey: (Context, Option) -> String = { context, option ->
            (context.commandNameWithParents().dropLast(1) + ValueSource.name(option)).joinToString(".")
        }
    }
}
