/*
 * Copyright 2000-2024 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.jetbrains;

import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.util.EnumMap;
import java.util.Map;
import java.util.function.Function;

/**
 * Entry point into JBR API.
 * Client and JBR side are linked dynamically at runtime and do not have to be of the same version.
 * In some cases (e.g. running on different JRE or old JBR) system will not be able to find
 * implementation for some services, so you'll need a fallback behavior for that case.
 * <h2>Simple usage example:</h2>
 * <blockquote><pre>{@code
 * if (JBR.isSomeServiceSupported()) {
 *     JBR.getSomeService().doSomething();
 * } else {
 *     planB();
 * }
 * }</pre></blockquote>
 * <h3>Implementation note:</h3>
 * JBR API is initialized on first access to this class (in static initializer).
 * Actual implementation is linked on demand, when corresponding service is requested by client.
 */
public final class JBR {

    private static final ServiceApi api;
    private static final Throwable bootstrapException;
    static {
        ServiceApi a = null;
        Throwable exception = null;
        try {
            MethodHandles.Lookup lookup = MethodHandles.lookup();
            try { // New version of bootstrap method
                Class<?> bootstrap = Class.forName("com.jetbrains.exported.JBRApiSupport");
                a = (ServiceApi) (Object) lookup
                        .findStatic(bootstrap, "bootstrap", MethodType.methodType(Object.class, Class.class,
                                Class.class, Class.class, Class.class, Map.class, Function.class))
                        .invokeExact(ServiceApi.class, Service.class, Provided.class, Provides.class,
                                Metadata.KNOWN_EXTENSIONS, Metadata.EXTENSION_EXTRACTOR);
            } catch (IllegalAccessException | ClassNotFoundException | NoSuchMethodException | NoSuchMethodError ignore) {
                // Old version of bootstrap method
                Class<?> bootstrap = Class.forName("com.jetbrains.bootstrap.JBRApiBootstrap");
                a = (ServiceApi) (Object) lookup
                        .findStatic(bootstrap, "bootstrap", MethodType.methodType(Object.class, MethodHandles.Lookup.class))
                        .invokeExact(lookup);
            }
        }  catch (IllegalAccessException | ClassNotFoundException | NoSuchMethodException | NoSuchMethodError e) {
            exception = e;
        } catch (Throwable e) {
            Throwable t = e.getCause();
            if (t instanceof Error) throw (Error) t;
            else throw new Error(t);
        }
        api = a;
        bootstrapException = exception;
        IMPL_VERSION = api == null ? "UNKNOWN" : api.getImplVersion();
    }

    private static final String IMPL_VERSION, API_VERSION = getApiVersionFromModule();
    private static String getApiVersionFromModule() {
        java.lang.module.ModuleDescriptor descriptor = JBR.class.getModule().getDescriptor();
        if (descriptor != null && descriptor.version().isPresent()) {
            return descriptor.version().get().toString();
        } else {
            return "SNAPSHOT";
        }
    }

    private JBR() {}

    private static <T> T getServiceWithFallback(Class<T> interFace, FallbackSupplier<T> fallback, Extensions... extensions) {
        T service = getService(interFace, extensions);
        try {
            return service != null ? service : fallback != null ? fallback.get() : null;
        } catch (Throwable ignore) {
            return null;
        }
    }

    static <T> T getService(Class<T> interFace, Extensions... extensions) {
        return api == null ? null : api.getService(interFace, extensions);
    }

    /**
     * Checks whether JBR API is available at runtime.
     * @return true when running on JBR which implements JBR API
     */
    public static boolean isAvailable() {
        return api != null;
    }

    /**
     * Returns JBR API version.
     * Development versions of JBR API return "SNAPSHOT".
     * When running on Java 8, returns "UNKNOWN".
     * <h4>Note:</h4>
     * This is an API version, which comes with client application, it is *almost*
     * a compile-time constant and has nothing to do with JRE it runs on.
     * @return JBR API version in form {@code MAJOR.MINOR.PATCH}, or "SNAPSHOT" / "UNKNOWN".
     */
    public static String getApiVersion() {
        return API_VERSION;
    }

    /**
     * Returns JBR API version supported by current runtime or "UNKNOWN".
     * <h4>Note:</h4>
     * This method can return "UNKNOWN" even when JBR API {@link #isAvailable()}.
     * @return JBR API version supported by current implementation or "UNKNOWN".
     */
    public static String getImplVersion() {
        return IMPL_VERSION;
    }

    /**
     * Checks whether given {@linkplain com.jetbrains.Extensions extension} is supported.
     * @param extension extension to check
     * @return true if extension is supported
     */
    public static boolean isExtensionSupported(Extensions extension) {
        return api != null && api.isExtensionSupported(extension);
    }

    /**
     * Internal API interface, contains most basic methods for communication between client and JBR.
     */
    @Service
    @Provided
    private interface ServiceApi {

        <T> T getService(Class<T> interFace);

        default <T> T getService(Class<T> interFace, Enum<?>... extensions) {
            return extensions.length == 0 ? getService(interFace) : null;
        }

        default String getImplVersion() { return "UNKNOWN"; }

        default boolean isExtensionSupported(Enum<?> extension) { return false; }
    }

    @FunctionalInterface
    private interface FallbackSupplier<T> {
        T get() throws Throwable;
    }

    // ========================== Generated metadata ==========================

    /**
     * Generated client-side metadata, needed by JBR when linking the implementation.
     */
    @SuppressWarnings({"rawtypes", "deprecation"})
    private static final class Metadata {
        // Needed only for compatibility.
        private static final String[] KNOWN_SERVICES = {"com.jetbrains.JBR$ServiceApi", "com.jetbrains.AccessibleAnnouncer", "com.jetbrains.DesktopActions", "com.jetbrains.FontExtensions", "com.jetbrains.FontMetricsAccessor", "com.jetbrains.GraphicsUtils", "com.jetbrains.JBRFileDialogService", "com.jetbrains.Jstack", "com.jetbrains.Keyboard", "com.jetbrains.NativeRasterLoader", "com.jetbrains.ProjectorUtils", "com.jetbrains.RoundedCornersManager", "com.jetbrains.SharedTextures", "com.jetbrains.SystemShortcuts", "com.jetbrains.SystemUtils", "com.jetbrains.TextInput", "com.jetbrains.WindowDecorations", "com.jetbrains.WindowMove"};
        private static final String[] KNOWN_PROXIES = {"com.jetbrains.AccessibleAnnouncer", "com.jetbrains.DesktopActions", "com.jetbrains.FontExtensions", "com.jetbrains.FontMetricsAccessor", "com.jetbrains.GraphicsUtils", "com.jetbrains.JBRFileDialog", "com.jetbrains.JBRFileDialogService", "com.jetbrains.Jstack", "com.jetbrains.Keyboard", "com.jetbrains.NativeRasterLoader", "com.jetbrains.ProjectorUtils", "com.jetbrains.RoundedCornersManager", "com.jetbrains.SharedTextures", "com.jetbrains.SystemShortcuts", "com.jetbrains.SystemShortcuts$Shortcut", "com.jetbrains.SystemUtils", "com.jetbrains.TextInput", "com.jetbrains.TextInput$SelectTextRangeEvent", "com.jetbrains.WindowDecorations", "com.jetbrains.WindowDecorations$CustomTitleBar", "com.jetbrains.WindowMove"};

        private static final Function<java.lang.reflect.Method, Extensions> EXTENSION_EXTRACTOR = m -> {
            Extension e = m.getAnnotation(Extension.class);
            return e == null ? null : e.value();
        };
        private static final Map<Extensions, Class[]> KNOWN_EXTENSIONS = new EnumMap<>(Extensions.class);
        static {
            KNOWN_EXTENSIONS.put(Extensions.SHRINKING_GC, new Class[] {com.jetbrains.SystemUtils.class});
            KNOWN_EXTENSIONS.put(Extensions.BUILTIN_DISPLAY_CHECKER, new Class[] {com.jetbrains.GraphicsUtils.class});
            for (Extensions e : Extensions.values()) KNOWN_EXTENSIONS.putIfAbsent(e, new Class[0]);
        }
    }

    // ======================= Generated static methods =======================

    private static class AccessibleAnnouncer__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<AccessibleAnnouncer> FALLBACK = null;
        private static final AccessibleAnnouncer INSTANCE = getServiceWithFallback(AccessibleAnnouncer.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link AccessibleAnnouncer} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link AccessibleAnnouncer}
     * and its dependencies (can fully implement given service).
     * @see #getAccessibleAnnouncer()
     */
    public static boolean isAccessibleAnnouncerSupported() {
        return AccessibleAnnouncer__Holder.INSTANCE != null;
    }
    
    /**
     * This interface provides the ability to speak a given string using screen readers.
     *
     *
     * @return full implementation of {@link AccessibleAnnouncer} service if any, or {@code null} otherwise
     */
    public static AccessibleAnnouncer getAccessibleAnnouncer() {
        return AccessibleAnnouncer__Holder.INSTANCE;
    }
    
    /**
     * This interface provides the ability to speak a given string using screen readers.
     *
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link AccessibleAnnouncer} service if any, or {@code null} otherwise
     */
    public static AccessibleAnnouncer getAccessibleAnnouncer(Extensions... extensions) {
        return getServiceWithFallback(AccessibleAnnouncer.class, AccessibleAnnouncer__Holder.FALLBACK, extensions);
    }

    private static class DesktopActions__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<DesktopActions> FALLBACK = null;
        private static final DesktopActions INSTANCE = getServiceWithFallback(DesktopActions.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link DesktopActions} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link DesktopActions}
     * and its dependencies (can fully implement given service).
     * @see #getDesktopActions()
     */
    public static boolean isDesktopActionsSupported() {
        return DesktopActions__Holder.INSTANCE != null;
    }
    
    /**
     * Allows desktop actions, like opening a file, or webpage to be overridden.
     * @see java.awt.Desktop
     *
     * @return full implementation of {@link DesktopActions} service if any, or {@code null} otherwise
     */
    public static DesktopActions getDesktopActions() {
        return DesktopActions__Holder.INSTANCE;
    }
    
    /**
     * Allows desktop actions, like opening a file, or webpage to be overridden.
     * @see java.awt.Desktop
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link DesktopActions} service if any, or {@code null} otherwise
     */
    public static DesktopActions getDesktopActions(Extensions... extensions) {
        return getServiceWithFallback(DesktopActions.class, DesktopActions__Holder.FALLBACK, extensions);
    }

    private static class FontExtensions__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<FontExtensions> FALLBACK = null;
        private static final FontExtensions INSTANCE = getServiceWithFallback(FontExtensions.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link FontExtensions} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link FontExtensions}
     * and its dependencies (can fully implement given service).
     * @see #getFontExtensions()
     */
    public static boolean isFontExtensionsSupported() {
        return FontExtensions__Holder.INSTANCE != null;
    }
    
    /**
     * Font-related utilities.
     *
     * @return full implementation of {@link FontExtensions} service if any, or {@code null} otherwise
     */
    public static FontExtensions getFontExtensions() {
        return FontExtensions__Holder.INSTANCE;
    }
    
    /**
     * Font-related utilities.
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link FontExtensions} service if any, or {@code null} otherwise
     */
    public static FontExtensions getFontExtensions(Extensions... extensions) {
        return getServiceWithFallback(FontExtensions.class, FontExtensions__Holder.FALLBACK, extensions);
    }

    private static class FontMetricsAccessor__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<FontMetricsAccessor> FALLBACK = com.jetbrains.FontMetricsAccessor_Fallback::new;
        private static final FontMetricsAccessor INSTANCE = getServiceWithFallback(FontMetricsAccessor.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link FontMetricsAccessor} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link FontMetricsAccessor}
     * and its dependencies (can fully implement given service).
     * @see #getFontMetricsAccessor()
     */
    public static boolean isFontMetricsAccessorSupported() {
        return FontMetricsAccessor__Holder.INSTANCE != null;
    }
    
    /**
     * Provides convenience methods to access {@link java.awt.FontMetrics} instances, and obtain character advances from them without
     * rounding. Also provides an (unsafe) way to override character advances in those instances with arbitrary specified
     * values.
     *
     * @return full implementation of {@link FontMetricsAccessor} service if any, or {@code null} otherwise
     */
    public static FontMetricsAccessor getFontMetricsAccessor() {
        return FontMetricsAccessor__Holder.INSTANCE;
    }
    
    /**
     * Provides convenience methods to access {@link java.awt.FontMetrics} instances, and obtain character advances from them without
     * rounding. Also provides an (unsafe) way to override character advances in those instances with arbitrary specified
     * values.
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link FontMetricsAccessor} service if any, or {@code null} otherwise
     */
    public static FontMetricsAccessor getFontMetricsAccessor(Extensions... extensions) {
        return getServiceWithFallback(FontMetricsAccessor.class, FontMetricsAccessor__Holder.FALLBACK, extensions);
    }

    private static class GraphicsUtils__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<GraphicsUtils> FALLBACK = null;
        private static final GraphicsUtils INSTANCE = getServiceWithFallback(GraphicsUtils.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link GraphicsUtils} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link GraphicsUtils}
     * and its dependencies (can fully implement given service).
     * @see #getGraphicsUtils()
     */
    public static boolean isGraphicsUtilsSupported() {
        return GraphicsUtils__Holder.INSTANCE != null;
    }
    
    /**
     * Graphics2D utilities.
     *
     * @return full implementation of {@link GraphicsUtils} service if any, or {@code null} otherwise
     */
    public static GraphicsUtils getGraphicsUtils() {
        return GraphicsUtils__Holder.INSTANCE;
    }
    
    /**
     * Graphics2D utilities.
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link GraphicsUtils} service if any, or {@code null} otherwise
     */
    public static GraphicsUtils getGraphicsUtils(Extensions... extensions) {
        return getServiceWithFallback(GraphicsUtils.class, GraphicsUtils__Holder.FALLBACK, extensions);
    }

    private static class Jstack__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<Jstack> FALLBACK = null;
        private static final Jstack INSTANCE = getServiceWithFallback(Jstack.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link Jstack} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link Jstack}
     * and its dependencies (can fully implement given service).
     * @see #getJstack()
     */
    public static boolean isJstackSupported() {
        return Jstack__Holder.INSTANCE != null;
    }
    
    /**
     * Jstack-related utilities.
     *
     * @return full implementation of {@link Jstack} service if any, or {@code null} otherwise
     */
    public static Jstack getJstack() {
        return Jstack__Holder.INSTANCE;
    }
    
    /**
     * Jstack-related utilities.
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link Jstack} service if any, or {@code null} otherwise
     */
    public static Jstack getJstack(Extensions... extensions) {
        return getServiceWithFallback(Jstack.class, Jstack__Holder.FALLBACK, extensions);
    }

    private static class Keyboard__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<Keyboard> FALLBACK = null;
        private static final Keyboard INSTANCE = getServiceWithFallback(Keyboard.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link Keyboard} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link Keyboard}
     * and its dependencies (can fully implement given service).
     * @see #getKeyboard()
     */
    public static boolean isKeyboardSupported() {
        return Keyboard__Holder.INSTANCE != null;
    }
    
    /**
     * JBR API to inspect additional properties of AWT key events and keyboards.
     *
     * @return full implementation of {@link Keyboard} service if any, or {@code null} otherwise
     */
    public static Keyboard getKeyboard() {
        return Keyboard__Holder.INSTANCE;
    }
    
    /**
     * JBR API to inspect additional properties of AWT key events and keyboards.
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link Keyboard} service if any, or {@code null} otherwise
     */
    public static Keyboard getKeyboard(Extensions... extensions) {
        return getServiceWithFallback(Keyboard.class, Keyboard__Holder.FALLBACK, extensions);
    }

    private static class NativeRasterLoader__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<NativeRasterLoader> FALLBACK = null;
        private static final NativeRasterLoader INSTANCE = getServiceWithFallback(NativeRasterLoader.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link NativeRasterLoader} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link NativeRasterLoader}
     * and its dependencies (can fully implement given service).
     * @see #getNativeRasterLoader()
     */
    public static boolean isNativeRasterLoaderSupported() {
        return NativeRasterLoader__Holder.INSTANCE != null;
    }
    
    /**
     * Direct raster loading for VolatileImage.
     *
     * @return full implementation of {@link NativeRasterLoader} service if any, or {@code null} otherwise
     */
    public static NativeRasterLoader getNativeRasterLoader() {
        return NativeRasterLoader__Holder.INSTANCE;
    }
    
    /**
     * Direct raster loading for VolatileImage.
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link NativeRasterLoader} service if any, or {@code null} otherwise
     */
    public static NativeRasterLoader getNativeRasterLoader(Extensions... extensions) {
        return getServiceWithFallback(NativeRasterLoader.class, NativeRasterLoader__Holder.FALLBACK, extensions);
    }

    private static class ProjectorUtils__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<ProjectorUtils> FALLBACK = null;
        private static final ProjectorUtils INSTANCE = getServiceWithFallback(ProjectorUtils.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link ProjectorUtils} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link ProjectorUtils}
     * and its dependencies (can fully implement given service).
     * @see #getProjectorUtils()
     */
    public static boolean isProjectorUtilsSupported() {
        return ProjectorUtils__Holder.INSTANCE != null;
    }
    
    /**
     * {@link java.awt.GraphicsEnvironment}-related utilities.
     *
     * @return full implementation of {@link ProjectorUtils} service if any, or {@code null} otherwise
     */
    public static ProjectorUtils getProjectorUtils() {
        return ProjectorUtils__Holder.INSTANCE;
    }
    
    /**
     * {@link java.awt.GraphicsEnvironment}-related utilities.
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link ProjectorUtils} service if any, or {@code null} otherwise
     */
    public static ProjectorUtils getProjectorUtils(Extensions... extensions) {
        return getServiceWithFallback(ProjectorUtils.class, ProjectorUtils__Holder.FALLBACK, extensions);
    }

    private static class RoundedCornersManager__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<RoundedCornersManager> FALLBACK = null;
        private static final RoundedCornersManager INSTANCE = getServiceWithFallback(RoundedCornersManager.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link RoundedCornersManager} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link RoundedCornersManager}
     * and its dependencies (can fully implement given service).
     * @see #getRoundedCornersManager()
     */
    public static boolean isRoundedCornersManagerSupported() {
        return RoundedCornersManager__Holder.INSTANCE != null;
    }
    
    /**
     * This manager allows decorate awt Window with rounded corners.
     * Appearance depends on operating system.
     *
     * @return full implementation of {@link RoundedCornersManager} service if any, or {@code null} otherwise
     */
    public static RoundedCornersManager getRoundedCornersManager() {
        return RoundedCornersManager__Holder.INSTANCE;
    }
    
    /**
     * This manager allows decorate awt Window with rounded corners.
     * Appearance depends on operating system.
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link RoundedCornersManager} service if any, or {@code null} otherwise
     */
    public static RoundedCornersManager getRoundedCornersManager(Extensions... extensions) {
        return getServiceWithFallback(RoundedCornersManager.class, RoundedCornersManager__Holder.FALLBACK, extensions);
    }

    private static class SharedTextures__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<SharedTextures> FALLBACK = null;
        private static final SharedTextures INSTANCE = getServiceWithFallback(SharedTextures.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link SharedTextures} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link SharedTextures}
     * and its dependencies (can fully implement given service).
     * @see #getSharedTextures()
     */
    public static boolean isSharedTexturesSupported() {
        return SharedTextures__Holder.INSTANCE != null;
    }
    
    /**
     * The service provides functionality for working with shared textures in JetBrainsRuntime.
     *
     * @return full implementation of {@link SharedTextures} service if any, or {@code null} otherwise
     */
    public static SharedTextures getSharedTextures() {
        return SharedTextures__Holder.INSTANCE;
    }
    
    /**
     * The service provides functionality for working with shared textures in JetBrainsRuntime.
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link SharedTextures} service if any, or {@code null} otherwise
     */
    public static SharedTextures getSharedTextures(Extensions... extensions) {
        return getServiceWithFallback(SharedTextures.class, SharedTextures__Holder.FALLBACK, extensions);
    }

    private static class SystemShortcuts__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<SystemShortcuts> FALLBACK = null;
        private static final SystemShortcuts INSTANCE = getServiceWithFallback(SystemShortcuts.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link SystemShortcuts} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link SystemShortcuts}
     * and its dependencies (can fully implement given service).
     * @see #getSystemShortcuts()
     */
    public static boolean isSystemShortcutsSupported() {
        return SystemShortcuts__Holder.INSTANCE != null;
    }
    
    /**
     * Querying system shortcuts
     *
     * @return full implementation of {@link SystemShortcuts} service if any, or {@code null} otherwise
     */
    public static SystemShortcuts getSystemShortcuts() {
        return SystemShortcuts__Holder.INSTANCE;
    }
    
    /**
     * Querying system shortcuts
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link SystemShortcuts} service if any, or {@code null} otherwise
     */
    public static SystemShortcuts getSystemShortcuts(Extensions... extensions) {
        return getServiceWithFallback(SystemShortcuts.class, SystemShortcuts__Holder.FALLBACK, extensions);
    }

    private static class SystemUtils__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<SystemUtils> FALLBACK = null;
        private static final SystemUtils INSTANCE = getServiceWithFallback(SystemUtils.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link SystemUtils} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link SystemUtils}
     * and its dependencies (can fully implement given service).
     * @see #getSystemUtils()
     */
    public static boolean isSystemUtilsSupported() {
        return SystemUtils__Holder.INSTANCE != null;
    }
    
    /**
     * Extends services provided by java.lang.System and similar.
     *
     * @return full implementation of {@link SystemUtils} service if any, or {@code null} otherwise
     */
    public static SystemUtils getSystemUtils() {
        return SystemUtils__Holder.INSTANCE;
    }
    
    /**
     * Extends services provided by java.lang.System and similar.
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link SystemUtils} service if any, or {@code null} otherwise
     */
    public static SystemUtils getSystemUtils(Extensions... extensions) {
        return getServiceWithFallback(SystemUtils.class, SystemUtils__Holder.FALLBACK, extensions);
    }

    private static class TextInput__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<TextInput> FALLBACK = null;
        private static final TextInput INSTANCE = getServiceWithFallback(TextInput.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link TextInput} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link TextInput}
     * and its dependencies (can fully implement given service).
     * @see #getTextInput()
     */
    public static boolean isTextInputSupported() {
        return TextInput__Holder.INSTANCE != null;
    }
    
    /**
     * This is a JBR API for text-input related functionality for applications that implement custom text components.
     * <p>
     * Suppose an application implements a custom text component called {@code CustomTextComponent}, that
     * doesn't inherit from {@link java.awt.TextComponent} or {@link javax.swing.text.JTextComponent}.
     * For this component to work correctly, the application needs to handle certain events that are missing from
     * the Java specification.
     * <p>
     * To do this, the application should add an event listener for the events provided by this API.
     * This is best done at application startup time, since the event listener is global, and not per-component.
     * For example, this would be a proper way to implement this event handler for {@code CustomTextComponent}:
     *
     * <pre>
     * {@code
     * var textInput = JBR.getTextInput();
     * if (textInput != null) {
     *     textInput.setGlobalEventListener(new TextInput.EventListener() {
     *         @Override
     *         public void handleSelectTextRangeEvent(TextInput.SelectTextRangeEvent event) {
     *             if (event.getSource() instanceof CustomTextComponent) {
     *                 ((CustomTextComponent)event.getSource()).select(event.getBegin(), event.getBegin() + event.getLength());
     *             }
     *         }
     *     });
     * }
     * }
     * </pre>
     * This assumes that {@code CustomTextComponent} has a method called {@code select}, that selects a text range,
     * similar to the {@link java.awt.TextComponent#select(int, int)} and {@link javax.swing.text.JTextComponent#select(int, int)}.
     * See {@link TextInput.SelectTextRangeEvent} for more information.
     *
     * @return full implementation of {@link TextInput} service if any, or {@code null} otherwise
     */
    public static TextInput getTextInput() {
        return TextInput__Holder.INSTANCE;
    }
    
    /**
     * This is a JBR API for text-input related functionality for applications that implement custom text components.
     * <p>
     * Suppose an application implements a custom text component called {@code CustomTextComponent}, that
     * doesn't inherit from {@link java.awt.TextComponent} or {@link javax.swing.text.JTextComponent}.
     * For this component to work correctly, the application needs to handle certain events that are missing from
     * the Java specification.
     * <p>
     * To do this, the application should add an event listener for the events provided by this API.
     * This is best done at application startup time, since the event listener is global, and not per-component.
     * For example, this would be a proper way to implement this event handler for {@code CustomTextComponent}:
     *
     * <pre>
     * {@code
     * var textInput = JBR.getTextInput();
     * if (textInput != null) {
     *     textInput.setGlobalEventListener(new TextInput.EventListener() {
     *         @Override
     *         public void handleSelectTextRangeEvent(TextInput.SelectTextRangeEvent event) {
     *             if (event.getSource() instanceof CustomTextComponent) {
     *                 ((CustomTextComponent)event.getSource()).select(event.getBegin(), event.getBegin() + event.getLength());
     *             }
     *         }
     *     });
     * }
     * }
     * </pre>
     * This assumes that {@code CustomTextComponent} has a method called {@code select}, that selects a text range,
     * similar to the {@link java.awt.TextComponent#select(int, int)} and {@link javax.swing.text.JTextComponent#select(int, int)}.
     * See {@link TextInput.SelectTextRangeEvent} for more information.
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link TextInput} service if any, or {@code null} otherwise
     */
    public static TextInput getTextInput(Extensions... extensions) {
        return getServiceWithFallback(TextInput.class, TextInput__Holder.FALLBACK, extensions);
    }

    private static class WindowDecorations__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<WindowDecorations> FALLBACK = null;
        private static final WindowDecorations INSTANCE = getServiceWithFallback(WindowDecorations.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link WindowDecorations} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link WindowDecorations}
     * and its dependencies (can fully implement given service).
     * @see #getWindowDecorations()
     */
    public static boolean isWindowDecorationsSupported() {
        return WindowDecorations__Holder.INSTANCE != null;
    }
    
    /**
     * Window decorations consist of title bar, window controls and border.
     * @see WindowDecorations.CustomTitleBar
     *
     * @return full implementation of {@link WindowDecorations} service if any, or {@code null} otherwise
     */
    public static WindowDecorations getWindowDecorations() {
        return WindowDecorations__Holder.INSTANCE;
    }
    
    /**
     * Window decorations consist of title bar, window controls and border.
     * @see WindowDecorations.CustomTitleBar
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link WindowDecorations} service if any, or {@code null} otherwise
     */
    public static WindowDecorations getWindowDecorations(Extensions... extensions) {
        return getServiceWithFallback(WindowDecorations.class, WindowDecorations__Holder.FALLBACK, extensions);
    }

    private static class WindowMove__Holder {
        @SuppressWarnings("auxiliaryclass")
        private static final FallbackSupplier<WindowMove> FALLBACK = null;
        private static final WindowMove INSTANCE = getServiceWithFallback(WindowMove.class, FALLBACK);
    }
    
    /**
     * Checks whether {@link WindowMove} service is supported by the runtime.
     * @return true if current runtime has implementation for all methods in {@link WindowMove}
     * and its dependencies (can fully implement given service).
     * @see #getWindowMove()
     */
    public static boolean isWindowMoveSupported() {
        return WindowMove__Holder.INSTANCE != null;
    }
    
    /**
     * X11 WM-assisted window moving facility.
     *
     * @return full implementation of {@link WindowMove} service if any, or {@code null} otherwise
     */
    public static WindowMove getWindowMove() {
        return WindowMove__Holder.INSTANCE;
    }
    
    /**
     * X11 WM-assisted window moving facility.
     *
     * @param extensions required extensions to enable
     * @return full implementation of {@link WindowMove} service if any, or {@code null} otherwise
     */
    public static WindowMove getWindowMove(Extensions... extensions) {
        return getServiceWithFallback(WindowMove.class, WindowMove__Holder.FALLBACK, extensions);
    }
}
