/*
 * Copyright 2000-2024 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.jetbrains;

import java.awt.event.KeyEvent;
import java.util.List;

/**
 * JBR API to inspect additional properties of AWT key events and keyboards.
 * */
@Service
@Provided
public interface Keyboard {
    /**
     * Inspect, which physical key was pressed on the keyboard, regardless of the active logical keyboard layout.
     * The list of virtual key codes is found in {@link KeyEvent}.
     * The values for the same scan code may depend on the physical keyboard layout type (ANSI, ISO or JIS)
     *
     * @param event                                 JBR-generated event to inspect.
     *                                              This must either be a {@link KeyEvent#KEY_PRESSED} or
     *                                              a {@link KeyEvent#KEY_RELEASED} event
     *
     * @return                                      The key code of the corresponding key on the standard QWERTY layout
     *
     * @throws UnsupportedOperationException        Inspecting this property is not supported on the current platform,
     *                                              or the event was not generated by JBR
     * @throws IllegalArgumentException             Passed KeyEvent is not of a valid type
     */
    int getKeyEventUSKeyCode(KeyEvent event);

    /**
     * Get the dead key code for the event.
     * If the key (without modifiers) is dead on the current layout (current ASCII-capable layout if
     * national keyboard layout support is active), then return the corresponding dead key code,
     * otherwise return {@link KeyEvent#VK_UNDEFINED}.
     * The dead key code will either be a {@code VK_DEAD_} keycode, or an extended key code for dead keys
     * that do not have the corresponding named key code.
     *
     * @param event                                 JBR-generated event to inspect.
     *                                              This must either be a {@link KeyEvent#KEY_PRESSED} or
     *                                              a {@link KeyEvent#KEY_RELEASED} event
     *
     * @return                                      The dead key code if the key is dead,
     *                                              or {@link KeyEvent#VK_UNDEFINED} otherwise
     *
     * @throws UnsupportedOperationException        Inspecting this property is not supported on the current platform,
     *                                              or the event was not generated by JBR
     * @throws IllegalArgumentException             Passed KeyEvent is not of a valid type
     */
    int getKeyEventDeadKeyCode(KeyEvent event);

    /**
     * Get the dead keystroke code for the event.
     * If the keystroke (with modifiers) is dead on the current (not necessarily ASCII-capable) keyboard layout,
     * return the corresponding dead key code for this keystroke,
     * otherwise return {@link KeyEvent#VK_UNDEFINED}.
     * The dead key code will either be a {@code VK_DEAD_} keycode, or an extended key code for dead keystrokes
     * that do not have the corresponding named key code.
     *
     * @param event                                 JBR-generated event to inspect.
     *                                              This must either be a {@link KeyEvent#KEY_PRESSED} or
     *                                              a {@link KeyEvent#KEY_RELEASED} event
     *
     * @return                                      The dead keystroke code if the key is dead,
     *                                              or {@link KeyEvent#VK_UNDEFINED} otherwise
     *
     * @throws UnsupportedOperationException        Inspecting this property is not supported on the current platform,
     *                                              or the event was not generated by JBR
     * @throws IllegalArgumentException             Passed KeyEvent is not of a valid type
     */
    int getKeyEventDeadKeyStroke(KeyEvent event);

    /**
     * Get the characters that this key event has produced.
     * For {@link KeyEvent#KEY_TYPED} it's equivalent to {@link KeyEvent#getKeyChar()}.
     * For {@link KeyEvent#KEY_RELEASED} it's always an empty string.
     *
     * @param event                                 JBR-generated event to inspect.
     *
     * @return                                      A string of characters that this event generated.
     *
     * @throws UnsupportedOperationException        Inspecting this property is not supported on the current platform,
     *                                              or the event was not generated by JBR
     */
    String getKeyEventCharacters(KeyEvent event);

    /**
     * Get platform-dependent logical keyboard layout identifier for the currently selected keyboard layout.
     *
     * @return                                      Keyboard layout identifier
     *
     * @throws UnsupportedOperationException        This method is not supported on the current platform
     */
    String getCurrentKeyboardLayout();

    /**
     * Get a list of logical keyboard layout identifiers for keyboard layouts enabled on the system.
     *
     * @return                                      List of keyboard layout identifiers
     *
     * @throws UnsupportedOperationException        This method is not supported on the current platform
     */
    List<String> getEnabledKeyboardLayouts();

    /**
     * Enables or disables reporting national key codes in {@link KeyEvent}s. When enabled, {@link KeyEvent#getKeyCode}
     * will return the key code, corresponding to the appropriate ASCII-capable/Latin keyboard.
     * When this option is disabled, the default Java behavior is used.
     * <p>
     * For example, with this enabled, pressing the key to the right of Tab (the Q key on the US layout) will result
     * in {@link KeyEvent#VK_A} being reported.
     * Pressing the same key on a non-Latin layout (like Russian) will result
     * in {@link KeyEvent#VK_Q}.
     * This way, the key codes reported through {@link KeyEvent#getKeyCode} would be the most
     * appropriate ones to detect keyboard shortcuts.
     *
     * @param value                                 `true` if this mode needs to be enabled, `false` otherwise
     *
     * @throws UnsupportedOperationException        This method is not supported on the current platform
     */
    void setReportNationalKeyCodes(boolean value);

    /**
     * Enables or disables reporting dead key codes as normal in {@link KeyEvent}s. When enabled, {@link KeyEvent#getKeyCode}
     * and {@link KeyEvent#getExtendedKeyCode()} will no longer report dead key codes, such as
     * {@link KeyEvent#VK_DEAD_GRAVE}.
     * Instead, they will report corresponding "normal" key codes, such as {@link KeyEvent#VK_BACK_QUOTE}.
     * When this option is disabled, the default Java behavior is used.
     *
     * @param value                                 `true` if this mode needs to be enabled, `false` otherwise
     *
     * @throws UnsupportedOperationException        This method is not supported on the current platform
     */
    void setConvertDeadKeyCodesToNormal(boolean value);
}