/*
 * Decompiled with CFR 0.152.
 */
package jetbrains.exodus.log;

import java.util.ArrayList;
import jetbrains.exodus.ExodusException;
import jetbrains.exodus.InvalidSettingException;
import jetbrains.exodus.crypto.EnvKryptKt;
import jetbrains.exodus.crypto.StreamCipherProvider;
import jetbrains.exodus.io.Block;
import jetbrains.exodus.io.DataWriter;
import jetbrains.exodus.log.BlockNotFoundException;
import jetbrains.exodus.log.BlockSet;
import jetbrains.exodus.log.Log;
import jetbrains.exodus.log.LogCache;
import jetbrains.exodus.log.LogTip;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class BufferedDataWriter {
    @NotNull
    private final Log log;
    @NotNull
    private final LogCache logCache;
    @NotNull
    private final DataWriter child;
    @NotNull
    private final LogTip initialPage;
    @Nullable
    private final StreamCipherProvider cipherProvider;
    private final byte[] cipherKey;
    private final long cipherBasicIV;
    private final int pageSize;
    @NotNull
    private final BlockSet.Mutable blockSetMutable;
    @NotNull
    private MutablePage currentPage;
    private long highAddress;
    private int count;

    BufferedDataWriter(@NotNull Log log, @NotNull DataWriter child, @NotNull LogTip page) {
        this.log = log;
        this.logCache = log.cache;
        this.blockSetMutable = page.blockSet.beginWrite();
        this.initialPage = page;
        this.child = child;
        this.highAddress = page.highAddress;
        boolean validInitialPage = page.count >= 0;
        this.pageSize = log.getCachePageSize();
        if (validInitialPage) {
            if (this.pageSize != page.bytes.length) {
                throw new InvalidSettingException("Configured page size doesn't match actual page size, pageSize = " + this.pageSize + ", actual page size = " + page.bytes.length);
            }
            this.currentPage = new MutablePage(null, page.bytes, page.pageAddress, page.count);
        } else {
            this.currentPage = new MutablePage(null, this.logCache.allocPage(), page.pageAddress, 0);
        }
        this.cipherProvider = log.getConfig().getCipherProvider();
        this.cipherKey = log.getConfig().getCipherKey();
        this.cipherBasicIV = log.getConfig().getCipherBasicIV();
    }

    @NotNull
    public BlockSet.Mutable getBlockSetMutable() {
        return this.blockSetMutable;
    }

    public void setHighAddress(long highAddress) {
        this.allocLastPage(highAddress - (long)((int)highAddress & this.log.getCachePageSize() - 1));
        this.highAddress = highAddress;
    }

    public MutablePage allocLastPage(long pageAddress) {
        MutablePage result = this.currentPage;
        if (pageAddress == result.pageAddress) {
            return result;
        }
        this.currentPage = result = new MutablePage(null, this.logCache.allocPage(), pageAddress, 0);
        return result;
    }

    void write(byte b) {
        int count = this.count;
        MutablePage currentPage = this.currentPage;
        int writtenCount = currentPage.writtenCount;
        if (writtenCount < this.pageSize) {
            currentPage.bytes[writtenCount] = b;
            currentPage.writtenCount = writtenCount + 1;
        } else {
            currentPage = this.allocNewPage();
            currentPage.bytes[0] = b;
            currentPage.writtenCount = 1;
        }
        this.count = count + 1;
    }

    void write(byte[] b, int len) throws ExodusException {
        int off = 0;
        int count = this.count + len;
        MutablePage currentPage = this.currentPage;
        while (len > 0) {
            int bytesToWrite = this.pageSize - currentPage.writtenCount;
            if (bytesToWrite == 0) {
                currentPage = this.allocNewPage();
                bytesToWrite = this.pageSize;
            }
            if (bytesToWrite > len) {
                bytesToWrite = len;
            }
            System.arraycopy(b, off, currentPage.bytes, currentPage.writtenCount, bytesToWrite);
            currentPage.writtenCount += bytesToWrite;
            len -= bytesToWrite;
            off += bytesToWrite;
        }
        this.count = count;
    }

    void commit() {
        this.count = 0;
        MutablePage currentPage = this.currentPage;
        currentPage.committedCount = currentPage.writtenCount;
        MutablePage previousPage = currentPage.previousPage;
        if (previousPage != null) {
            ArrayList<MutablePage> fullPages = new ArrayList<MutablePage>();
            do {
                fullPages.add(0, previousPage);
            } while ((previousPage = previousPage.previousPage) != null);
            for (MutablePage mutablePage : fullPages) {
                byte[] bytes = mutablePage.bytes;
                int off = mutablePage.flushedCount;
                int len = this.pageSize - off;
                long pageAddress = mutablePage.pageAddress;
                if (this.cipherProvider == null) {
                    this.writePage(bytes, off, len);
                } else {
                    this.writePage(EnvKryptKt.cryptBlocksImmutable(this.cipherProvider, this.cipherKey, this.cipherBasicIV, pageAddress, bytes, off, len, 1024), 0, len);
                }
                this.cachePage(bytes, pageAddress);
            }
            currentPage.previousPage = null;
        }
    }

    void flush() {
        if (this.count > 0) {
            throw new IllegalStateException("Can't flush uncommitted writer: " + this.count);
        }
        MutablePage currentPage = this.currentPage;
        int committedCount = currentPage.committedCount;
        int flushedCount = currentPage.flushedCount;
        if (committedCount > flushedCount) {
            byte[] bytes = currentPage.bytes;
            int len = committedCount - flushedCount;
            long pageAddress = currentPage.pageAddress;
            if (this.cipherProvider == null) {
                this.writePage(bytes, flushedCount, len);
            } else {
                this.writePage(EnvKryptKt.cryptBlocksImmutable(this.cipherProvider, this.cipherKey, this.cipherBasicIV, pageAddress, bytes, flushedCount, len, 1024), 0, len);
            }
            if (committedCount == this.pageSize) {
                this.cachePage(bytes, pageAddress);
            }
            currentPage.flushedCount = committedCount;
        }
    }

    Block openOrCreateBlock(long address, long length) {
        return this.child.openOrCreateBlock(address, length);
    }

    long getHighAddress() {
        return this.highAddress;
    }

    public void incHighAddress(long delta) {
        this.highAddress += delta;
    }

    public void setLastPageLength(int lastPageLength) {
        this.currentPage.setCounts(lastPageLength);
    }

    public int getLastPageLength() {
        return this.currentPage.writtenCount;
    }

    long getLastWrittenFileLength(long fileLengthBound) {
        return this.getHighAddress() % fileLengthBound;
    }

    @NotNull
    LogTip getStartingTip() {
        return this.initialPage;
    }

    @NotNull
    LogTip getUpdatedTip() {
        MutablePage currentPage = this.currentPage;
        BlockSet.Immutable blockSetImmutable = this.blockSetMutable.endWrite();
        return new LogTip(currentPage.bytes, currentPage.pageAddress, currentPage.committedCount, this.highAddress, this.highAddress, blockSetImmutable);
    }

    byte getByte(long address, byte max) {
        byte result;
        int offset = (int)address & this.pageSize - 1;
        long pageAddress = address - (long)offset;
        MutablePage page = this.getWrittenPage(pageAddress);
        if (page != null) {
            byte result2 = (byte)(page.bytes[offset] ^ 0x80);
            if (result2 < 0 || result2 > max) {
                throw new IllegalStateException("Unknown written page loggable type: " + result2);
            }
            return result2;
        }
        long fileAddress = this.log.getFileAddress(address);
        if (!this.blockSetMutable.contains(fileAddress)) {
            BlockNotFoundException.raise("Address is out of log space, underflow", this.log, address);
        }
        byte[] output = new byte[this.pageSize];
        Block block = this.blockSetMutable.getBlock(fileAddress);
        int readBytes = block.read(output, pageAddress - fileAddress, 0, output.length);
        if (readBytes < offset) {
            throw new ExodusException("Can't read expected page bytes");
        }
        if (this.cipherProvider != null) {
            EnvKryptKt.cryptBlocksMutable(this.cipherProvider, this.cipherKey, this.cipherBasicIV, pageAddress, output, 0, readBytes, 1024);
        }
        if ((result = (byte)(output[offset] ^ 0x80)) < 0 || result > max) {
            throw new IllegalStateException("Unknown written file loggable type: " + result + ", address: " + address);
        }
        return result;
    }

    private void writePage(@NotNull byte[] bytes, int off, int len) {
        Block block = this.child.write(bytes, off, len);
        this.blockSetMutable.add(block.getAddress(), block);
    }

    private void cachePage(@NotNull byte[] bytes, long pageAddress) {
        this.logCache.cachePage(this.log, pageAddress, bytes);
    }

    private MutablePage getWrittenPage(long alignedAddress) {
        MutablePage currentPage = this.currentPage;
        do {
            long highPageAddress;
            if (alignedAddress != (highPageAddress = currentPage.pageAddress)) continue;
            return currentPage;
        } while ((currentPage = currentPage.previousPage) != null);
        return null;
    }

    private MutablePage allocNewPage() {
        MutablePage currentPage = this.currentPage;
        this.currentPage = new MutablePage(currentPage, this.logCache.allocPage(), currentPage.pageAddress + (long)this.pageSize, 0);
        return this.currentPage;
    }

    public static class MutablePage {
        @Nullable
        MutablePage previousPage;
        @NotNull
        final byte[] bytes;
        final long pageAddress;
        int flushedCount;
        int committedCount;
        int writtenCount;

        MutablePage(@Nullable MutablePage previousPage, @NotNull byte[] page, long pageAddress, int count) {
            this.previousPage = previousPage;
            this.bytes = page;
            this.pageAddress = pageAddress;
            this.committedCount = this.writtenCount = count;
            this.flushedCount = this.writtenCount;
        }

        public byte[] getBytes() {
            return this.bytes;
        }

        public int getCount() {
            return this.writtenCount;
        }

        void setCounts(int count) {
            this.committedCount = this.writtenCount = count;
            this.flushedCount = this.writtenCount;
        }
    }
}

