/*
 * (C) Copyright 2016-2017, by Dimitrios Michail and Contributors.
 *
 * JGraphT : a free Java graph-theory library
 *
 * This program and the accompanying materials are dual-licensed under
 * either
 *
 * (a) the terms of the GNU Lesser General Public License version 2.1
 * as published by the Free Software Foundation, or (at your option) any
 * later version.
 *
 * or (per the licensee's choosing)
 *
 * (b) the terms of the Eclipse Public License v1.0 as published by
 * the Eclipse Foundation.
 */
package org.jgrapht.ext;

import java.io.*;
import java.util.*;

import org.jgrapht.*;

/**
 * Exports a graph into a CSV Format or any other Delimiter-separated value format.
 * 
 * <p>
 * The exporter supports three different formats which can be adjusted using the
 * {@link #setFormat(CSVFormat) setFormat} method. The supported formats are the same CSV formats
 * used by <a href="https://gephi.org/users/supported-graph-formats/csv-format">Gephi </a>. For some
 * of the formats, the behavior of the exporter can be adjusted using the
 * {@link #setParameter(org.jgrapht.ext.CSVFormat.Parameter, boolean) setParameter} method. See
 * {@link CSVFormat} for a description of the formats.
 * </p>
 * 
 * <p>
 * The default output respects <a href="http://www.ietf.org/rfc/rfc4180.txt">rfc4180</a>. The caller
 * can also adjust the separator to something like semicolon or pipe instead of comma. In such a
 * case, all fields are escaped using the new separator. See
 * <a href="https://en.wikipedia.org/wiki/Delimiter-separated_values">Delimiter- separated
 * values</a> for more information.
 * </p>
 * 
 * @see CSVFormat
 * 
 * @param <V> the graph vertex type
 * @param <E> the graph edge type
 * 
 * @author Dimitrios Michail
 * @since August 2016
 */
public class CSVExporter<V, E>
    implements GraphExporter<V, E>
{
    private static final char DEFAULT_DELIMITER = ',';

    private final ComponentNameProvider<V> vertexIDProvider;
    private final Set<CSVFormat.Parameter> parameters;
    private CSVFormat format;
    private char delimiter;

    /**
     * Creates a new CSVExporter with {@link CSVFormat#ADJACENCY_LIST} format and integer name
     * provider for the vertices.
     */
    public CSVExporter()
    {
        this(new IntegerComponentNameProvider<>(), CSVFormat.ADJACENCY_LIST, DEFAULT_DELIMITER);
    }

    /**
     * Creates a new CSVExporter with integer name providers for the vertices.
     * 
     * @param format the format to use
     */
    public CSVExporter(CSVFormat format)
    {
        this(new IntegerComponentNameProvider<>(), format, DEFAULT_DELIMITER);
    }

    /**
     * Creates a new CSVExporter with integer name providers for the vertices.
     * 
     * @param format the format to use
     * @param delimiter delimiter to use
     */
    public CSVExporter(CSVFormat format, char delimiter)
    {
        this(new IntegerComponentNameProvider<>(), format, delimiter);
    }

    /**
     * Constructs a new CSVExporter with the given ID providers and format.
     *
     * @param vertexIDProvider for generating vertex IDs. Must not be null.
     * @param format the format to use
     * @param delimiter delimiter to use
     */
    public CSVExporter(ComponentNameProvider<V> vertexIDProvider, CSVFormat format, char delimiter)
    {
        if (vertexIDProvider == null) {
            throw new IllegalArgumentException("Vertex id provider cannot be null");
        }
        this.vertexIDProvider = vertexIDProvider;
        this.format = format;
        if (!DSVUtils.isValidDelimiter(delimiter)) {
            throw new IllegalArgumentException("Character cannot be used as a delimiter");
        }
        this.delimiter = delimiter;
        this.parameters = new HashSet<>();
    }

    /**
     * Exports a graph
     *
     * @param g the graph
     * @param writer the writer
     */
    @Override
    public void exportGraph(Graph<V, E> g, Writer writer)
    {
        PrintWriter out = new PrintWriter(writer);
        switch (format) {
        case EDGE_LIST:
            exportAsEdgeList(g, out);
            break;
        case ADJACENCY_LIST:
            exportAsAdjacencyList(g, out);
            break;
        case MATRIX:
            exportAsMatrix(g, out);
            break;
        }
        out.flush();
    }

    /**
     * Return if a particular parameter of the exporter is enabled
     * 
     * @param p the parameter
     * @return {@code true} if the parameter is set, {@code false} otherwise
     */
    public boolean isParameter(CSVFormat.Parameter p)
    {
        return parameters.contains(p);
    }

    /**
     * Set the value of a parameter of the exporter
     * 
     * @param p the parameter
     * @param value the value to set
     */
    public void setParameter(CSVFormat.Parameter p, boolean value)
    {
        if (value) {
            parameters.add(p);
        } else {
            parameters.remove(p);
        }
    }

    /**
     * Get the format of the exporter
     * 
     * @return the format of the exporter
     */
    public CSVFormat getFormat()
    {
        return format;
    }

    /**
     * Set the format of the exporter
     * 
     * @param format the format to use
     */
    public void setFormat(CSVFormat format)
    {
        this.format = format;
    }

    /**
     * Get the delimiter (comma, semicolon, pipe, etc).
     * 
     * @return the delimiter
     */
    public char getDelimiter()
    {
        return delimiter;
    }

    /**
     * Set the delimiter (comma, semicolon, pipe, etc).
     * 
     * @param delimiter the delimiter to use
     */
    public void setDelimiter(char delimiter)
    {
        if (!DSVUtils.isValidDelimiter(delimiter)) {
            throw new IllegalArgumentException("Character cannot be used as a delimiter");
        }
        this.delimiter = delimiter;
    }

    private void exportAsEdgeList(Graph<V, E> g, PrintWriter out)
    {
        for (E e : g.edgeSet()) {
            exportEscapedField(out, vertexIDProvider.getName(g.getEdgeSource(e)));
            out.print(delimiter);
            exportEscapedField(out, vertexIDProvider.getName(g.getEdgeTarget(e)));
            out.println();
        }
    }

    private void exportAsAdjacencyList(Graph<V, E> g, PrintWriter out)
    {
        if (g instanceof DirectedGraph<?, ?>) {
            for (V v : g.vertexSet()) {
                exportEscapedField(out, vertexIDProvider.getName(v));
                for (E e : ((DirectedGraph<V, E>) g).outgoingEdgesOf(v)) {
                    V w = Graphs.getOppositeVertex(g, e, v);
                    out.print(delimiter);
                    exportEscapedField(out, vertexIDProvider.getName(w));
                }
                out.println();
            }
        } else {
            for (V v : g.vertexSet()) {
                exportEscapedField(out, vertexIDProvider.getName(v));
                for (E e : g.edgesOf(v)) {
                    V w = Graphs.getOppositeVertex(g, e, v);
                    out.print(delimiter);
                    exportEscapedField(out, vertexIDProvider.getName(w));
                }
                out.println();
            }
        }
    }

    private void exportAsMatrix(Graph<V, E> g, PrintWriter out)
    {
        boolean exportNodeId = parameters.contains(CSVFormat.Parameter.MATRIX_FORMAT_NODEID);
        boolean exportEdgeWeights =
            parameters.contains(CSVFormat.Parameter.MATRIX_FORMAT_EDGE_WEIGHTS);
        boolean zeroWhenNoEdge =
            parameters.contains(CSVFormat.Parameter.MATRIX_FORMAT_ZERO_WHEN_NO_EDGE);

        if (exportNodeId) {
            for (V v : g.vertexSet()) {
                out.print(delimiter);
                exportEscapedField(out, vertexIDProvider.getName(v));
            }
            out.println();
        }
        int n = g.vertexSet().size();
        for (V v : g.vertexSet()) {
            if (exportNodeId) {
                exportEscapedField(out, vertexIDProvider.getName(v));
                out.print(delimiter);
            }
            int i = 0;
            for (V u : g.vertexSet()) {
                E e = g.getEdge(v, u);
                if (e == null) {
                    if (zeroWhenNoEdge) {
                        exportEscapedField(out, "0");
                    }
                } else {
                    if (exportEdgeWeights) {
                        exportEscapedField(out, String.valueOf(g.getEdgeWeight(e)));
                    } else {
                        exportEscapedField(out, "1");
                    }
                }
                if (i++ < n - 1) {
                    out.print(delimiter);
                }
            }
            out.println();
        }
    }

    private void exportEscapedField(PrintWriter out, String field)
    {
        out.print(DSVUtils.escapeDSV(field, delimiter));
    }

}

// End CSVExporter.java
