/*
 * The MIT License
 *
 * Copyright (c) 2016-2019 Kai Burjack
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package org.joml.internal;


import org.joml.Matrix2f;
import org.joml.Matrix2d;
import org.joml.Matrix3d;
import org.joml.Matrix3f;
import org.joml.Matrix3x2d;
import org.joml.Matrix3x2f;
import org.joml.Matrix4d;
import org.joml.Matrix4f;
import org.joml.Matrix4x3d;
import org.joml.Matrix4x3f;
import org.joml.Quaternionf;
import org.joml.Vector2d;
import org.joml.Vector2f;
import org.joml.Vector2i;
import org.joml.Vector3d;
import org.joml.Vector3f;
import org.joml.Vector3i;
import org.joml.Vector4d;
import org.joml.Vector4f;
import org.joml.Vector4i;

/**
 * Helper class to do efficient memory operations on all JOML objects, NIO buffers and primitive arrays.
 * This class is used internally throughout JOML, is undocumented and is subject to change.
 * Use with extreme caution!
 * 
 * @author The LWJGL authors
 * @author Kai Burjack
 */
public abstract class MemUtil {
    public static final MemUtil INSTANCE = createInstance();
    private static MemUtil createInstance() {
        MemUtil accessor;
        accessor = new MemUtilNIO();
        return accessor;
    }


    public abstract void copy(Matrix4f src, Matrix4f dest);
    public abstract void copy(Matrix4x3f src, Matrix4x3f dest);
    public abstract void copy(Matrix4f src, Matrix4x3f dest);
    public abstract void copy(Matrix4x3f src, Matrix4f dest);
    public abstract void copy(Matrix3f src, Matrix3f dest);
    public abstract void copy(Matrix3f src, Matrix4f dest);
    public abstract void copy(Matrix4f src, Matrix3f dest);
    public abstract void copy(Matrix3f src, Matrix4x3f dest);
    public abstract void copy(Matrix3x2f src, Matrix3x2f dest);
    public abstract void copy(Matrix3x2d src, Matrix3x2d dest);
    public abstract void copy(Matrix2f src, Matrix2f dest);
    public abstract void copy(Matrix2d src, Matrix2d dest);
    public abstract void copy(Matrix2f src, Matrix3f dest);
    public abstract void copy(Matrix3f src, Matrix2f dest);
    public abstract void copy(Matrix2f src, Matrix3x2f dest);
    public abstract void copy(Matrix3x2f src, Matrix2f dest);
    public abstract void copy(Matrix2d src, Matrix3d dest);
    public abstract void copy(Matrix3d src, Matrix2d dest);
    public abstract void copy(Matrix2d src, Matrix3x2d dest);
    public abstract void copy(Matrix3x2d src, Matrix2d dest);
    public abstract void copy3x3(Matrix4f src, Matrix4f dest);
    public abstract void copy3x3(Matrix4x3f src, Matrix4x3f dest);
    public abstract void copy3x3(Matrix3f src, Matrix4x3f dest);
    public abstract void copy3x3(Matrix3f src, Matrix4f dest);
    public abstract void copy4x3(Matrix4f src, Matrix4f dest);
    public abstract void copy4x3(Matrix4x3f src, Matrix4f dest);
    public abstract void copy(Vector4f src, Vector4f dst);
    public abstract void copy(Vector4i src, Vector4i dst);
    public abstract void copy(Quaternionf src, Quaternionf dst);
    public abstract void copy(float[] arr, int off, Matrix4f dest);
    public abstract void copy(float[] arr, int off, Matrix3f dest);
    public abstract void copy(float[] arr, int off, Matrix4x3f dest);
    public abstract void copy(float[] arr, int off, Matrix3x2f dest);
    public abstract void copy(double[] arr, int off, Matrix3x2d dest);
    public abstract void copy(float[] arr, int off, Matrix2f dest);
    public abstract void copy(double[] arr, int off, Matrix2d dest);
    public abstract void copy(Matrix4f src, float[] dest, int off);
    public abstract void copy(Matrix3f src, float[] dest, int off);
    public abstract void copy(Matrix4x3f src, float[] dest, int off);
    public abstract void copy(Matrix3x2f src, float[] dest, int off);
    public abstract void copy(Matrix3x2d src, double[] dest, int off);
    public abstract void copy(Matrix2f src, float[] dest, int off);
    public abstract void copy(Matrix2d src, double[] dest, int off);
    public abstract void copy4x4(Matrix4x3f src, float[] dest, int off);
    public abstract void copy4x4(Matrix4x3d src, float[] dest, int off);
    public abstract void copy4x4(Matrix4x3d src, double[] dest, int off);
    public abstract void copy4x4(Matrix3x2f src, float[] dest, int off);
    public abstract void copy4x4(Matrix3x2d src, double[] dest, int off);
    public abstract void copy3x3(Matrix3x2f src, float[] dest, int off);
    public abstract void copy3x3(Matrix3x2d src, double[] dest, int off);
    public abstract void identity(Matrix4f dest);
    public abstract void identity(Matrix4x3f dest);
    public abstract void identity(Matrix3f dest);
    public abstract void identity(Matrix3x2f dest);
    public abstract void identity(Matrix3x2d dest);
    public abstract void identity(Matrix2f dest);
    public abstract void identity(Quaternionf dest);
    public abstract void swap(Matrix4f m1, Matrix4f m2);
    public abstract void swap(Matrix4x3f m1, Matrix4x3f m2);
    public abstract void swap(Matrix3f m1, Matrix3f m2);
    public abstract void swap(Matrix2f m1, Matrix2f m2);
    public abstract void swap(Matrix2d m1, Matrix2d m2);
    public abstract void zero(Matrix4f dest);
    public abstract void zero(Matrix4x3f dest);
    public abstract void zero(Matrix3f dest);
    public abstract void zero(Matrix3x2f dest);
    public abstract void zero(Matrix3x2d dest);
    public abstract void zero(Matrix2f dest);
    public abstract void zero(Matrix2d dest);
    public abstract void zero(Vector4f dest);
    public abstract void zero(Vector4i dest);


    public abstract void set(Matrix4f dest, Vector4f col0, Vector4f col1, Vector4f col2, Vector4f col3);
    public abstract void set(Matrix4x3f dest, Vector3f col0, Vector3f col1, Vector3f col2, Vector3f col3);
    public abstract void set(Matrix3f dest, Vector3f col0, Vector3f col1, Vector3f col2);
    public abstract void set(Matrix2f dest, Vector2f col0, Vector2f col1);

    public abstract void putColumn0(Matrix4f src, Vector4f dest);
    public abstract void putColumn1(Matrix4f src, Vector4f dest);
    public abstract void putColumn2(Matrix4f src, Vector4f dest);
    public abstract void putColumn3(Matrix4f src, Vector4f dest);

    public abstract void putColumn0(Matrix4f src, Vector3f dest);
    public abstract void putColumn1(Matrix4f src, Vector3f dest);
    public abstract void putColumn2(Matrix4f src, Vector3f dest);
    public abstract void putColumn3(Matrix4f src, Vector3f dest);

    public abstract void getColumn0(Matrix4f dest, Vector4f src);
    public abstract void getColumn1(Matrix4f dest, Vector4f src);
    public abstract void getColumn2(Matrix4f dest, Vector4f src);
    public abstract void getColumn3(Matrix4f dest, Vector4f src);

    public abstract void broadcast(float c, Vector4f dest);
    public abstract void broadcast(int c, Vector4i dest);

    public static class MemUtilNIO extends MemUtil {

        public void copy(Matrix4f src, Matrix4f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m03(src.m03());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m13(src.m13());
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
            dest._m23(src.m23());
            dest._m30(src.m30());
            dest._m31(src.m31());
            dest._m32(src.m32());
            dest._m33(src.m33());
        }

        public void copy(Matrix3f src, Matrix4f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m03(0.0f);
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m13(0.0f);
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
            dest._m23(0.0f);
            dest._m30(0.0f);
            dest._m31(0.0f);
            dest._m32(0.0f);
            dest._m33(1.0f);
        }

        public void copy(Matrix4f src, Matrix3f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
        }

        public void copy(Matrix3f src, Matrix4x3f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
            dest._m30(0.0f);
            dest._m31(0.0f);
            dest._m32(0.0f);
        }

        public void copy(Matrix3x2f src, Matrix3x2f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m20(src.m20());
            dest._m21(src.m21());
        }

        public void copy(Matrix3x2d src, Matrix3x2d dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m20(src.m20());
            dest._m21(src.m21());
        }

        public void copy(Matrix2f src, Matrix2f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m10(src.m10());
            dest._m11(src.m11());
        }

        public void copy(Matrix2d src, Matrix2d dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m10(src.m10());
            dest._m11(src.m11());
        }

        public void copy(Matrix2f src, Matrix3f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(0.0f);
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(0.0f);
            dest._m20(0.0f);
            dest._m21(0.0f);
            dest._m22(1.0f);
        }

        public void copy(Matrix3f src, Matrix2f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m10(src.m10());
            dest._m11(src.m11());
        }

        public void copy(Matrix2f src, Matrix3x2f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m20(0.0f);
            dest._m21(0.0f);
        }

        public void copy(Matrix3x2f src, Matrix2f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m10(src.m10());
            dest._m11(src.m11());
        }

        public void copy(Matrix2d src, Matrix3d dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(0.0);
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(0.0);
            dest._m20(0.0);
            dest._m21(0.0);
            dest._m22(1.0);
        }

        public void copy(Matrix3d src, Matrix2d dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m10(src.m10());
            dest._m11(src.m11());
        }

        public void copy(Matrix2d src, Matrix3x2d dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m20(0.0);
            dest._m21(0.0);
        }

        public void copy(Matrix3x2d src, Matrix2d dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m10(src.m10());
            dest._m11(src.m11());
        }

        public void copy3x3(Matrix4f src, Matrix4f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
        }

        public void copy3x3(Matrix4x3f src, Matrix4x3f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
        }

        public void copy3x3(Matrix3f src, Matrix4x3f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
        }

        public void copy3x3(Matrix3f src, Matrix4f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
        }

        public void copy4x3(Matrix4x3f src, Matrix4f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
            dest._m30(src.m30());
            dest._m31(src.m31());
            dest._m32(src.m32());
        }

        public void copy(Vector4f src, Vector4f dst) {
            dst.x = src.x;
            dst.y = src.y;
            dst.z = src.z;
            dst.w = src.w;
        }

        public void copy(Vector4i src, Vector4i dst) {
            dst.x = src.x;
            dst.y = src.y;
            dst.z = src.z;
            dst.w = src.w;
        }

        public void copy(Quaternionf src, Quaternionf dst) {
            dst.x = src.x;
            dst.y = src.y;
            dst.z = src.z;
            dst.w = src.w;
        }

        public void copy4x3(Matrix4f src, Matrix4f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
            dest._m30(src.m30());
            dest._m31(src.m31());
            dest._m32(src.m32());
        }

        public void copy(Matrix4f src, Matrix4x3f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
            dest._m30(src.m30());
            dest._m31(src.m31());
            dest._m32(src.m32());
        }

        public void copy(Matrix4x3f src, Matrix4f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m03(0.0f);
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m13(0.0f);
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
            dest._m23(0.0f);
            dest._m30(src.m30());
            dest._m31(src.m31());
            dest._m32(src.m32());
            dest._m33(1.0f);
        }

        public void copy(Matrix4x3f src, Matrix4x3f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
            dest._m30(src.m30());
            dest._m31(src.m31());
            dest._m32(src.m32());
        }

        public void copy(Matrix3f src, Matrix3f dest) {
            dest._m00(src.m00());
            dest._m01(src.m01());
            dest._m02(src.m02());
            dest._m10(src.m10());
            dest._m11(src.m11());
            dest._m12(src.m12());
            dest._m20(src.m20());
            dest._m21(src.m21());
            dest._m22(src.m22());
        }

        public void copy(float[] arr, int off, Matrix4f dest) {
            dest._m00(arr[off+0]);
            dest._m01(arr[off+1]);
            dest._m02(arr[off+2]);
            dest._m03(arr[off+3]);
            dest._m10(arr[off+4]);
            dest._m11(arr[off+5]);
            dest._m12(arr[off+6]);
            dest._m13(arr[off+7]);
            dest._m20(arr[off+8]);
            dest._m21(arr[off+9]);
            dest._m22(arr[off+10]);
            dest._m23(arr[off+11]);
            dest._m30(arr[off+12]);
            dest._m31(arr[off+13]);
            dest._m32(arr[off+14]);
            dest._m33(arr[off+15]);
        }

        public void copy(float[] arr, int off, Matrix3f dest) {
            dest._m00(arr[off+0]);
            dest._m01(arr[off+1]);
            dest._m02(arr[off+2]);
            dest._m10(arr[off+3]);
            dest._m11(arr[off+4]);
            dest._m12(arr[off+5]);
            dest._m20(arr[off+6]);
            dest._m21(arr[off+7]);
            dest._m22(arr[off+8]);
        }

        public void copy(float[] arr, int off, Matrix4x3f dest) {
            dest._m00(arr[off+0]);
            dest._m01(arr[off+1]);
            dest._m02(arr[off+2]);
            dest._m10(arr[off+3]);
            dest._m11(arr[off+4]);
            dest._m12(arr[off+5]);
            dest._m20(arr[off+6]);
            dest._m21(arr[off+7]);
            dest._m22(arr[off+8]);
            dest._m30(arr[off+9]);
            dest._m31(arr[off+10]);
            dest._m32(arr[off+11]);
        }

        public void copy(float[] arr, int off, Matrix3x2f dest) {
            dest._m00(arr[off+0]);
            dest._m01(arr[off+1]);
            dest._m10(arr[off+2]);
            dest._m11(arr[off+3]);
            dest._m20(arr[off+4]);
            dest._m21(arr[off+5]);
        }

        public void copy(double[] arr, int off, Matrix3x2d dest) {
            dest._m00(arr[off+0]);
            dest._m01(arr[off+1]);
            dest._m10(arr[off+2]);
            dest._m11(arr[off+3]);
            dest._m20(arr[off+4]);
            dest._m21(arr[off+5]);
        }

        public void copy(float[] arr, int off, Matrix2f dest) {
            dest._m00(arr[off+0]);
            dest._m01(arr[off+1]);
            dest._m10(arr[off+2]);
            dest._m11(arr[off+3]);
        }

        public void copy(double[] arr, int off, Matrix2d dest) {
            dest._m00(arr[off+0]);
            dest._m01(arr[off+1]);
            dest._m10(arr[off+2]);
            dest._m11(arr[off+3]);
        }

        public void copy(Matrix4f src, float[] dest, int off) {
            dest[off+0]  = src.m00();
            dest[off+1]  = src.m01();
            dest[off+2]  = src.m02();
            dest[off+3]  = src.m03();
            dest[off+4]  = src.m10();
            dest[off+5]  = src.m11();
            dest[off+6]  = src.m12();
            dest[off+7]  = src.m13();
            dest[off+8]  = src.m20();
            dest[off+9]  = src.m21();
            dest[off+10] = src.m22();
            dest[off+11] = src.m23();
            dest[off+12] = src.m30();
            dest[off+13] = src.m31();
            dest[off+14] = src.m32();
            dest[off+15] = src.m33();
        }

        public void copy(Matrix3f src, float[] dest, int off) {
            dest[off+0] = src.m00();
            dest[off+1] = src.m01();
            dest[off+2] = src.m02();
            dest[off+3] = src.m10();
            dest[off+4] = src.m11();
            dest[off+5] = src.m12();
            dest[off+6] = src.m20();
            dest[off+7] = src.m21();
            dest[off+8] = src.m22();
        }

        public void copy(Matrix4x3f src, float[] dest, int off) {
            dest[off+0]  = src.m00();
            dest[off+1]  = src.m01();
            dest[off+2]  = src.m02();
            dest[off+3]  = src.m10();
            dest[off+4]  = src.m11();
            dest[off+5]  = src.m12();
            dest[off+6]  = src.m20();
            dest[off+7]  = src.m21();
            dest[off+8]  = src.m22();
            dest[off+9]  = src.m30();
            dest[off+10] = src.m31();
            dest[off+11] = src.m32();
        }

        public void copy(Matrix3x2f src, float[] dest, int off) {
            dest[off+0] = src.m00();
            dest[off+1] = src.m01();
            dest[off+2] = src.m10();
            dest[off+3] = src.m11();
            dest[off+4] = src.m20();
            dest[off+5] = src.m21();
        }

        public void copy(Matrix3x2d src, double[] dest, int off) {
            dest[off+0] = src.m00();
            dest[off+1] = src.m01();
            dest[off+2] = src.m10();
            dest[off+3] = src.m11();
            dest[off+4] = src.m20();
            dest[off+5] = src.m21();
        }

        public void copy(Matrix2f src, float[] dest, int off) {
            dest[off+0] = src.m00();
            dest[off+1] = src.m01();
            dest[off+2] = src.m10();
            dest[off+3] = src.m11();
        }

        public void copy(Matrix2d src, double[] dest, int off) {
            dest[off+0] = src.m00();
            dest[off+1] = src.m01();
            dest[off+2] = src.m10();
            dest[off+3] = src.m11();
        }

        public void copy4x4(Matrix4x3f src, float[] dest, int off) {
            dest[off+0]  = src.m00();
            dest[off+1]  = src.m01();
            dest[off+2]  = src.m02();
            dest[off+3]  = 0.0f;
            dest[off+4]  = src.m10();
            dest[off+5]  = src.m11();
            dest[off+6]  = src.m12();
            dest[off+7]  = 0.0f;
            dest[off+8]  = src.m20();
            dest[off+9]  = src.m21();
            dest[off+10] = src.m22();
            dest[off+11] = 0.0f;
            dest[off+12] = src.m30();
            dest[off+13] = src.m31();
            dest[off+14] = src.m32();
            dest[off+15] = 1.0f;
        }

        public void copy4x4(Matrix4x3d src, float[] dest, int off) {
            dest[off+0]  = (float) src.m00();
            dest[off+1]  = (float) src.m01();
            dest[off+2]  = (float) src.m02();
            dest[off+3]  = 0.0f;
            dest[off+4]  = (float) src.m10();
            dest[off+5]  = (float) src.m11();
            dest[off+6]  = (float) src.m12();
            dest[off+7]  = 0.0f;
            dest[off+8]  = (float) src.m20();
            dest[off+9]  = (float) src.m21();
            dest[off+10] = (float) src.m22();
            dest[off+11] = 0.0f;
            dest[off+12] = (float) src.m30();
            dest[off+13] = (float) src.m31();
            dest[off+14] = (float) src.m32();
            dest[off+15] = 1.0f;
        }

        public void copy4x4(Matrix4x3d src, double[] dest, int off) {
            dest[off+0]  = src.m00();
            dest[off+1]  = src.m01();
            dest[off+2]  = src.m02();
            dest[off+3]  = 0.0;
            dest[off+4]  = src.m10();
            dest[off+5]  = src.m11();
            dest[off+6]  = src.m12();
            dest[off+7]  = 0.0;
            dest[off+8]  = src.m20();
            dest[off+9]  = src.m21();
            dest[off+10] = src.m22();
            dest[off+11] = 0.0;
            dest[off+12] = src.m30();
            dest[off+13] = src.m31();
            dest[off+14] = src.m32();
            dest[off+15] = 1.0;
        }

        public void copy3x3(Matrix3x2f src, float[] dest, int off) {
            dest[off+0] = src.m00();
            dest[off+1] = src.m01();
            dest[off+2] = 0.0f;
            dest[off+3] = src.m10();
            dest[off+4] = src.m11();
            dest[off+5] = 0.0f;
            dest[off+6] = src.m20();
            dest[off+7] = src.m21();
            dest[off+8] = 1.0f;
        }

        public void copy3x3(Matrix3x2d src, double[] dest, int off) {
            dest[off+0] = src.m00();
            dest[off+1] = src.m01();
            dest[off+2] = 0.0;
            dest[off+3] = src.m10();
            dest[off+4] = src.m11();
            dest[off+5] = 0.0;
            dest[off+6] = src.m20();
            dest[off+7] = src.m21();
            dest[off+8] = 1.0;
        }

        public void copy4x4(Matrix3x2f src, float[] dest, int off) {
            dest[off+0]  = src.m00();
            dest[off+1]  = src.m01();
            dest[off+2]  = 0.0f;
            dest[off+3]  = 0.0f;
            dest[off+4]  = src.m10();
            dest[off+5]  = src.m11();
            dest[off+6]  = 0.0f;
            dest[off+7]  = 0.0f;
            dest[off+8]  = 0.0f;
            dest[off+9]  = 0.0f;
            dest[off+10] = 1.0f;
            dest[off+11] = 0.0f;
            dest[off+12] = src.m20();
            dest[off+13] = src.m21();
            dest[off+14] = 0.0f;
            dest[off+15] = 1.0f;
        }

        public void copy4x4(Matrix3x2d src, double[] dest, int off) {
            dest[off+0]  = src.m00();
            dest[off+1]  = src.m01();
            dest[off+2]  = 0.0;
            dest[off+3]  = 0.0;
            dest[off+4]  = src.m10();
            dest[off+5]  = src.m11();
            dest[off+6]  = 0.0;
            dest[off+7]  = 0.0;
            dest[off+8]  = 0.0;
            dest[off+9]  = 0.0;
            dest[off+10] = 1.0;
            dest[off+11] = 0.0;
            dest[off+12] = src.m20();
            dest[off+13] = src.m21();
            dest[off+14] = 0.0;
            dest[off+15] = 1.0;
        }

        public void identity(Matrix4f dest) {
            dest._m00(1.0f);
            dest._m01(0.0f);
            dest._m02(0.0f);
            dest._m03(0.0f);
            dest._m10(0.0f);
            dest._m11(1.0f);
            dest._m12(0.0f);
            dest._m13(0.0f);
            dest._m20(0.0f);
            dest._m21(0.0f);
            dest._m22(1.0f);
            dest._m23(0.0f);
            dest._m30(0.0f);
            dest._m31(0.0f);
            dest._m32(0.0f);
            dest._m33(1.0f);
        }

        public void identity(Matrix4x3f dest) {
            dest._m00(1.0f);
            dest._m01(0.0f);
            dest._m02(0.0f);
            dest._m10(0.0f);
            dest._m11(1.0f);
            dest._m12(0.0f);
            dest._m20(0.0f);
            dest._m21(0.0f);
            dest._m22(1.0f);
            dest._m30(0.0f);
            dest._m31(0.0f);
            dest._m32(0.0f);
        }

        public void identity(Matrix3f dest) {
            dest._m00(1.0f);
            dest._m01(0.0f);
            dest._m02(0.0f);
            dest._m10(0.0f);
            dest._m11(1.0f);
            dest._m12(0.0f);
            dest._m20(0.0f);
            dest._m21(0.0f);
            dest._m22(1.0f);
        }

        public void identity(Matrix3x2f dest) {
            dest._m00(1.0f);
            dest._m01(0.0f);
            dest._m10(0.0f);
            dest._m11(1.0f);
            dest._m20(0.0f);
            dest._m21(0.0f);
        }

        public void identity(Matrix3x2d dest) {
            dest._m00(1.0);
            dest._m01(0.0);
            dest._m10(0.0);
            dest._m11(1.0);
            dest._m20(0.0);
            dest._m21(0.0);
        }

        public void identity(Matrix2f dest) {
            dest._m00(1.0f);
            dest._m01(0.0f);
            dest._m10(0.0f);
            dest._m11(1.0f);
        }

        public void identity(Quaternionf dest) {
            dest.x = 0.0f;
            dest.y = 0.0f;
            dest.z = 0.0f;
            dest.w = 1.0f;
        }

        public void swap(Matrix4f m1, Matrix4f m2) {
            float tmp;
            tmp = m1.m00(); m1._m00(m2.m00()); m2._m00(tmp);
            tmp = m1.m01(); m1._m01(m2.m01()); m2._m01(tmp);
            tmp = m1.m02(); m1._m02(m2.m02()); m2._m02(tmp);
            tmp = m1.m03(); m1._m03(m2.m03()); m2._m03(tmp);
            tmp = m1.m10(); m1._m10(m2.m10()); m2._m10(tmp);
            tmp = m1.m11(); m1._m11(m2.m11()); m2._m11(tmp);
            tmp = m1.m12(); m1._m12(m2.m12()); m2._m12(tmp);
            tmp = m1.m13(); m1._m13(m2.m13()); m2._m13(tmp);
            tmp = m1.m20(); m1._m20(m2.m20()); m2._m20(tmp);
            tmp = m1.m21(); m1._m21(m2.m21()); m2._m21(tmp);
            tmp = m1.m22(); m1._m22(m2.m22()); m2._m22(tmp);
            tmp = m1.m23(); m1._m23(m2.m23()); m2._m23(tmp);
            tmp = m1.m30(); m1._m30(m2.m30()); m2._m30(tmp);
            tmp = m1.m31(); m1._m31(m2.m31()); m2._m31(tmp);
            tmp = m1.m32(); m1._m32(m2.m32()); m2._m32(tmp);
            tmp = m1.m33(); m1._m33(m2.m33()); m2._m33(tmp);
        }

        public void swap(Matrix4x3f m1, Matrix4x3f m2) {
            float tmp;
            tmp = m1.m00(); m1._m00(m2.m00()); m2._m00(tmp);
            tmp = m1.m01(); m1._m01(m2.m01()); m2._m01(tmp);
            tmp = m1.m02(); m1._m02(m2.m02()); m2._m02(tmp);
            tmp = m1.m10(); m1._m10(m2.m10()); m2._m10(tmp);
            tmp = m1.m11(); m1._m11(m2.m11()); m2._m11(tmp);
            tmp = m1.m12(); m1._m12(m2.m12()); m2._m12(tmp);
            tmp = m1.m20(); m1._m20(m2.m20()); m2._m20(tmp);
            tmp = m1.m21(); m1._m21(m2.m21()); m2._m21(tmp);
            tmp = m1.m22(); m1._m22(m2.m22()); m2._m22(tmp);
            tmp = m1.m30(); m1._m30(m2.m30()); m2._m30(tmp);
            tmp = m1.m31(); m1._m31(m2.m31()); m2._m31(tmp);
            tmp = m1.m32(); m1._m32(m2.m32()); m2._m32(tmp);
        }
        
        public void swap(Matrix3f m1, Matrix3f m2) {
            float tmp;
            tmp = m1.m00(); m1._m00(m2.m00()); m2._m00(tmp);
            tmp = m1.m01(); m1._m01(m2.m01()); m2._m01(tmp);
            tmp = m1.m02(); m1._m02(m2.m02()); m2._m02(tmp);
            tmp = m1.m10(); m1._m10(m2.m10()); m2._m10(tmp);
            tmp = m1.m11(); m1._m11(m2.m11()); m2._m11(tmp);
            tmp = m1.m12(); m1._m12(m2.m12()); m2._m12(tmp);
            tmp = m1.m20(); m1._m20(m2.m20()); m2._m20(tmp);
            tmp = m1.m21(); m1._m21(m2.m21()); m2._m21(tmp);
            tmp = m1.m22(); m1._m22(m2.m22()); m2._m22(tmp);
        }

        public void swap(Matrix2f m1, Matrix2f m2) {
            float tmp;
            tmp = m1.m00(); m1._m00(m2.m00()); m2._m00(tmp);
            tmp = m1.m01(); m1._m00(m2.m01()); m2._m01(tmp);
            tmp = m1.m10(); m1._m00(m2.m10()); m2._m10(tmp);
            tmp = m1.m11(); m1._m00(m2.m11()); m2._m11(tmp);
        }

        public void swap(Matrix2d m1, Matrix2d m2) {
            double tmp;
            tmp = m1.m00(); m1._m00(m2.m00()); m2._m00(tmp);
            tmp = m1.m01(); m1._m00(m2.m01()); m2._m01(tmp);
            tmp = m1.m10(); m1._m00(m2.m10()); m2._m10(tmp);
            tmp = m1.m11(); m1._m00(m2.m11()); m2._m11(tmp);
        }

        public void zero(Matrix4f dest) {
            dest._m00(0.0f);
            dest._m01(0.0f);
            dest._m02(0.0f);
            dest._m03(0.0f);
            dest._m10(0.0f);
            dest._m11(0.0f);
            dest._m12(0.0f);
            dest._m13(0.0f);
            dest._m20(0.0f);
            dest._m21(0.0f);
            dest._m22(0.0f);
            dest._m23(0.0f);
            dest._m30(0.0f);
            dest._m31(0.0f);
            dest._m32(0.0f);
            dest._m33(0.0f);
        }

        public void zero(Matrix4x3f dest) {
            dest._m00(0.0f);
            dest._m01(0.0f);
            dest._m02(0.0f);
            dest._m10(0.0f);
            dest._m11(0.0f);
            dest._m12(0.0f);
            dest._m20(0.0f);
            dest._m21(0.0f);
            dest._m22(0.0f);
            dest._m30(0.0f);
            dest._m31(0.0f);
            dest._m32(0.0f);
        }

        public void zero(Matrix3f dest) {
            dest._m00(0.0f);
            dest._m01(0.0f);
            dest._m02(0.0f);
            dest._m10(0.0f);
            dest._m11(0.0f);
            dest._m12(0.0f);
            dest._m20(0.0f);
            dest._m21(0.0f);
            dest._m22(0.0f);
        }

        public void zero(Matrix3x2f dest) {
            dest._m00(0.0f);
            dest._m01(0.0f);
            dest._m10(0.0f);
            dest._m11(0.0f);
            dest._m20(0.0f);
            dest._m21(0.0f);
        }

        public void zero(Matrix3x2d dest) {
            dest._m00(0.0);
            dest._m01(0.0);
            dest._m10(0.0);
            dest._m11(0.0);
            dest._m20(0.0);
            dest._m21(0.0);
        }

        public void zero(Matrix2f dest) {
            dest._m00(0.0f);
            dest._m01(0.0f);
            dest._m10(0.0f);
            dest._m11(0.0f);
        }

        public void zero(Matrix2d dest) {
            dest._m00(0.0);
            dest._m01(0.0);
            dest._m10(0.0);
            dest._m11(0.0);
        }

        public void zero(Vector4f dest) {
            dest.x = 0.0f;
            dest.y = 0.0f;
            dest.z = 0.0f;
            dest.w = 0.0f;
        }

        public void zero(Vector4i dest) {
            dest.x = 0;
            dest.y = 0;
            dest.z = 0;
            dest.w = 0;
        }


        public void set(Matrix4f m, Vector4f col0, Vector4f col1, Vector4f col2, Vector4f col3) {
            m._m00(col0.x);
            m._m01(col0.y);
            m._m02(col0.z);
            m._m03(col0.w);
            m._m10(col1.x);
            m._m11(col1.y);
            m._m12(col1.z);
            m._m13(col1.w);
            m._m20(col2.x);
            m._m21(col2.y);
            m._m22(col2.z);
            m._m23(col2.w);
            m._m30(col3.x);
            m._m31(col3.y);
            m._m32(col3.z);
            m._m33(col3.w);
        }

        public void set(Matrix4x3f m, Vector3f col0, Vector3f col1, Vector3f col2, Vector3f col3) {
            m._m00(col0.x);
            m._m01(col0.y);
            m._m02(col0.z);
            m._m10(col1.x);
            m._m11(col1.y);
            m._m12(col1.z);
            m._m20(col2.x);
            m._m21(col2.y);
            m._m22(col2.z);
            m._m30(col3.x);
            m._m31(col3.y);
            m._m32(col3.z);
        }

        public void set(Matrix3f m, Vector3f col0, Vector3f col1, Vector3f col2) {
            m._m00(col0.x);
            m._m01(col0.y);
            m._m02(col0.z);
            m._m10(col1.x);
            m._m11(col1.y);
            m._m12(col1.z);
            m._m20(col2.x);
            m._m21(col2.y);
            m._m22(col2.z);
        }

        public void set(Matrix2f m, Vector2f col0, Vector2f col1) {
            m._m00(col0.x);
            m._m01(col0.y);
            m._m10(col1.x);
            m._m11(col1.y);
        }

        public void putColumn0(Matrix4f src, Vector4f dest) {
            dest.x = src.m00();
            dest.y = src.m01();
            dest.z = src.m02();
            dest.w = src.m03();
        }

        public void putColumn1(Matrix4f src, Vector4f dest) {
            dest.x = src.m10();
            dest.y = src.m11();
            dest.z = src.m12();
            dest.w = src.m13();
        }

        public void putColumn2(Matrix4f src, Vector4f dest) {
            dest.x = src.m20();
            dest.y = src.m21();
            dest.z = src.m22();
            dest.w = src.m23();
        }

        public void putColumn3(Matrix4f src, Vector4f dest) {
            dest.x = src.m30();
            dest.y = src.m31();
            dest.z = src.m32();
            dest.w = src.m33();
        }

        public void putColumn0(Matrix4f src, Vector3f dest) {
            dest.x = src.m00();
            dest.y = src.m01();
            dest.z = src.m02();
        }

        public void putColumn1(Matrix4f src, Vector3f dest) {
            dest.x = src.m10();
            dest.y = src.m11();
            dest.z = src.m12();
        }

        public void putColumn2(Matrix4f src, Vector3f dest) {
            dest.x = src.m20();
            dest.y = src.m21();
            dest.z = src.m22();
        }

        public void putColumn3(Matrix4f src, Vector3f dest) {
            dest.x = src.m30();
            dest.y = src.m31();
            dest.z = src.m32();
        }

        public void getColumn0(Matrix4f dest, Vector4f src) {
            dest._m00(src.x);
            dest._m01(src.y);
            dest._m02(src.z);
            dest._m03(src.w);
        }

        public void getColumn1(Matrix4f dest, Vector4f src) {
            dest._m10(src.x);
            dest._m11(src.y);
            dest._m12(src.z);
            dest._m13(src.w);
        }

        public void getColumn2(Matrix4f dest, Vector4f src) {
            dest._m20(src.x);
            dest._m21(src.y);
            dest._m22(src.z);
            dest._m23(src.w);
        }

        public void getColumn3(Matrix4f dest, Vector4f src) {
            dest._m30(src.x);
            dest._m31(src.y);
            dest._m32(src.z);
            dest._m33(src.w);
        }

        public void broadcast(float c, Vector4f dest) {
            dest.x = c;
            dest.y = c;
            dest.z = c;
            dest.w = c;
        }

        public void broadcast(int c, Vector4i dest) {
            dest.x = c;
            dest.y = c;
            dest.z = c;
            dest.w = c;
        }
    }

}
