/**
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2014 Edgar Espina
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.jooby.hbm;

import com.google.inject.Binder;
import com.google.inject.Key;
import com.google.inject.name.Names;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.boot.Metadata;
import org.hibernate.boot.MetadataSources;
import org.hibernate.boot.SessionFactoryBuilder;
import org.hibernate.boot.model.naming.ImplicitNamingStrategyJpaCompliantImpl;
import org.hibernate.boot.registry.BootstrapServiceRegistry;
import org.hibernate.boot.registry.BootstrapServiceRegistryBuilder;
import org.hibernate.boot.registry.StandardServiceRegistry;
import org.hibernate.boot.registry.StandardServiceRegistryBuilder;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.event.service.spi.EventListenerRegistry;
import org.hibernate.event.spi.EventType;
import org.hibernate.service.spi.ServiceRegistryImplementor;
import org.jooby.Env;
import org.jooby.Env.ServiceKey;
import org.jooby.Jooby;
import org.jooby.Registry;
import org.jooby.Route;
import org.jooby.internal.hbm.GuiceBeanManager;
import org.jooby.internal.hbm.OpenSessionInView;
import org.jooby.internal.hbm.ScanEnvImpl;
import org.jooby.internal.hbm.SessionProvider;
import org.jooby.internal.hbm.UnitOfWorkProvider;
import org.jooby.jdbc.Jdbc;

import javax.inject.Provider;
import javax.inject.Singleton;
import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.sql.DataSource;
import java.net.URL;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * <h1>hibernate</h1>
 * <p>
 * <a href="http://hibernate.org/orm">Hibernate ORM</a> enables developers to more easily write
 * applications whose data outlives the application process. As an Object/Relational Mapping (ORM)
 * framework, Hibernate is concerned with data persistence as it applies to relational databases.
 * </p>
 * <p>
 * This module setup and configure <a href="http://hibernate.org/orm">Hibernate ORM</a> and
 * <code>JPA Provider</code>.
 * </p>
 *
 * <p>
 * This module depends on {@link Jdbc} module, make sure you read the doc of the {@link Jdbc}
 * module.
 * </p>
 *
 * <h2>exports</h2>
 * <ul>
 * <li>SessionFactory / EntityManagerFactory</li>
 * <li>Session / EntityManager</li>
 * <li>UnitOfWork</li>
 * </ul>
 *
 * <h2>usage</h2>
 *
 * <pre>{@code
 * {
 *   use(new Hbm("jdbc:mysql://localhost/mydb")
 *       .classes(Beer.class)
 *   );
 *
 *   get("/api/beer/", req -> {
 *     return require(UnitOfWork.class).apply(em -> {
 *       return em.createQuery("from Beer").getResultList();
 *     });
 *   });
 * }
 * }</pre>
 *
 * <h2>unit of work</h2>
 * <p>
 * We provide an {@link UnitOfWork} to simplify the amount of code required to interact within the
 * database.
 * </p>
 * <p>
 * For example the next line:
 * </p>
 *
 * <pre>{@code
 * {
 *   require(UnitOfWork.class).apply(em -> {
 *     return em.createQuery("from Beer").getResultList();
 *   });
 * }
 * }</pre>
 *
 * <p>
 * Is the same as:
 * </p>
 *
 * <pre>{@code
 * {
 *    Session session = require(SessionFactory.class).openSession();
 *    Transaction trx = session.getTransaction();
 *    try {
 *      trx.begin();
 *      List<Beer> beers = em.createQuery("from Beer").getResultList();
 *      trx.commit();
 *    } catch (Exception ex) {
 *      trx.rollback();
 *    } finally {
 *      session.close();
 *    }
 * }
 * }</pre>
 *
 * <p>
 * An {@link UnitOfWork} takes care of transactions and session life-cycle. It's worth to mention
 * too that a first requested {@link UnitOfWork} bind the Session to the current request. If later
 * in the execution flow an {@link UnitOfWork}, {@link Session} and/or {@link EntityManager} is
 * required then the one that belong to the current request (first requested) will be provided it.
 * </p>
 *
 * <h2>open session in view</h2>
 * <p>
 * We provide an advanced and recommended <a href=
 * "https://developer.jboss.org/wiki/OpenSessionInView#jive_content_id_Can_I_use_two_transactions_in_one_Session"
 * >Open Session in View</a> pattern, which basically keep the {@link Session} opened until the view
 * is rendered, but it uses two database transactions:
 * </p>
 *
 * <ol>
 * <li>first transaction is committed before rendering the view and then</li>
 * <li>a read only transaction is opened for rendering the view</li>
 * </ol>
 *
 * <p>
 * Here is an example on how to setup the open session in view filter:
 * </p>
 *
 * <pre>{@code
 * {
 *    use(new Hbm());
 *
 *    use("*", Hbm.openSessionInView());
 * }
 * }</pre>
 *
 * <h2>event listeners</h2>
 * <p>
 * JPA event listeners are provided by Guice, which means you can inject dependencies into your
 * event
 * listeners:
 * </p>
 *
 * <pre>{@code
 *
 * &#64;Entity
 * &#64;EntityListeners({BeerListener.class})
 * public class Beer {
 *
 * }
 *
 * public class BeerListener {
 *   &#64;Inject
 *   public BeerListener(DependencyA depA) {
 *     this.depA = depA;
 *   }
 *
 *   &#64;PostLoad
 *   public void postLoad(Beer beer) {
 *     this.depA.complementBeer(beer);
 *   }
 * }
 *
 * }</pre>
 *
 * <p>
 * Hibernate event listeners are supported too via {@link #onEvent(EventType, Class)}:
 * </p>
 *
 * <pre>{@code
 * {
 *    use(new Hbm()
 *        .onEvent(EventType.POST_LOAD, MyPostLoadListener.class));
 * }
 * }</pre>
 *
 * <p>
 * Again, <code>MyPostLoadListener</code> will be provided by Guice.
 * </p>
 *
 * <h2>persistent classes</h2>
 * <p>
 * Persistent classes must be provided at application startup time via
 * {@link #classes(Class...)}:
 * </p>
 *
 * <pre>{@code
 * {
 *   use(new Hbm()
 *       .classes(Entity1.class, Entity2.class, ..., )
 *   );
 * }
 * }</pre>
 *
 * <p>
 * Or via {@link #scan()}:
 * </p>
 *
 * <pre>{@code
 * {
 *   use(new Hbm()
 *       .scan()
 *   );
 * }
 * }</pre>
 *
 * <p>
 * Which <code>scan</code> the application package defined by <code>hibernate.packagesToScan</code>,
 * or you can provide where to look:
 * <p>
 *
 * <pre>{@code
 * {
 *   use(new Hbm()
 *       .scan("foo.bar", "x.y.z")
 *   );
 * }
 * }</pre>
 *
 * <h2>advanced configuration</h2>
 * <p>
 * Advanced configuration is provided via <code>doWithXXX</code> callbacks:
 * </p>
 * <pre>{@code
 * {
 *   use(new Hbm()
 *       .doWithBootstrap(bsrb -> {
 *         // do with bsrb
 *       })
 *       .doWithRegistry(ssrb -> {
 *         // do with ssrb
 *       })
 *   );
 * }
 * }</pre>
 *
 * <p>
 * Or via <code>hibernate.*</code> properties from your <code>.conf</code> file:
 * </p>
 *
 * <pre>{@code
 *   hibernate.hbm2ddl.auto = update
 * }</pre>
 *
 *
 * <h2>life-cycle</h2>
 * <p>
 * You are free to inject a {@link SessionFactory} or {@link EntityManagerFactory} create a new
 * {@link EntityManagerFactory#createEntityManager()}, start transactions and do everything you
 * need.
 * </p>
 *
 * <p>
 * For the time being, this doesn't work for a {@link Session} or {@link EntityManager}. A
 * {@link Session} {@link EntityManager} is bound to the current request, which means you can't
 * freely access from every single thread (like manually started thread, started by an executor
 * service, quartz, etc...).
 * </p>
 *
 * <p>
 * Another restriction, is the access from {@link Singleton} services. If you need access from a
 * singleton services, you need to inject a {@link Provider}.
 * </p>
 *
 * <pre>
 *
 * &#64;Singleton
 * public class MySingleton {
 *
 *   &#64;Inject
 *   public MySingleton(Provider&lt;EntityManager&gt; em) {
 *     this.em = em;
 *   }
 * }
 * </pre>
 *
 * <p>
 * Still, we strongly recommend to leave your services in the default scope and avoid to use
 * {@link Singleton} objects, except of course for really expensive resources. This is also
 * recommend it by Guice.
 * </p>
 *
 * <p>
 * Services in the default scope won't have this problem and are free to inject the
 * {@link Session} or {@link EntityManager} directly.
 * </p>
 *
 * @author edgar
 * @since 1.0.0.CR7
 */
public class Hbm implements Jooby.Module {

  private static final BiConsumer NOOP = (r, c) -> {
  };

  private final List<BiConsumer<SessionFactoryImplementor, Registry>> listeners = new ArrayList<>();

  private final List<Consumer<Binder>> bindings = new ArrayList<>();

  private final List<BiConsumer<MetadataSources, Config>> sources = new ArrayList<>();

  private final String name;

  private BiConsumer<BootstrapServiceRegistryBuilder, Config> bsrb = NOOP;

  private BiConsumer<StandardServiceRegistryBuilder, Config> ssrb = NOOP;

  private BiConsumer<MetadataSources, Config> metaSources = NOOP;

  private BiConsumer<SessionFactoryBuilder, Config> sfb = NOOP;

  private BiConsumer<SessionFactory, Config> sf = NOOP;

  /**
   * Creates a new {@link Hbm} module.
   *
   * @param db A jdbc connection string or a property with a jdbc connection string.
   */
  public Hbm(final String db) {
    this.name = db;
  }

  /**
   * Creates a new {@link Hbm} module. A <code>db</code> property must be present in your
   * <code>.conf</code> file.
   */
  public Hbm() {
    this("db");
  }

  /**
   * Append persistent classes (classess annotated with Entity).
   *
   * @param classes Persistent classes.
   * @return This module.
   */
  @SuppressWarnings("rawtypes")
  public Hbm classes(final Class... classes) {
    sources.add((m, c) -> Arrays.stream(classes).forEach(m::addAnnotatedClass));
    return this;
  }

  /**
   * Scan the provided packages and discover persistent classes (annotated with Entity).
   *
   * @param packages Package to scan.
   * @return This module.
   */
  public Hbm scan(final String... packages) {
    sources.add((m, c) -> Arrays.stream(packages).forEach(m::addPackage));
    return this;
  }

  /**
   * Scan the packages defined by <code>hibernate.packagesToScan</code> property or the application
   * package (that's the package where you application was defined) and discover persistent classes
   * (annotated with Entity).
   *
   * @return This module.
   */
  public Hbm scan() {
    sources.add((m, conf) ->
        Stream.of(conf.getAnyRef("hibernate.packagesToScan"))
            .flatMap(it -> {
              if (it instanceof List) {
                return ((List) it).stream();
              }
              return Stream.of(it);
            })
            .forEach(it -> m.addPackage(it.toString()))
    );
    return this;
  }

  /**
   * Creates an open session in view filter as described <a href=
   * "https://developer.jboss.org/wiki/OpenSessionInView#jive_content_id_Can_I_use_two_transactions_in_one_Session">here</a>.
   *
   * Please note a call to this method give you only the filter, you must add it to your application
   * like:
   *
   * <pre>{@code
   *  {
   *     use(new Hbm());
   *
   *     use("*", Hbm.openSessionInView());
   *  }
   * }</pre>
   *
   * @return This module.
   */
  public static Route.Filter openSessionInView() {
    return new OpenSessionInView();
  }

  /**
   * Register an hibernate event listener. Listener will be created and injected by Guice.
   *
   * @param type Event type.
   * @param listenerType Listener type.
   * @return This module.
   */
  @SuppressWarnings("unchecked")
  public <T> Hbm onEvent(final EventType<T> type, final Class<? extends T> listenerType) {
    bindings.add(b -> b.bind(listenerType).asEagerSingleton());

    listeners.add((s, r) -> {
      ServiceRegistryImplementor serviceRegistry = s.getServiceRegistry();
      EventListenerRegistry service = serviceRegistry.getService(EventListenerRegistry.class);
      T listener = r.require(listenerType);
      service.appendListeners(type, listener);
    });
    return this;
  }

  /**
   * Configurer callback to apply advanced configuration while bootstrapping hibernate:
   *
   * @param configurer Configurer callback.
   * @return This module
   */
  public <T> Hbm doWithBootstrap(
      final BiConsumer<BootstrapServiceRegistryBuilder, Config> configurer) {
    this.bsrb = configurer;
    return this;
  }

  /**
   * Configurer callback to apply advanced configuration while bootstrapping hibernate:
   *
   * @param configurer Configurer callback.
   * @return This module
   */
  public <T> Hbm doWithBootstrap(final Consumer<BootstrapServiceRegistryBuilder> configurer) {
    return doWithBootstrap((builder, conf) -> configurer.accept(builder));
  }

  /**
   * Configurer callback to apply advanced configuration while bootstrapping hibernate:
   *
   * @param configurer Configurer callback.
   * @return This module
   */
  public <T> Hbm doWithRegistry(final Consumer<StandardServiceRegistryBuilder> configurer) {
    return doWithRegistry((builder, conf) -> configurer.accept(builder));
  }

  /**
   * Configurer callback to apply advanced configuration while bootstrapping hibernate:
   *
   * @param configurer Configurer callback.
   * @return This module
   */
  public <T> Hbm doWithRegistry(
      final BiConsumer<StandardServiceRegistryBuilder, Config> configurer) {
    this.ssrb = configurer;
    return this;
  }

  /**
   * Configurer callback to apply advanced configuration while bootstrapping hibernate:
   *
   * @param configurer Configurer callback.
   * @return This module
   */
  public <T> Hbm doWithSources(final Consumer<MetadataSources> configurer) {
    return doWithSources((builder, conf) -> configurer.accept(builder));
  }

  /**
   * Configurer callback to apply advanced configuration while bootstrapping hibernate:
   *
   * @param configurer Configurer callback.
   * @return This module
   */
  public <T> Hbm doWithSources(final BiConsumer<MetadataSources, Config> configurer) {
    this.metaSources = configurer;
    return this;
  }

  /**
   * Configurer callback to apply advanced configuration while bootstrapping hibernate:
   *
   * @param configurer Configurer callback.
   * @return This module
   */
  public <T> Hbm doWithSessionFactoryBuilder(final Consumer<SessionFactoryBuilder> configurer) {
    return doWithSessionFactoryBuilder((builder, conf) -> configurer.accept(builder));
  }

  /**
   * Configurer callback to apply advanced configuration while bootstrapping hibernate:
   *
   * @param configurer Configurer callback.
   * @return This module
   */
  public <T> Hbm doWithSessionFactory(final BiConsumer<SessionFactory, Config> configurer) {
    this.sf = configurer;
    return this;
  }

  /**
   * Configurer callback to apply advanced configuration while bootstrapping hibernate:
   *
   * @param configurer Configurer callback.
   * @return This module
   */
  public <T> Hbm doWithSessionFactory(final Consumer<SessionFactory> configurer) {
    return doWithSessionFactory((builder, conf) -> configurer.accept(builder));
  }

  /**
   * Configurer callback to apply advanced configuration while bootstrapping hibernate:
   *
   * @param configurer Configurer callback.
   * @return This module
   */
  public <T> Hbm doWithSessionFactoryBuilder(
      final BiConsumer<SessionFactoryBuilder, Config> configurer) {
    this.sfb = configurer;
    return this;
  }

  @Override
  public void configure(final Env env, final Config conf, final Binder binder) {
    Key<DataSource> dskey = Key.get(DataSource.class, Names.named(name));
    DataSource ds = env.get(dskey)
        .orElseThrow(() -> new NoSuchElementException("DataSource missing: " + dskey));

    BootstrapServiceRegistryBuilder bsrb = new BootstrapServiceRegistryBuilder();

    this.bsrb.accept(bsrb, conf);

    String ddl_auto = env.name().equals("dev") ? "update" : "none";

    BootstrapServiceRegistry bsr = bsrb.build();
    StandardServiceRegistryBuilder ssrb = new StandardServiceRegistryBuilder(bsr);
    ssrb.applySetting(AvailableSettings.HBM2DDL_AUTO, ddl_auto);

    ssrb.applySettings(settings(env, conf));

    this.ssrb.accept(ssrb, conf);

    ssrb.applySetting(AvailableSettings.DATASOURCE, ds);
    ssrb.applySetting(org.hibernate.cfg.AvailableSettings.DELAY_CDI_ACCESS, true);

    CompletableFuture<Registry> registry = new CompletableFuture<>();
    ssrb.applySetting(org.hibernate.cfg.AvailableSettings.CDI_BEAN_MANAGER, GuiceBeanManager.beanManager(registry));

    StandardServiceRegistry serviceRegistry = ssrb.build();

    MetadataSources sources = new MetadataSources(serviceRegistry);
    this.sources.forEach(src -> src.accept(sources, conf));
    this.metaSources.accept(sources, conf);

    /** scan package? */
    List<URL> packages = sources.getAnnotatedPackages()
        .stream()
        .map(pkg -> getClass().getResource("/" + pkg.replace('.', '/')))
        .collect(Collectors.toList());

    Metadata metadata = sources.getMetadataBuilder()
        .applyImplicitNamingStrategy(ImplicitNamingStrategyJpaCompliantImpl.INSTANCE)
        .applyScanEnvironment(new ScanEnvImpl(packages))
        .build();

    SessionFactoryBuilder sfb = metadata.getSessionFactoryBuilder();
    this.sfb.accept(sfb, conf);
    sfb.applyName(name);

    SessionFactory sessionFactory = sfb.build();
    this.sf.accept(sessionFactory, conf);

    Provider<Session> session = new SessionProvider(sessionFactory);

    ServiceKey serviceKey = env.serviceKey();
    serviceKey.generate(SessionFactory.class, name,
        k -> binder.bind(k).toInstance(sessionFactory));
    serviceKey.generate(EntityManagerFactory.class, name,
        k -> binder.bind(k).toInstance(sessionFactory));

    /** Session/Entity Manager . */
    serviceKey.generate(Session.class, name,
        k -> binder.bind(k).toProvider(session));
    serviceKey.generate(EntityManager.class, name,
        k -> binder.bind(k).toProvider(session));

    /** Unit of work . */
    Provider<UnitOfWork> uow = new UnitOfWorkProvider(sessionFactory);
    serviceKey.generate(UnitOfWork.class, name,
        k -> binder.bind(k).toProvider(uow));

    bindings.forEach(it -> it.accept(binder));

    env.onStart(r -> {
      registry.complete(r);
      listeners.forEach(it -> it.accept((SessionFactoryImplementor) sessionFactory, r));
    });

    env.onStop(sessionFactory::close);
  }

  @Override
  public Config config() {
    return ConfigFactory.parseResources(getClass(), "hbm.conf");
  }

  private static Map<Object, Object> settings(final Env env, final Config config) {
    Map<Object, Object> $ = new HashMap<>();
    config.getConfig("hibernate")
        .entrySet()
        .stream()
        .filter(it -> !it.getKey().equals("packagesToScan"))
        .forEach(e -> $.put("hibernate." + e.getKey(), e.getValue().unwrapped()));

    return $;
  }
}
