/**
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2014 Edgar Espina
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package org.jooby.jdbc;

import com.google.common.base.CharMatcher;
import static com.google.common.base.Preconditions.checkArgument;
import com.google.common.base.Splitter;
import com.google.inject.Binder;
import com.google.inject.Key;
import com.google.inject.name.Names;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigException;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigObject;
import com.typesafe.config.ConfigValue;
import com.typesafe.config.ConfigValueFactory;
import com.typesafe.config.ConfigValueType;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import static java.util.Objects.requireNonNull;
import org.jooby.Env;
import org.jooby.Jooby;
import org.jooby.funzy.Throwing;
import org.jooby.funzy.Try;

import javax.sql.DataSource;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * <h1>jdbc</h1>
 * <p>
 * Production-ready jdbc data source, powered by the
 * <a href="https://github.com/brettwooldridge/HikariCP">HikariCP</a> library.
 * </p>
 *
 * <h2>usage</h2>
 *
 * Via connection string:
 * <pre>
 * {
 *   use(new Jdbc("jdbc:mysql://localhost/db"));
 *
 *   // accessing to the data source
 *   get("/my-api", (req, rsp) {@literal ->} {
 *     DataSource db = req.getInstance(DataSource.class);
 *     // do something with datasource
 *   });
 * }
 * </pre>
 *
 * Via <code>db</code> property:
 * <pre>
 * {
 *   use(new Jdbc("db"));
 *
 *   // accessing to the data source
 *   get("/my-api", (req, rsp) {@literal ->} {
 *     DataSource db = req.getInstance(DataSource.class);
 *     // do something with datasource
 *   });
 * }
 * </pre>
 *
 * <h2>db configuration</h2>
 * <p>
 * Database configuration is controlled from your <code>application.conf</code> file using the
 * <code>db</code> property and friends: <code>db.*</code>.
 * </p>
 *
 * <h3>mem db</h3>
 *
 * <pre>
 *   db = mem
 * </pre>
 *
 * Mem db is implemented with <a href="http://www.h2database.com/">h2 database</a>, before using it
 * make sure to add the h2 dependency to your <code>pom.xml</code>:
 *
 * <pre>
 *   &lt;dependency&gt;
 *     &lt;groupId&gt;com.h2database&lt;/groupId&gt;
 *     &lt;artifactId&gt;h2&lt;/artifactId&gt;
 *   &lt;/dependency&gt;
 * </pre>
 *
 * Mem db is useful for dev environment and/or transient data that can be regenerated.
 *
 * <h3>fs db</h3>
 *
 * <pre>
 *   db = fs
 * </pre>
 *
 * File system db is implemented with <a href="http://www.h2database.com/">h2 database</a>, before
 * using it make sure to add the h2 dependency to your ```pom.xml```:
 *
 * File system db is useful for dev environment and/or transient data that can be
 * regenerated. Keep in mind this db is saved in a tmp directory and db will be deleted it
 * on restarts.
 *
 * <h3>db.url</h3>
 * <p>
 * Connect to a database using a jdbc url, some examples here:
 * </p>
 *
 * <pre>
 *   # mysql
 *   db.url = jdbc:mysql://localhost/mydb
 *   db.user=myuser
 *   db.password=password
 * </pre>
 *
 * Previous example, show you how to connect to <strong>mysql</strong>, setting user and password.
 * But of course you need the jdbc driver on your <code>pom.xml</code>:
 *
 * <h3>hikari configuration</h3>
 * <p>
 * If you need to configure or tweak the <a
 * href="https://github.com/brettwooldridge/HikariCP">hikari pool</a> just add <code>hikari.*</code>
 * entries to your <code>application.conf</code> file:
 * </p>
 *
 * <pre>
 *   db.url = jdbc:mysql://localhost/mydb
 *   db.user=myuser
 *   db.password=password
 *   db.cachePrepStmts=true
 *
 *   # hikari
 *   hikari.autoCommit = true
 *   hikari.maximumPoolSize = 20
 *   # etc...
 * </pre>
 *
 * <p>
 * Also, all the <code>db.*</code> properties are converted to <code>dataSource.*</code> to let <a
 * href="https://github.com/brettwooldridge/HikariCP">hikari</a> configure the target jdbc
 * connection.
 * </p>
 *
 * <h2>multiple connections</h2>
 * It is pretty simple to configure two or more db connections.
 *
 * Let's suppose we have a main database and an audit database for tracking changes:
 *
 * <pre>
 * {
 *   use(new Jdbc("db.main")); // main database
 *   use(new Jdbc("db.audit")); // audit database
 * }
 * </pre>
 *
 * <p>
 * application.conf:
 * </p>
 *
 * <pre>
 * # main database
 * db.main.url = ...
 * db.main.user=...
 * db.main.password = ...
 *
 * # audit
 * db.audit.url = ....
 * db.audit.user = ....
 * db.audit.password = ....
 * </pre>
 *
 * <p>
 * Same principle applies if you need to tweak hikari:
 * </p>
 *
 * <pre>
 * # max pool size for main db
 * hikari.main.maximumPoolSize = 100
 *
 * # max pool size for audit db
 * hikari.audit.maximumPoolSize = 20
 * </pre>
 *
 * <p>
 * Finally, if you need to inject the audit data source, all you have to do is to use the
 * <strong>Name</strong> annotation, like <code>@Name("db.audit")</code>
 * </p>
 *
 *
 * That's all folks! Enjoy it!!!
 *
 * @author edgar
 * @since 0.1.0
 */
public final class Jdbc implements Jooby.Module {

  public final static Function<String, String> DB_NAME = url -> {
    Throwing.Function3<String, String, String, Object[]> indexOf = (str, t1,
        t2) -> {
      int i = str.indexOf(t1);
      int len = i >= 0 ? i : str.length() - 1;
      Map<String, String> params = Splitter.on(t2)
          .trimResults()
          .omitEmptyStrings()
          .withKeyValueSeparator('=')
          .split(str.substring(len + 1));
      return new Object[]{str.substring(0, len + 1), params};
    };
    // strip ; or ?
    Object[] result = indexOf.apply(url, "?", "&");
    Map<String, String> params = new HashMap<>((Map<String, String>) result[1]);
    result = indexOf.apply(result[0].toString(), ";", ";");
    params.putAll((Map<String, String>) result[1]);
    List<String> parts = Splitter.on(CharMatcher.javaLetterOrDigit().negate())
        .trimResults()
        .omitEmptyStrings()
        .splitToList(result[0].toString());
    return Optional.ofNullable(params.get("database"))
        .orElse(Optional.ofNullable(params.get("databaseName"))
            .orElse(parts.get(parts.size() - 1)));
  };

  private static final int DEFAULT_POOL_SIZE = 10;

  private BiConsumer<HikariConfig, Config> callback;

  protected final String dbref;

  protected Optional<String> dbtype;

  /**
   * Creates a new {@link Jdbc} module.
   *
   * @param name A connection string or property with a connection string.
   */
  public Jdbc(final String name) {
    checkArgument(name != null && name.length() > 0,
        "Connection String/Database property required.");
    this.dbref = name;
  }

  /**
   * Creates a new {@link Jdbc} module. The <code>db</code> property must be present in your
   * <code>.conf</code> file.
   */
  public Jdbc() {
    this("db");
  }

  /**
   * Apply advanced configuration options:
   *
   * <pre>{@code
   * {
   *   use(new Jdbc()
   *       .doWith((hikari, conf) -> {
   *         // do with hikari
   *       })
   *   );
   * }
   * }</pre>
   *
   * @param configurer Configurer callback.
   * @return This module.
   */
  public <T> Jdbc doWith(final BiConsumer<HikariConfig, Config> configurer) {
    this.callback = requireNonNull(configurer, "Callback required.");
    return this;
  }

  /**
   * Apply advanced configuration options:
   *
   * <pre>{@code
   * {
   *   use(new Jdbc()
   *       .doWith(hikari -> {
   *         // do with hikari
   *       })
   *   );
   * }
   * }</pre>
   *
   * @param configurer Configurer callback.
   * @return This module
   */
  public Jdbc doWith(final Consumer<HikariConfig> configurer) {
    requireNonNull(configurer, "Configurer required.");
    return doWith((h, c) -> configurer.accept(h));
  }

  @Override
  public void configure(final Env env, final Config config, final Binder binder) {
    Config dbconf;
    String url, dbname, dbkey;
    boolean seturl = false;
    if (dbref.startsWith("jdbc:")) {
      dbconf = config;
      url = dbref;
      dbname = DB_NAME.apply(url);
      dbkey = dbname;
      seturl = true;
    } else {
      dbconf = dbConfig(dbref, config);
      url = dbconf.getString(dbref + ".url");
      dbname = DB_NAME.apply(url);
      dbkey = dbref;
    }

    Set<String> names = Stream.of(dbkey, dbname)
        .distinct()
        .filter(it -> !env.get(Key.get(DataSource.class, Names.named(it))).isPresent())
        .collect(Collectors.toSet());
    if (names.size() == 0) {
      throw new IllegalArgumentException("DataSource(s) already registered.");
    }

    HikariConfig hikariConf = hikariConfig(url, dbkey, dbname, dbconf);

    if (seturl) {
      Properties props = hikariConf.getDataSourceProperties();
      props.setProperty("url", url);
    }

    if (callback != null) {
      callback.accept(hikariConf, config);
    }
    HikariDataSource ds = new HikariDataSource(hikariConf);

    // bind DataSource using dbkey and dbname
    Set<Key<DataSource>> dskeys = new HashSet<>();
    names.forEach(it -> env.serviceKey().generate(DataSource.class, it, k -> {
          binder.bind(k).toInstance(ds);
          env.set(k, ds);
          dskeys.add(k);
        })
    );

    // dbnamekey.url
    env.set(Key.get(String.class, Names.named(dbkey + ".url")), url);

    // db type
    env.set(Key.get(String.class, Names.named(dbkey + ".dbtype")), dbtype.orElse("unknown"));

    // remove global datasource access once application has been initialized:
    env.onStarted(() -> dskeys.forEach(env::unset));

    // close datasource on shutdown:
    env.onStop(ds::close);
  }

  @Override
  public Config config() {
    return ConfigFactory.parseResources(Jdbc.class, "jdbc.conf");
  }

  private Config dbConfig(final String key, final Config source) {
    Object db = source.getAnyRef(key);

    if (db instanceof String) {
      // embedded db?
      return Try.apply(() -> source.getConfig("databases." + db))
          .map(it -> {
            // Rewrite embedded db
            it = it.getConfig("dataSource");
            Config dbtree = it.withValue("url", ConfigValueFactory.fromAnyRef(
                it.getString("url").replace("{mem.seed}", System.currentTimeMillis() + "")));
            // write embedded with current key
            return ConfigFactory.empty()
                .withValue(key, dbtree.root())
                .withFallback(source);
          }).orElseGet(() -> {
            // assume it is a just the url
            return ConfigFactory.empty()
                .withValue(key + ".url", ConfigValueFactory.fromAnyRef(db.toString()))
                .withFallback(source);
          });
    } else {
      return source;
    }
  }

  private HikariConfig hikariConfig(final String url, final String key, final String db,
      final Config conf) {
    Properties props = new Properties();

    BiConsumer<String, Entry<String, ConfigValue>> dumper = (prefix, entry) -> {
      String propertyName = prefix + entry.getKey();
      String propertyValue = entry.getValue().unwrapped().toString();
      props.setProperty(propertyName, propertyValue);
    };

    Throwing.Function<String, Config> dbconf = Throwing.<String, Config>throwingFunction(
        path -> conf.getConfig(path))
        .orElse(ConfigFactory.empty());

    Config $hikari = dbconf.apply(key + ".hikari")
        .withFallback(dbconf.apply("db." + db + ".hikari"))
        .withFallback(dbconf.apply("hikari"));

    // figure it out db type.
    dbtype = dbtype(url);

    /**
     * dump properties from less to higher precedence
     *
     * # databases.[type]
     * # db.* -> dataSource.*
     * # hikari.* -> * (no prefix)
     */
    Stream.of(url.split(":"))
        .forEach(type -> dbconf(conf, type)
            .entrySet().forEach(entry -> dumper.accept("", entry)));

    dbconf.apply(key)
        .withoutPath("hikari")
        .entrySet().forEach(entry -> dumper.accept("dataSource.", entry));

    $hikari.entrySet().forEach(entry -> dumper.accept("", entry));

    if (props.containsKey("driverClassName")) {
      props.remove("dataSourceClassName");
      props.setProperty("jdbcUrl", url);
    }
    // set pool name
    props.setProperty("poolName", dbtype.map(type -> type + "." + db).orElse(db));

    Integer defaultPoolSize = Math.max(DEFAULT_POOL_SIZE, conf.getInt("runtime.processors-x2") + 1);
    props.setProperty("maximumPoolSize",
        props.getOrDefault("maximumPoolSize", defaultPoolSize.toString()).toString());

    return new HikariConfig(props);
  }

  @SuppressWarnings("unchecked")
  private Config dbconf(final Config conf, final String type) {
    try {
      String dbtype = "databases." + type;
      ConfigValue value = conf.getValue(dbtype);
      if (value.valueType() == ConfigValueType.OBJECT) {
        return ((ConfigObject) value).toConfig();
      }
      List<Config> list = (List<Config>) conf.getConfigList(dbtype);
      ClassLoader loader = getClass().getClassLoader();
      return list.stream()
          .filter(it -> dataSourcePresent(loader, it.getString("dataSourceClassName")))
          .findFirst()
          .orElse(list.get(0));
    } catch (ConfigException.Missing | ConfigException.BadPath x) {
      return ConfigFactory.empty();
    }
  }

  private boolean dataSourcePresent(final ClassLoader loader, final String className) {
    try {
      loader.loadClass(className.trim());
      return true;
    } catch (ClassNotFoundException e) {
      return false;
    }
  }

  private Optional<String> dbtype(final String url) {
    String type = Arrays.stream(url.toLowerCase().split(":"))
        .filter(
            token -> !(token.equals("jdbc") || token.equals("jtds") || token.equals("log4jdbc")))
        .findFirst()
        .get();

    return Optional.of(type);
  }

}
