'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

var babelPresetEnv = _interopDefault(require('@babel/preset-env'));
var rollupPluginCommonjs = _interopDefault(require('@rollup/plugin-commonjs'));
var rollupPluginJson = _interopDefault(require('@rollup/plugin-json'));
var rollupPluginNodeResolve = _interopDefault(require('@rollup/plugin-node-resolve'));
var rollupPluginReplace = _interopDefault(require('@rollup/plugin-replace'));
var chalk = _interopDefault(require('chalk'));
var fs = _interopDefault(require('fs'));
var hasha = _interopDefault(require('hasha'));
var isNodeBuiltin = _interopDefault(require('is-builtin-module'));
var mkdirp = _interopDefault(require('mkdirp'));
var ora = _interopDefault(require('ora'));
var path = _interopDefault(require('path'));
var rimraf = _interopDefault(require('rimraf'));
var rollup = require('rollup');
var rollupPluginBabel = _interopDefault(require('rollup-plugin-babel'));
var rollupPluginTerser = require('rollup-plugin-terser');
var validatePackageName = _interopDefault(require('validate-npm-package-name'));
var yargs = _interopDefault(require('yargs-parser'));
var cosmiconfig = require('cosmiconfig');
var jsonschema = require('jsonschema');
var deepmerge = require('deepmerge');
var cacache = _interopDefault(require('cacache'));
var PQueue = _interopDefault(require('p-queue'));
var got = _interopDefault(require('got'));
var cachedir = _interopDefault(require('cachedir'));
var zlib = _interopDefault(require('zlib'));
var babel = _interopDefault(require('@babel/core'));
var glob = _interopDefault(require('glob'));
var esModuleLexer = require('es-module-lexer');

function _defineProperty(obj, key, value) {
  if (key in obj) {
    Object.defineProperty(obj, key, {
      value: value,
      enumerable: true,
      configurable: true,
      writable: true
    });
  } else {
    obj[key] = value;
  }

  return obj;
}

function ownKeys(object, enumerableOnly) {
  var keys = Object.keys(object);

  if (Object.getOwnPropertySymbols) {
    var symbols = Object.getOwnPropertySymbols(object);
    if (enumerableOnly) symbols = symbols.filter(function (sym) {
      return Object.getOwnPropertyDescriptor(object, sym).enumerable;
    });
    keys.push.apply(keys, symbols);
  }

  return keys;
}

function _objectSpread2(target) {
  for (var i = 1; i < arguments.length; i++) {
    var source = arguments[i] != null ? arguments[i] : {};

    if (i % 2) {
      ownKeys(Object(source), true).forEach(function (key) {
        _defineProperty(target, key, source[key]);
      });
    } else if (Object.getOwnPropertyDescriptors) {
      Object.defineProperties(target, Object.getOwnPropertyDescriptors(source));
    } else {
      ownKeys(Object(source)).forEach(function (key) {
        Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key));
      });
    }
  }

  return target;
}

function _objectWithoutPropertiesLoose(source, excluded) {
  if (source == null) return {};
  var target = {};
  var sourceKeys = Object.keys(source);
  var key, i;

  for (i = 0; i < sourceKeys.length; i++) {
    key = sourceKeys[i];
    if (excluded.indexOf(key) >= 0) continue;
    target[key] = source[key];
  }

  return target;
}

function _objectWithoutProperties(source, excluded) {
  if (source == null) return {};

  var target = _objectWithoutPropertiesLoose(source, excluded);

  var key, i;

  if (Object.getOwnPropertySymbols) {
    var sourceSymbolKeys = Object.getOwnPropertySymbols(source);

    for (i = 0; i < sourceSymbolKeys.length; i++) {
      key = sourceSymbolKeys[i];
      if (excluded.indexOf(key) >= 0) continue;
      if (!Object.prototype.propertyIsEnumerable.call(source, key)) continue;
      target[key] = source[key];
    }
  }

  return target;
}

const CONFIG_NAME = 'snowpack'; // default settings

const DEFAULT_CONFIG = {
  dedupe: [],
  installOptions: {
    clean: false,
    hash: false,
    dest: 'web_modules',
    exclude: ['**/__tests__/*', '**/*.@(spec|test).@(js|mjs)'],
    externalPackage: [],
    nomoduleOutput: 'app.nomodule.js',
    optimize: false,
    remoteUrl: 'https://cdn.pika.dev',
    stat: false,
    strict: false,
    env: {}
  },
  rollup: {
    plugins: []
  }
};
const configSchema = {
  type: 'object',
  properties: {
    source: {
      type: 'string'
    },
    entrypoints: {
      type: 'array',
      items: {
        type: 'string'
      }
    },
    // TODO: Array of strings data format is deprecated, remove for v2
    webDependencies: {
      type: ['array', 'object'],
      additionalProperties: {
        type: 'string'
      },
      items: {
        type: 'string'
      }
    },
    dedupe: {
      type: 'array',
      items: {
        type: 'string'
      }
    },
    namedExports: {
      type: 'object',
      additionalProperties: {
        type: 'array',
        items: {
          type: 'string'
        }
      }
    },
    installOptions: {
      type: 'object',
      properties: {
        babel: {
          type: 'boolean'
        },
        hash: {
          type: 'boolean'
        },
        clean: {
          type: 'boolean'
        },
        dest: {
          type: 'string'
        },
        exclude: {
          type: 'array',
          items: {
            type: 'string'
          }
        },
        externalPackage: {
          type: 'array',
          items: {
            type: 'string'
          }
        },
        include: {
          type: 'string'
        },
        nomodule: {
          type: 'string'
        },
        nomoduleOutput: {
          type: 'string'
        },
        optimize: {
          type: 'boolean'
        },
        remotePackage: {
          type: 'array',
          items: {
            type: 'string'
          }
        },
        remoteUrl: {
          type: 'string'
        },
        sourceMap: {
          oneOf: [{
            type: 'boolean'
          }, {
            type: 'string'
          }]
        },
        stat: {
          type: 'boolean'
        },
        strict: {
          type: 'boolean'
        },
        env: {
          type: 'object',
          additionalProperties: {
            oneOf: [{
              id: 'EnvVarString',
              type: 'string'
            }, {
              id: 'EnvVarNumber',
              type: 'number'
            }, {
              id: 'EnvVarTrue',
              type: 'boolean',
              enum: [true]
            }]
          }
        }
      }
    },
    rollup: {
      type: 'object',
      properties: {
        plugins: {
          type: 'array',
          items: {
            type: 'object'
          }
        }
      }
    }
  }
};
/**
 * Convert CLI flags to an incomplete Snowpack config representation.
 * We need to be careful about setting properties here if the flag value
 * is undefined, since the deep merge strategy would then overwrite good
 * defaults with 'undefined'.
 */

function expandCliFlags(flags) {
  const {
    source,
    env,
    help,
    version
  } = flags,
        installOptions = _objectWithoutProperties(flags, ["source", "env", "help", "version"]);

  const result = {
    installOptions
  };

  if (source) {
    result.source = source;
  }

  result.installOptions.env = (env || []).reduce((acc, id) => {
    const index = id.indexOf('=');
    const [key, val] = index > 0 ? [id.substr(0, index), id.substr(index + 1)] : [id, true];
    acc[key] = val;
    return acc;
  }, {});
  return result;
}
/** resolve --dest relative to cwd, and set the default "source" */


function normalizeConfig(config) {
  config.installOptions.dest = path.resolve(process.cwd(), config.installOptions.dest);

  if (Array.isArray(config.webDependencies)) {
    config.entrypoints = config.webDependencies;
    delete config.webDependencies;
  }

  if (!config.source) {
    const isDetailedObject = config.webDependencies && typeof config.webDependencies === 'object';
    config.source = isDetailedObject ? 'pika' : 'local';
  }

  return config;
}

function loadConfig(flags, pkgManifest) {
  const cliConfig = expandCliFlags(flags);
  const explorerSync = cosmiconfig.cosmiconfigSync(CONFIG_NAME, {
    // only support these 3 types of config for now
    searchPlaces: ['package.json', 'snowpack.config.js', 'snowpack.config.json'],
    // don't support crawling up the folder tree:
    stopDir: path.dirname(process.cwd())
  });
  let result; // if user specified --config path, load that

  const errors = [];

  if (flags.config) {
    result = explorerSync.load(path.resolve(flags.config));

    if (!result) {
      errors.push(`Could not locate Snowpack config at ${flags.config}`);
    }
  } else {
    // if --config not given, try searching up the file tree
    result = explorerSync.search();
  } // no config found


  if (!result || !result.config || result.isEmpty) {
    // if CLI flags present, apply those as overrides
    return {
      config: normalizeConfig(deepmerge.all([DEFAULT_CONFIG, {
        webDependencies: pkgManifest.webDependencies
      }, cliConfig])),
      errors
    };
  }

  const config = result.config; // validate against schema; throw helpful user if invalid

  const validation = jsonschema.validate(config, configSchema, {
    allowUnknownAttributes: false,
    propertyName: CONFIG_NAME
  }); // if valid, apply config over defaults

  const mergedConfig = deepmerge.all([DEFAULT_CONFIG, {
    webDependencies: pkgManifest.webDependencies
  }, config, cliConfig]); // if CLI flags present, apply those as overrides

  return {
    config: normalizeConfig(mergedConfig),
    errors: validation.errors.map(msg => `${path.basename(result.filepath)}: ${msg.toString()}`)
  };
}

const PIKA_CDN = `https://cdn.pika.dev`;
const CACHE_DIR = cachedir('snowpack');
const RESOURCE_CACHE = path.join(CACHE_DIR, 'pkg-cache-1.4');
const HAS_CDN_HASH_REGEX = /\-[a-zA-Z0-9]{16,}/;
async function readLockfile(cwd) {
  try {
    var lockfileContents = fs.readFileSync(path.join(cwd, 'snowpack.lock.json'), {
      encoding: 'utf8'
    });
  } catch (err) {
    // no lockfile found, ignore and continue
    return null;
  } // If this fails, we actually do want to alert the user by throwing


  return JSON.parse(lockfileContents);
}
async function writeLockfile(loc, importMap) {
  const sortedImportMap = {
    imports: {}
  };

  for (const key of Object.keys(importMap.imports).sort()) {
    sortedImportMap.imports[key] = importMap.imports[key];
  }

  fs.writeFileSync(loc, JSON.stringify(sortedImportMap, undefined, 2), {
    encoding: 'utf8'
  });
}
function fetchCDNResource(resourceUrl) {
  if (!resourceUrl.startsWith(PIKA_CDN)) {
    resourceUrl = PIKA_CDN + resourceUrl;
  }

  return got(resourceUrl, {
    headers: {
      'user-agent': `snowpack/v1.4 (https://snowpack.dev)`
    },
    throwHttpErrors: false
  });
}
function isTruthy(item) {
  return Boolean(item);
}
/**
 * Given a package name, look for that package's package.json manifest.
 * Return both the manifest location (if believed to exist) and the
 * manifest itself (if found).
 *
 * NOTE: You used to be able to require() a package.json file directly,
 * but now with export map support in Node v13 that's no longer possible.
 */

function resolveDependencyManifest(dep, cwd) {
  // Attempt #1: Resolve the dependency manifest normally. This works for most
  // packages, but fails when the package defines an export map that doesn't
  // include a package.json. If we detect that to be the reason for failure,
  // move on to our custom implementation.
  try {
    const depManifest = require.resolve(`${dep}/package.json`, {
      paths: [cwd]
    });

    return [depManifest, require(depManifest)];
  } catch (err) {
    if ((err.code === 'MODULE_NOT_FOUND' || err.code === 'ERR_PACKAGE_PATH_NOT_EXPORTED') && err.message.includes(`'./package.json'`)) ; else {
      throw err;
    }
  } // Attempt #2: Resolve the dependency manifest manually. This involves resolving
  // the dep itself to find the entrypoint file, and then haphazardly replacing the
  // file path within the package with a "./package.json" instead. It's not as
  // thorough as Attempt #1, but it should work well until export maps become more
  // established & move out of experimental mode.


  let result = [null, null];

  try {
    const fullPath = require.resolve(dep, {
      paths: [cwd]
    }); // Strip everything after the package name to get the package root path
    // NOTE: This find-replace is very gross, replace with something like upath.


    const searchPath = `${path.sep}node_modules${path.sep}${dep.replace('/', path.sep)}`;
    const indexOfSearch = fullPath.lastIndexOf(searchPath);

    if (indexOfSearch >= 0) {
      const manifestPath = fullPath.substring(0, indexOfSearch + searchPath.length + 1) + 'package.json';
      result[0] = manifestPath;
      const manifestStr = fs.readFileSync(manifestPath, {
        encoding: 'utf8'
      });
      result[1] = JSON.parse(manifestStr);
    }
  } catch (err) {// ignore
  } finally {
    return result;
  }
}
/**
 * If Rollup erred parsing a particular file, show suggestions based on its
 * file extension (note: lowercase is fine).
 */

const MISSING_PLUGIN_SUGGESTIONS = {
  '.css': 'Try installing rollup-plugin-postcss and adding it to Snowpack (https://www.snowpack.dev/#custom-rollup-plugins)',
  '.svelte': 'Try installing rollup-plugin-svelte and adding it to Snowpack (https://www.snowpack.dev/#custom-rollup-plugins)',
  '.vue': 'Try installing rollup-plugin-vue and adding it to Snowpack (https://www.snowpack.dev/#custom-rollup-plugins)'
};

/**
 * Given an install specifier, attempt to resolve it from the CDN.
 * If no lockfile exists or if the entry is not found in the lockfile, attempt to resolve
 * it from the CDN directly. Otherwise, use the URL found in the lockfile and attempt to
 * check the local cache first.
 *
 * All resolved URLs are populated into the local cache, where our internal Rollup engine
 * will load them from when it installs your dependencies to disk.
 */

async function resolveDependency(installSpecifier, packageSemver, lockfile, canRetry = true) {
  // Right now, the CDN is only for top-level JS packages. The CDN doesn't support CSS,
  // non-JS assets, and has limited support for deep package imports. Snowpack
  // will automatically fall-back any failed/not-found assets from local
  // node_modules/ instead.
  if (!validatePackageName(installSpecifier).validForNewPackages) {
    return null;
  } // Grab the installUrl from our lockfile if it exists, otherwise resolve it yourself.


  let installUrl;
  let installUrlType;

  if (lockfile && lockfile.imports[installSpecifier]) {
    installUrl = lockfile.imports[installSpecifier];
    installUrlType = 'pin';
  } else {
    if (packageSemver === 'latest') {
      console.warn(`warn(${installSpecifier}): Not found in "dependencies". Using latest package version...`);
    }

    if (packageSemver.startsWith('npm:@reactesm') || packageSemver.startsWith('npm:@pika/react')) {
      throw new Error(`React workaround packages no longer needed! Revert to the official React & React-DOM packages.`);
    }

    if (packageSemver.includes(' ') || packageSemver.includes(':')) {
      console.warn(`warn(${installSpecifier}): Can't fetch complex semver "${packageSemver}" from remote CDN.`);
      return null;
    }

    installUrlType = 'lookup';
    installUrl = `${PIKA_CDN}/${installSpecifier}@${packageSemver}`;
  } // Hashed CDN urls never change, so its safe to grab them directly from the local cache
  // without a network request.


  if (installUrlType === 'pin') {
    const cachedResult = await cacache.get.info(RESOURCE_CACHE, installUrl).catch(() => null);

    if (cachedResult) {
      if (cachedResult.metadata) {
        const {
          pinnedUrl
        } = cachedResult.metadata;
        return pinnedUrl;
      }
    }
  } // Otherwise, resolve from the CDN remotely.


  const {
    statusCode,
    headers,
    body
  } = await fetchCDNResource(installUrl);

  if (statusCode !== 200) {
    console.warn(`Failed to resolve [${statusCode}]: ${installUrl} (${body})`);
    console.warn(`Falling back to local copy...`);
    return null;
  }

  if (installUrlType === 'pin') {
    const pinnedUrl = installUrl;
    await cacache.put(RESOURCE_CACHE, pinnedUrl, body, {
      metadata: {
        pinnedUrl
      }
    });
    return pinnedUrl;
  }

  let importUrlPath = headers['x-import-url'];
  let pinnedUrlPath = headers['x-pinned-url'];
  const buildStatus = headers['x-import-status'];

  if (pinnedUrlPath) {
    const pinnedUrl = `${PIKA_CDN}${pinnedUrlPath}`;
    await cacache.put(RESOURCE_CACHE, pinnedUrl, body, {
      metadata: {
        pinnedUrl
      }
    });
    return pinnedUrl;
  }

  if (buildStatus === 'SUCCESS') {
    console.warn(`Failed to lookup [${statusCode}]: ${installUrl}`);
    console.warn(`Falling back to local copy...`);
    return null;
  }

  if (!canRetry || buildStatus === 'FAIL') {
    console.warn(`Failed to build: ${installSpecifier}@${packageSemver}`);
    console.warn(`Falling back to local copy...`);
    return null;
  }

  console.log(chalk.cyan(`Building ${installSpecifier}@${packageSemver}... (This takes a moment, but will be cached for future use)`));

  if (!importUrlPath) {
    throw new Error('X-Import-URL header expected, but none received.');
  }

  const {
    statusCode: lookupStatusCode
  } = await fetchCDNResource(importUrlPath);

  if (lookupStatusCode !== 200) {
    throw new Error(`Unexpected response [${lookupStatusCode}]: ${PIKA_CDN}${importUrlPath}`);
  }

  return resolveDependency(installSpecifier, packageSemver, lockfile, false);
}

async function resolveTargetsFromRemoteCDN(installTargets, lockfile, pkgManifest, config) {
  const downloadQueue = new PQueue({
    concurrency: 16
  });
  const newLockfile = {
    imports: {}
  };
  let resolutionError;
  const allInstallSpecifiers = new Set(installTargets.map(dep => dep.specifier));

  for (const installSpecifier of allInstallSpecifiers) {
    const installSemver = (config.webDependencies || {})[installSpecifier] || (pkgManifest.webDependencies || {})[installSpecifier] || (pkgManifest.dependencies || {})[installSpecifier] || (pkgManifest.devDependencies || {})[installSpecifier] || (pkgManifest.peerDependencies || {})[installSpecifier] || 'latest';
    downloadQueue.add(async () => {
      try {
        const resolvedUrl = await resolveDependency(installSpecifier, installSemver, lockfile);

        if (resolvedUrl) {
          newLockfile.imports[installSpecifier] = resolvedUrl;
        }
      } catch (err) {
        resolutionError = resolutionError || err;
      }
    });
  }

  await downloadQueue.onIdle();

  if (resolutionError) {
    throw resolutionError;
  }

  return newLockfile;
}
function clearCache() {
  return cacache.rm.all(RESOURCE_CACHE);
}

const CACHED_FILE_ID_PREFIX = 'snowpack-pkg-cache:';
const PIKA_CDN_TRIM_LENGTH = PIKA_CDN.length;
/**
 * rollup-plugin-remote-cdn
 *
 * Load import URLs from a remote CDN, sitting behind a local cache. The local
 * cache acts as a go-between for the resolve & load step: when we get back a
 * successful CDN resolution, we save the file to the local cache and then tell
 * rollup that it's safe to load from the cache in the `load()` hook.
 */

function rollupPluginDependencyCache({
  log
}) {
  return {
    name: 'snowpack:rollup-plugin-remote-cdn',

    async resolveId(source, importer) {
      let cacheKey;

      if (source.startsWith(PIKA_CDN)) {
        cacheKey = source;
      } else if (source.startsWith('/-/')) {
        cacheKey = PIKA_CDN + source;
      } else if (source.startsWith('/pin/')) {
        cacheKey = PIKA_CDN + source;
      } else {
        return null;
      } // If the source path is a CDN path including a hash, it's assumed the
      // file will never change and it is safe to pull from our local cache
      // without a network request.


      log(cacheKey);

      if (HAS_CDN_HASH_REGEX.test(cacheKey)) {
        const cachedResult = await cacache.get.info(RESOURCE_CACHE, cacheKey).catch(() =>
        /* ignore */
        null);

        if (cachedResult) {
          return CACHED_FILE_ID_PREFIX + cacheKey;
        }
      } // Otherwise, make the remote request and cache the file on success.


      const response = await fetchCDNResource(cacheKey);

      if (response.statusCode === 200) {
        await cacache.put(RESOURCE_CACHE, cacheKey, response.body);
        return CACHED_FILE_ID_PREFIX + cacheKey;
      } // If lookup failed, skip this plugin and resolve the import locally instead.
      // TODO: Log that this has happened (if some sort of verbose mode is enabled).


      const packageName = cacheKey.substring(PIKA_CDN_TRIM_LENGTH).replace('/-/', '').replace('/pin/', '').split('@')[0];
      return this.resolve(packageName, importer, {
        skipSelf: true
      }).then(resolved => {
        let finalResult = resolved;

        if (!finalResult) {
          finalResult = {
            id: packageName
          };
        }

        return finalResult;
      });
    },

    async load(id) {
      if (!id.startsWith(CACHED_FILE_ID_PREFIX)) {
        return null;
      }

      const cacheKey = id.substring(CACHED_FILE_ID_PREFIX.length);
      log(cacheKey);
      const cachedResult = await cacache.get(RESOURCE_CACHE, cacheKey);
      return cachedResult.data.toString('utf8');
    }

  };
}

const IS_DEEP_PACKAGE_IMPORT = /^(@[\w-]+\/)?([\w-]+)\/(.*)/;
/**
 * rollup-plugin-entrypoint-alias
 *
 * Aliases any deep imports from a package to the package name, so that
 * chunking can happen more accurately.
 *
 * Example: lit-element imports from both 'lit-html' & 'lit-html/lit-html.js'.
 * Even though both eventually resolve to the same place, without this plugin
 * we lose the ability to mark "lit-html" as an external package.
 */

function rollupPluginEntrypointAlias({
  cwd
}) {
  return {
    name: 'snowpack:rollup-plugin-entrypoint-alias',

    resolveId(source, importer) {
      if (!IS_DEEP_PACKAGE_IMPORT.test(source)) {
        return null;
      }

      const [, packageScope, packageName] = source.match(IS_DEEP_PACKAGE_IMPORT);
      const packageFullName = packageScope ? `${packageScope}${packageName}` : packageName;
      const [, manifest] = resolveDependencyManifest(packageFullName, cwd);

      if (!manifest) {
        return null;
      }

      let needsAlias = typeof manifest.module === 'string' && source === path.posix.join(packageFullName, manifest.module) || typeof manifest.browser === 'string' && source === path.posix.join(packageFullName, manifest.browser) || typeof manifest.main === 'string' && source === path.posix.join(packageFullName, manifest.main);

      if (!needsAlias) {
        return null;
      }

      return this.resolve(packageFullName, importer, {
        skipSelf: true
      }).then(resolved => {
        return resolved || null;
      });
    }

  };
}

function rollupPluginDependencyStats(cb) {
  let outputDir;
  let existingFileCache = {};
  let statsSummary = {
    direct: {},
    common: {}
  };

  function buildExistingFileCache(bundle) {
    for (let fileName of Object.keys(bundle)) {
      const filePath = path.join(outputDir, fileName);

      if (fs.existsSync(filePath)) {
        const {
          size
        } = fs.statSync(filePath);
        existingFileCache[fileName] = size;
      }
    }
  }

  function compareDependencies(files, type) {
    for (let {
      fileName,
      contents
    } of files) {
      const size = contents.byteLength;
      statsSummary[type][fileName] = {
        size: size,
        gzip: zlib.gzipSync(contents).byteLength,
        brotli: zlib.brotliCompressSync(contents).byteLength
      };

      if (existingFileCache[fileName]) {
        const delta = (size - existingFileCache[fileName]) / 1000;
        statsSummary[type][fileName].delta = delta;
      }
    }
  }

  return {
    generateBundle(options, bundle) {
      outputDir = options.dir;
      buildExistingFileCache(bundle);
    },

    writeBundle(options, bundle) {
      const directDependencies = [];
      const commonDependencies = [];

      for (const [fileName, assetOrChunk] of Object.entries(bundle)) {
        const raw = assetOrChunk.type === 'asset' ? assetOrChunk.source : assetOrChunk.code;
        const contents = Buffer.isBuffer(raw) ? raw : typeof raw === 'string' ? Buffer.from(raw, 'utf8') : Buffer.from(raw);

        if (fileName.startsWith('common')) {
          commonDependencies.push({
            fileName,
            contents
          });
        } else {
          directDependencies.push({
            fileName,
            contents
          });
        }
      }

      compareDependencies(directDependencies, 'direct');
      compareDependencies(commonDependencies, 'common');
      cb(statsSummary);
    }

  };
}

/**
 * rollup-plugin-treeshake-inputs
 *
 * How it works:
 * 1. An array of "install targets" are passed in, describing all known imports + metadata.
 * 2. Known imports are marked for tree-shaking by appending 'pika-treeshake:' to the input value.
 * 3. On load, we return a false virtual file for all "pika-treeshake:" inputs.
 *    a. That virtual file contains only `export ... from 'ACTUAL_FILE_PATH';` exports
 *    b. Rollup uses those exports to drive its tree-shaking algorithm.
 */

function rollupPluginTreeshakeInputs(allImports) {
  const installTargetsByFile = {};
  return {
    name: 'snowpack:treeshake-inputs',

    // Mark some inputs for tree-shaking.
    options(inputOptions) {
      const input = inputOptions.input;

      for (const [key, val] of Object.entries(input)) {
        installTargetsByFile[val] = allImports.filter(imp => imp.specifier === key); // If an input has known install targets, and none of those have "all=true", mark for treeshaking.

        if (installTargetsByFile[val].length > 0 && !installTargetsByFile[val].some(imp => imp.all)) {
          input[key] = `pika-treeshake:${val}`;
        }
      }

      return inputOptions;
    },

    resolveId(source) {
      if (source.startsWith('pika-treeshake:')) {
        return source;
      }

      return null;
    },

    load(id) {
      if (!id.startsWith('pika-treeshake:')) {
        return null;
      }

      const fileLoc = id.substring('pika-treeshake:'.length); // Reduce all install targets into a single "summarized" install target.

      const treeshakeSummary = installTargetsByFile[fileLoc].reduce((summary, imp) => {
        summary.default = summary.default || imp.default;
        summary.namespace = summary.namespace || imp.namespace;
        summary.named = [...summary.named, ...imp.named];
        return summary;
      });
      const uniqueNamedImports = new Set(treeshakeSummary.named);
      const normalizedFileLoc = fileLoc.split(path.win32.sep).join(path.posix.sep);
      const result = `
        ${treeshakeSummary.namespace ? `export * from '${normalizedFileLoc}';` : ''}
        ${treeshakeSummary.default ? `import __pika_web_default_export_for_treeshaking__ from '${normalizedFileLoc}'; export default __pika_web_default_export_for_treeshaking__;` : ''}
        ${`export {${[...uniqueNamedImports].join(',')}} from '${normalizedFileLoc}';`}
      `;
      return result;
    }

  };
}

const WEB_MODULES_TOKEN = 'web_modules/';
const WEB_MODULES_TOKEN_LENGTH = WEB_MODULES_TOKEN.length; // [@\w] - Match a word-character or @ (valid package name)
// (?!.*(:\/\/)) - Ignore if previous match was a protocol (ex: http://)

const BARE_SPECIFIER_REGEX = /^[@\w](?!.*(:\/\/))/;
const HAS_NAMED_IMPORTS_REGEX = /^[\t-\r ,0-9A-Z_a-z\xA0\u1680\u2000-\u200A\u2028\u2029\u202F\u205F\u3000\uFEFF]*\{([\s\S]*)\}/;
const SPLIT_NAMED_IMPORTS_REGEX = /\bas[\t-\r \xA0\u1680\u2000-\u200A\u2028\u2029\u202F\u205F\u3000\uFEFF]+[0-9A-Z_a-z]+|,/;
const DEFAULT_IMPORT_REGEX = /import[\t-\r \xA0\u1680\u2000-\u200A\u2028\u2029\u202F\u205F\u3000\uFEFF]+([0-9A-Z_a-z])+(,[\t-\r \xA0\u1680\u2000-\u200A\u2028\u2029\u202F\u205F\u3000\uFEFF]\{[\t-\r 0-9A-Z_a-z\xA0\u1680\u2000-\u200A\u2028\u2029\u202F\u205F\u3000\uFEFF]*\})?[\t-\r \xA0\u1680\u2000-\u200A\u2028\u2029\u202F\u205F\u3000\uFEFF]+from/;

function stripJsExtension(dep) {
  return dep.replace(/\.m?js$/i, '');
}

function createInstallTarget(specifier, all = true) {
  return {
    specifier,
    all,
    default: false,
    namespace: false,
    named: []
  };
}

function removeSpecifierQueryString(specifier) {
  const queryStringIndex = specifier.indexOf('?');

  if (queryStringIndex >= 0) {
    specifier = specifier.substring(0, queryStringIndex);
  }

  return specifier;
}

function getWebModuleSpecifierFromCode(code, imp) {
  if (imp.d > -1) {
    return code.substring(imp.s + 1, imp.e - 1);
  }

  return code.substring(imp.s, imp.e);
}
/**
 * parses an import specifier, looking for a web modules to install. If a web module is not detected,
 * null is returned.
 */


function parseWebModuleSpecifier(specifier) {
  // If specifier is a "bare module specifier" (ie: package name) just return it directly
  if (BARE_SPECIFIER_REGEX.test(specifier)) {
    return specifier;
  } // Clean the specifier, remove any query params that may mess with matching


  const cleanedSpecifier = removeSpecifierQueryString(specifier); // Otherwise, check that it includes the "web_modules/" directory

  const webModulesIndex = cleanedSpecifier.indexOf(WEB_MODULES_TOKEN);

  if (webModulesIndex === -1) {
    return null;
  } // Check if this matches `@scope/package.js` or `package.js` format.
  // If it is, assume that this is a top-level pcakage that should be installed without the “.js”


  const resolvedSpecifier = cleanedSpecifier.substring(webModulesIndex + WEB_MODULES_TOKEN_LENGTH);
  const resolvedSpecifierWithoutExtension = stripJsExtension(resolvedSpecifier);

  if (validatePackageName(resolvedSpecifierWithoutExtension).validForNewPackages) {
    return resolvedSpecifierWithoutExtension;
  } // Otherwise, this is an explicit import to a file within a package.


  return resolvedSpecifier;
}

function parseImportStatement(code, imp) {
  const webModuleSpecifier = parseWebModuleSpecifier(getWebModuleSpecifierFromCode(code, imp));

  if (!webModuleSpecifier) {
    return null;
  }

  const importStatement = code.substring(imp.ss, imp.se);
  const dynamicImport = imp.d > -1;
  const defaultImport = !dynamicImport && DEFAULT_IMPORT_REGEX.test(importStatement);
  const namespaceImport = !dynamicImport && importStatement.includes('*');
  const namedImports = (importStatement.match(HAS_NAMED_IMPORTS_REGEX) || [, ''])[1].split(SPLIT_NAMED_IMPORTS_REGEX).map(name => name.trim()).filter(isTruthy);
  return {
    specifier: webModuleSpecifier,
    all: dynamicImport,
    default: defaultImport,
    namespace: namespaceImport,
    named: namedImports
  };
}

function getInstallTargetsForFile(code) {
  const [imports] = esModuleLexer.parse(code) || [];
  const allImports = imports.map(imp => parseImportStatement(code, imp)).filter(isTruthy);
  return allImports;
}

function scanDepList(depList, cwd) {
  return depList.map(whitelistItem => {
    if (!glob.hasMagic(whitelistItem)) {
      return [createInstallTarget(whitelistItem, true)];
    } else {
      const nodeModulesLoc = path.join(cwd, 'node_modules');
      return scanDepList(glob.sync(whitelistItem, {
        cwd: nodeModulesLoc,
        nodir: true
      }), cwd);
    }
  }).reduce((flat, item) => flat.concat(item), []);
}
async function scanImports({
  include,
  exclude
}) {
  await esModuleLexer.init;
  const includeFiles = glob.sync(include, {
    ignore: exclude,
    nodir: true
  });

  if (!includeFiles.length) {
    console.warn(`[SCAN ERROR]: No files matching "${include}"`);
    return [];
  } // Scan every matched JS file for web dependency imports


  const loadedFiles = await Promise.all(includeFiles.map(async filePath => {
    // Our import scanner can handle normal JS & even TypeScript without a problem.
    if (filePath.endsWith('.js') || filePath.endsWith('.mjs') || filePath.endsWith('.ts')) {
      return fs.promises.readFile(filePath, 'utf-8');
    } // JSX breaks our import scanner, so we need to transform it before sending it to our scanner.


    if (filePath.endsWith('.jsx') || filePath.endsWith('.tsx')) {
      const result = await babel.transformFileAsync(filePath, {
        plugins: [[require('@babel/plugin-transform-react-jsx'), {
          runtime: 'classic'
        }], [require('@babel/plugin-syntax-typescript'), {
          isTSX: true
        }]],
        babelrc: false,
        configFile: false
      });
      return result && result.code;
    }

    console.warn(chalk.dim(`ignoring unsupported file "${filePath}"`));
    return null;
  }));
  return loadedFiles.filter(code => !!code).map(code => getInstallTargetsForFile(code)).reduce((flat, item) => flat.concat(item), []).sort((impA, impB) => impA.specifier.localeCompare(impB.specifier));
}

/** The minimum width, in characters, of each size column */

const SIZE_COLUMN_WIDTH = 11;
/** Generic Object.entries() alphabetical sort by keys. */

function entriesSort([filenameA], [filenameB]) {
  return filenameA.localeCompare(filenameB);
}
/** Pretty-prints number of bytes as "XXX KB" */


function formatSize(size) {
  let kb = Math.round(size / 1000 * 100) / 100;

  if (kb >= 1000) {
    kb = Math.floor(kb);
  }

  let color;

  if (kb < 15) {
    color = 'green';
  } else if (kb < 30) {
    color = 'yellow';
  } else {
    color = 'red';
  }

  return chalk[color](`${kb} KB`.padEnd(SIZE_COLUMN_WIDTH));
}

function formatDelta(delta) {
  const kb = Math.round(delta * 100) / 100;
  const color = delta > 0 ? 'red' : 'green';
  return chalk[color](`Δ ${delta > 0 ? '+' : ''}${kb} KB`);
}

function formatFileInfo(filename, stats, padEnd, isLastFile) {
  const lineGlyph = chalk.dim(isLastFile ? '└─' : '├─');
  const lineName = filename.padEnd(padEnd);
  const fileStat = formatSize(stats.size);
  const gzipStat = formatSize(stats.gzip);
  const brotliStat = formatSize(stats.brotli);
  const lineStat = fileStat + gzipStat + brotliStat;
  let lineDelta = '';

  if (stats.delta) {
    lineDelta = chalk.dim('[') + formatDelta(stats.delta) + chalk.dim(']');
  } // Trim trailing whitespace (can mess with formatting), but keep indentation.


  return `    ` + `${lineGlyph} ${lineName} ${lineStat} ${lineDelta}`.trim();
}

function formatFiles(files, padEnd) {
  const strippedFiles = files.map(([filename, stats]) => [filename.replace(/^common\//, ''), stats]);
  return strippedFiles.map(([filename, stats], index) => formatFileInfo(filename, stats, padEnd, index >= files.length - 1)).join('\n');
}

function printStats(dependencyStats) {
  let output = '';
  const {
    direct,
    common
  } = dependencyStats;
  const allDirect = Object.entries(direct).sort(entriesSort);
  const allCommon = Object.entries(common).sort(entriesSort);
  const maxFileNameLength = [...allCommon, ...allDirect].reduce((max, [filename]) => Math.max(filename.length, max), 'web_modules/'.length) + 1;
  output += `  ⦿ ${chalk.bold('web_modules/'.padEnd(maxFileNameLength + 4))}` + chalk.bold.underline('size'.padEnd(SIZE_COLUMN_WIDTH - 2)) + '  ' + chalk.bold.underline('gzip'.padEnd(SIZE_COLUMN_WIDTH - 2)) + '  ' + chalk.bold.underline('brotli'.padEnd(SIZE_COLUMN_WIDTH - 2)) + `\n`;
  output += `${formatFiles(allDirect, maxFileNameLength)}\n`;

  if (Object.values(common).length > 0) {
    output += `  ⦿ ${chalk.bold('web_modules/common/ (Shared)')}\n`;
    output += `${formatFiles(allCommon, maxFileNameLength)}\n`;
  }

  return `\n${output}\n`;
}

const ALWAYS_SHOW_ERRORS = new Set(['react', 'react-dom']);
const cwd = process.cwd();
const banner = chalk.bold(`snowpack`) + ` installing... `;
const installResults = [];
let dependencyStats = null;
let spinner = ora(banner);
let spinnerHasError = false;

function printHelp() {
  console.log(`
${chalk.bold(`snowpack`)} - Install npm dependencies to run natively on the web.
${chalk.bold('Options:')}
  --dest [path]             Specify destination directory (default: "web_modules/").
  --clean                   Clear out the destination directory before install.
  --optimize                Transpile, minify, and optimize installed dependencies for production.
  --env                     Set environment variable(s) inside dependencies:
                                - if only NAME given, reads value from real env var
                                - if \`NAME=value\`, uses given value
                                - NODE_ENV defaults to "production" with "--optimize" (overridable)
  --babel                   Transpile installed dependencies. Also enabled with "--optimize".
  --include [glob]          Auto-detect imports from file(s). Supports glob.
  --exclude [glob]          Exclude files from --include. Follows glob’s ignore pattern.
  --config [path]           Location of Snowpack config file.
  --strict                  Only install pure ESM dependency trees. Fail if a CJS module is encountered.
  --no-source-map           Skip emitting source map files (.js.map) into dest
  --stat                    Logs install statistics after installing, with information on install targets and file sizes. Useful for CI, performance review.
  --nomodule [path]         Your app’s entry file for generating a <script nomodule> bundle
  --nomodule-output [path]  Filename for nomodule output (default: "app.nomodule.js")
    ${chalk.bold('Advanced:')}
  --external-package [val]  Internal use only, may be removed at any time.
    `.trim());
}

async function generateHashFromFile(targetLoc) {
  const longHash = await hasha.fromFile(targetLoc, {
    algorithm: 'md5'
  });
  return longHash === null || longHash === void 0 ? void 0 : longHash.slice(0, 10);
}

function formatInstallResults(skipFailures) {
  return installResults.map(([d, result]) => {
    if (result === 'SUCCESS') {
      return chalk.green(d);
    }

    if (result === 'ASSET') {
      return chalk.yellow(d);
    }

    if (result === 'FAIL') {
      return skipFailures ? chalk.dim(d) : chalk.red(d);
    }

    return d;
  }).join(', ');
}

function logError(msg) {
  if (!spinnerHasError) {
    spinner.stopAndPersist({
      symbol: chalk.cyan('⠼')
    });
  }

  spinnerHasError = true;
  spinner = ora(chalk.red(msg));
  spinner.fail();
}

function logUpdate(msg) {
  spinner.text = banner + msg;
}

class ErrorWithHint extends Error {
  constructor(message, hint) {
    super(message);
    this.hint = hint;
  }

} // Add common, well-used non-esm packages here so that Rollup doesn't die trying to analyze them.


const PACKAGES_TO_AUTO_DETECT_EXPORTS = [path.join('react', 'index.js'), path.join('react-dom', 'index.js'), 'react-is', 'prop-types', 'scheduler', 'rxjs', 'exenv', 'body-scroll-lock'];

function detectExports(filePath) {
  try {
    const fileLoc = require.resolve(filePath, {
      paths: [cwd]
    });

    if (fs.existsSync(fileLoc)) {
      return Object.keys(require(fileLoc)).filter(e => e[0] !== '_');
    }
  } catch (err) {// ignore
  }
}
/**
 * Resolve a "webDependencies" input value to the correct absolute file location.
 * Supports both npm package names, and file paths relative to the node_modules directory.
 * Follows logic similar to Node's resolution logic, but using a package.json's ESM "module"
 * field instead of the CJS "main" field.
 */


function resolveWebDependency(dep, isExplicit) {
  // if dep includes a file extension, check that dep isn't a package before returning
  if (path.extname(dep) && !validatePackageName(dep).validForNewPackages) {
    const isJSFile = ['.js', '.mjs', '.cjs'].includes(path.extname(dep));
    return {
      type: isJSFile ? 'JS' : 'ASSET',
      loc: require.resolve(dep, {
        paths: [cwd]
      })
    };
  }

  const [depManifestLoc, depManifest] = resolveDependencyManifest(dep, cwd);

  if (!depManifest) {
    throw new ErrorWithHint(`"${dep}" not found. Have you installed the package via npm?`, depManifestLoc && chalk.italic(depManifestLoc));
  }

  let foundEntrypoint = depManifest['browser:module'] || depManifest.module || depManifest['main:esnext'] || depManifest.browser; // Some packages define "browser" as an object. We'll do our best to find the
  // right entrypoint in an entrypoint object, or fail otherwise.
  // See: https://github.com/defunctzombie/package-browser-field-spec

  if (typeof foundEntrypoint === 'object') {
    foundEntrypoint = foundEntrypoint[dep] || foundEntrypoint['./index.js'] || foundEntrypoint['./index'] || foundEntrypoint['./'] || foundEntrypoint['.'];
  } // If the package was a part of the explicit whitelist, fallback to it's main CJS entrypoint.


  if (!foundEntrypoint && isExplicit) {
    foundEntrypoint = depManifest.main || 'index.js';
  }

  if ((dep === 'react' || dep === 'react-dom') && (!foundEntrypoint || foundEntrypoint === 'index.js')) {
    throw new ErrorWithHint(chalk.bold(`Dependency "${dep}" has no native "module" entrypoint.`) + `
  To continue, install our drop-in, ESM-ready builds of "react" & "react-dom" to your project:
    npm: npm install react@npm:@pika/react react-dom@npm:@pika/react-dom
    yarn: yarn add react@npm:@pika/react react-dom@npm:@pika/react-dom`, chalk.italic(`See ${chalk.underline('https://www.snowpack.dev/#react')} for more info.`));
  }

  if (!foundEntrypoint) {
    throw new ErrorWithHint(`dependency "${dep}" has no native "module" entrypoint.`, chalk.italic(`Tip: Find modern, web-ready packages at ${chalk.underline('https://www.pika.dev')}`));
  }

  if (typeof foundEntrypoint !== 'string') {
    throw new Error(`"${dep}" has unexpected entrypoint: ${JSON.stringify(foundEntrypoint)}.`);
  }

  return {
    type: 'JS',
    loc: path.join(depManifestLoc, '..', foundEntrypoint)
  };
}
/**
 * Formats the snowpack dependency name from a "webDependencies" input value:
 * 2. Remove any ".js"/".mjs" extension (will be added automatically by Rollup)
 */


function getWebDependencyName(dep) {
  return dep.replace(/\.m?js$/i, '');
}
/**
 * Takes object of env var mappings and converts it to actual
 * replacement specs as expected by @rollup/plugin-replace. The
 * `optimize` arg is used to derive NODE_ENV default.
 *
 * @param env
 * @param optimize
 */


function getRollupReplaceKeys(env, optimize) {
  const result = Object.keys(env).reduce((acc, id) => {
    const val = env[id];
    acc[`process.env.${id}`] = `"${val === true ? process.env[id] : val}"`;
    return acc;
  }, {
    'process.env.NODE_ENV': optimize ? '"production"' : '"development"',
    'process.env.': '({}).'
  });
  return result;
}

async function install(installTargets, {
  hasBrowserlistConfig,
  isExplicit,
  lockfile
}, config) {
  const {
    dedupe,
    namedExports,
    source,
    installOptions: {
      babel: isBabel,
      dest: destLoc,
      hash: useHash,
      externalPackage: externalPackages,
      nomodule,
      nomoduleOutput,
      optimize: isOptimized,
      sourceMap,
      strict: isStrict,
      stat: withStats,
      env
    },
    rollup: userDefinedRollup
  } = config;

  const knownNamedExports = _objectSpread2({}, namedExports);

  for (const filePath of PACKAGES_TO_AUTO_DETECT_EXPORTS) {
    knownNamedExports[filePath] = knownNamedExports[filePath] || detectExports(filePath) || [];
  }

  if (source === 'local' && !fs.existsSync(path.join(cwd, 'node_modules'))) {
    logError('no "node_modules" directory exists. Did you run "npm install" first?');
    return;
  }

  const allInstallSpecifiers = new Set(installTargets.map(dep => dep.specifier).sort());
  const installEntrypoints = {};
  const assetEntrypoints = {};
  const importMap = {
    imports: {}
  };
  const installTargetsMap = {};
  const skipFailures = !isExplicit;

  for (const installSpecifier of allInstallSpecifiers) {
    const targetName = getWebDependencyName(installSpecifier);

    if (lockfile && lockfile.imports[installSpecifier]) {
      installEntrypoints[targetName] = lockfile.imports[installSpecifier];
      importMap.imports[installSpecifier] = `./${targetName}.js`;
      installResults.push([targetName, 'SUCCESS']);
      logUpdate(formatInstallResults(skipFailures));
      continue;
    }

    try {
      const {
        type: targetType,
        loc: targetLoc
      } = resolveWebDependency(installSpecifier, isExplicit);

      if (targetType === 'JS') {
        const hashQs = useHash ? `?rev=${await generateHashFromFile(targetLoc)}` : '';
        installEntrypoints[targetName] = targetLoc;
        importMap.imports[installSpecifier] = `./${targetName}.js${hashQs}`;
        installTargetsMap[targetLoc] = installTargets.filter(t => installSpecifier === t.specifier);
        installResults.push([installSpecifier, 'SUCCESS']);
      } else if (targetType === 'ASSET') {
        assetEntrypoints[targetName] = targetLoc;
        installResults.push([installSpecifier, 'ASSET']);
      }

      logUpdate(formatInstallResults(skipFailures));
    } catch (err) {
      installResults.push([installSpecifier, 'FAIL']);
      logUpdate(formatInstallResults(skipFailures));

      if (skipFailures && !ALWAYS_SHOW_ERRORS.has(installSpecifier)) {
        continue;
      } // An error occurred! Log it.


      logError(err.message || err);

      if (err.hint) {
        // Note: Wait 1ms to guarantee a log message after the spinner
        setTimeout(() => console.log(err.hint), 1);
      }

      return false;
    }
  }

  if (Object.keys(installEntrypoints).length === 0 && Object.keys(assetEntrypoints).length === 0) {
    logError(`No ESM dependencies found!`);
    console.log(chalk.dim(`  At least one dependency must have an ESM "module" entrypoint. You can find modern, web-ready packages at ${chalk.underline('https://www.pika.dev')}`));
    return false;
  }

  const inputOptions = {
    input: installEntrypoints,
    external: externalPackages,
    treeshake: {
      moduleSideEffects: 'no-external'
    },
    plugins: [!isStrict && rollupPluginReplace(getRollupReplaceKeys(env, isOptimized)), rollupPluginEntrypointAlias({
      cwd
    }), source === 'pika' && rollupPluginDependencyCache({
      log: url => logUpdate(chalk.dim(url))
    }), rollupPluginNodeResolve({
      mainFields: ['browser:module', 'module', 'browser', !isStrict && 'main'].filter(isTruthy),
      modulesOnly: isStrict,
      extensions: ['.mjs', '.cjs', '.js', '.json'],
      // whether to prefer built-in modules (e.g. `fs`, `path`) or local ones with the same names
      preferBuiltins: false,
      dedupe
    }), !isStrict && rollupPluginJson({
      preferConst: true,
      indent: '  ',
      compact: isOptimized,
      namedExports: true
    }), !isStrict && rollupPluginCommonjs({
      extensions: ['.js', '.cjs'],
      namedExports: knownNamedExports
    }), !!isBabel && rollupPluginBabel({
      compact: false,
      babelrc: false,
      configFile: false,
      presets: [[babelPresetEnv, {
        modules: false,
        targets: hasBrowserlistConfig ? undefined : '>0.75%, not ie 11, not UCAndroid >0, not OperaMini all'
      }]]
    }), !!isOptimized && rollupPluginTreeshakeInputs(installTargets), !!isOptimized && rollupPluginTerser.terser(), !!withStats && rollupPluginDependencyStats(info => dependencyStats = info), ...userDefinedRollup.plugins],

    onwarn(warning, warn) {
      if (warning.code === 'UNRESOLVED_IMPORT') {
        logError(`'${warning.source}' is imported by '${warning.importer}', but could not be resolved.`);

        if (isNodeBuiltin(warning.source)) {
          console.log(chalk.dim(`  '${warning.source}' is a Node.js builtin module that won't exist in the browser.`));
          console.log(chalk.dim(`  Find a more web-friendly alternative, or add the "rollup-plugin-node-polyfills" plugin to your Snowpack config file.`));
        } else {
          console.log(chalk.dim(`  Make sure that the package is installed and that the file exists.`));
        }

        return;
      }

      warn(warning);
    }

  };
  const outputOptions = {
    dir: destLoc,
    format: 'esm',
    sourcemap: sourceMap !== null && sourceMap !== void 0 ? sourceMap : isOptimized,
    exports: 'named',
    chunkFileNames: 'common/[name]-[hash].js'
  };

  if (Object.keys(installEntrypoints).length > 0) {
    try {
      const packageBundle = await rollup.rollup(inputOptions);
      logUpdate(formatInstallResults(skipFailures));
      await packageBundle.write(outputOptions);
    } catch (err) {
      const {
        loc
      } = err;

      if (!loc || !loc.file) {
        throw err;
      } // NOTE: Rollup will fail instantly on error. Because of that, we can
      // only report one error at a time. `err.watchFiles` also exists, but
      // for now `err.loc.file` has all the information that we need.


      const failedExtension = path.extname(loc.file);
      const suggestion = MISSING_PLUGIN_SUGGESTIONS[failedExtension];

      if (!suggestion) {
        throw err;
      } // Display posix-style on all environments, mainly to help with CI :)


      const fileName = loc.file.replace(cwd + path.sep, '').replace(/\\/g, '/');
      logError(`${chalk.bold('snowpack')} could not import ${fileName}. ${suggestion}`);
      return;
    }
  }

  if (nomodule) {
    const nomoduleStart = Date.now();

    function rollupResolutionHelper() {
      return {
        name: 'rename-import-plugin',

        resolveId(source) {
          // resolve from import map
          if (importMap.imports[source]) {
            return importMap.imports[source];
          } // resolve web_modules


          if (source.includes('/web_modules/')) {
            const suffix = source.split('/web_modules/')[1];
            return {
              id: path.join(destLoc, suffix)
            };
          } // null means try to resolve as-is


          return null;
        }

      };
    }

    try {
      // Strip the replace plugin from the set of plugins that we use.
      // Since we've already run it once it can cause trouble on a second run.
      console.assert(inputOptions.plugins[0].name === 'replace');
      const noModulePlugins = inputOptions.plugins.slice(1);
      const noModuleBundle = await rollup.rollup({
        input: path.resolve(cwd, nomodule),
        inlineDynamicImports: true,
        plugins: [...noModulePlugins, rollupResolutionHelper()]
      });
      await noModuleBundle.write({
        file: path.resolve(destLoc, nomoduleOutput),
        format: 'iife',
        name: 'App'
      });
      const nomoduleEnd = Date.now() - nomoduleStart;
      spinner.info(`${chalk.bold('snowpack')} bundled your application for legacy browsers: ${nomoduleOutput} ${chalk.dim(`[${(nomoduleEnd / 1000).toFixed(2)}s]`)}`);
    } catch (err) {
      spinner.warn(`${chalk.bold('snowpack')} encountered an error bundling for legacy browsers: ${err.message}`);
    }
  }

  await writeLockfile(path.join(destLoc, 'import-map.json'), importMap);
  Object.entries(assetEntrypoints).forEach(([assetName, assetLoc]) => {
    mkdirp.sync(path.dirname(`${destLoc}/${assetName}`));
    fs.copyFileSync(assetLoc, `${destLoc}/${assetName}`);
  });
  return true;
}
async function cli(args) {
  // parse CLI flags
  const cliFlags = yargs(args, {
    array: ['env', 'exclude', 'externalPackage']
  }); // if printing help, stop here

  if (cliFlags.help) {
    printHelp();
    process.exit(0);
  }

  if (cliFlags.reload) {
    console.log(`${chalk.yellow('ℹ')} clearing CDN cache...`);
    await clearCache();
  } // Load the current package manifest


  let pkgManifest;

  try {
    pkgManifest = require(path.join(cwd, 'package.json'));
  } catch (err) {
    console.log(chalk.red('[ERROR] package.json required but no file was found.'));
    process.exit(0);
  } // load config


  const {
    config,
    errors
  } = loadConfig(cliFlags, pkgManifest); // handle config errors (if any)

  if (Array.isArray(errors) && errors.length) {
    errors.forEach(logError);
    process.exit(0);
  }

  if (cliFlags.source || config.source === 'pika') {
    console.log(`${chalk.yellow('ℹ')} "source: pika" mode enabled. Behavior is still experimental and may change before the next major version...`);
    await clearCache();
  } // load lockfile


  let lockfile = await readLockfile(cwd);
  let newLockfile = null;
  const {
    installOptions: {
      clean,
      dest,
      exclude,
      include
    },
    entrypoints: configEntrypoints,
    source,
    webDependencies
  } = config;
  const implicitDependencies = [...Object.keys(pkgManifest.peerDependencies || {}), ...Object.keys(pkgManifest.dependencies || {})];
  const hasBrowserlistConfig = !!pkgManifest.browserslist || !!process.env.BROWSERSLIST || fs.existsSync(path.join(cwd, '.browserslistrc')) || fs.existsSync(path.join(cwd, 'browserslist'));
  let isExplicit = false;
  const installTargets = [];

  if (configEntrypoints) {
    isExplicit = true;
    installTargets.push(...scanDepList(configEntrypoints, cwd));
  }

  if (webDependencies) {
    isExplicit = true;
    installTargets.push(...scanDepList(Object.keys(webDependencies), cwd));
  }

  if (include) {
    isExplicit = true;
    installTargets.push(...(await scanImports({
      include,
      exclude
    })));
  }

  if (!isExplicit) {
    installTargets.push(...scanDepList(implicitDependencies, cwd));
  }

  if (installTargets.length === 0) {
    logError('Nothing to install.');
    return;
  }

  spinner.start();
  const startTime = Date.now();

  if (source === 'pika') {
    newLockfile = await resolveTargetsFromRemoteCDN(installTargets, lockfile, pkgManifest, config).catch(err => {
      logError(err.message || err);
      process.exit(1);
    });
  }

  if (clean) {
    rimraf.sync(dest);
  }

  await mkdirp(dest);
  const finalResult = await install(installTargets, {
    hasBrowserlistConfig,
    isExplicit,
    lockfile: newLockfile
  }, config).catch(err => {
    if (err.loc) {
      console.log('\n' + chalk.red.bold(`✘ ${err.loc.file}`));
    }

    if (err.url) {
      console.log(chalk.dim(`👉 ${err.url}`));
    }

    throw err;
  });

  if (finalResult) {
    spinner.succeed(chalk.bold(`snowpack`) + ` installed: ` + formatInstallResults(!isExplicit) + '.' + chalk.dim(` [${((Date.now() - startTime) / 1000).toFixed(2)}s]`));

    if (!!dependencyStats) {
      console.log(printStats(dependencyStats));
    }
  }

  if (newLockfile) {
    await writeLockfile(path.join(cwd, 'snowpack.lock.json'), newLockfile);
  } // If an error happened, set the exit code so that programmatic usage of the CLI knows.
  // We were seeing race conditions here, so add a little buffer.


  if (spinnerHasError) {
    setTimeout(() => {
      spinner.warn(chalk(`Finished with warnings.`));
      process.exitCode = 1;
    }, 20);
  }
}

exports.cli = cli;
exports.install = install;
//# sourceMappingURL=index.js.map
