/*
 * Copyright 2016 Red Hat, Inc. and/or its affiliates
 * and other contributors as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.keycloak.services.scheduled;

import org.jboss.logging.Logger;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.timer.ScheduledTask;
import org.keycloak.timer.TaskRunner;

/**
 * @author <a href="mailto:sthorger@redhat.com">Stian Thorgersen</a>
 */
public class ScheduledTaskRunner implements TaskRunner {

    private static final Logger logger = Logger.getLogger(ScheduledTaskRunner.class);

    protected final KeycloakSessionFactory sessionFactory;

    protected final ScheduledTask task;

    protected final int transactionLimit;

    public ScheduledTaskRunner(KeycloakSessionFactory sessionFactory, ScheduledTask task) {
        this(sessionFactory, task, 0);
    }

    public ScheduledTaskRunner(KeycloakSessionFactory sessionFactory, ScheduledTask task, int transactionLimit) {
        this.sessionFactory = sessionFactory;
        this.task = task;
        this.transactionLimit = transactionLimit;
    }

    @Override
    public void run() {
        try {
            KeycloakModelUtils.runJobInTransaction(sessionFactory, session -> {
                try {
                    if (transactionLimit != 0) {
                        KeycloakModelUtils.setTransactionLimit(sessionFactory, transactionLimit);
                    }

                    runTask(session);
                } finally {
                    if (transactionLimit != 0) {
                        KeycloakModelUtils.setTransactionLimit(sessionFactory, 0);
                    }
                }
            });
        } catch (Throwable t) {
            logger.errorf(t, "Failed to run scheduled task %s", task.getTaskName());
        }
    }

    protected void runTask(KeycloakSession session) {
        task.run(session);

        logger.debugf("Executed scheduled task %s", task.getTaskName());
    }

    @Override
    public ScheduledTask getTask() {
        return task;
    }
}
