/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.managers;

import java.net.URI;
import java.security.PublicKey;
import java.util.HashSet;
import java.util.LinkedList;
import javax.ws.rs.core.Cookie;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.UriInfo;
import org.jboss.logging.Logger;
import org.keycloak.ClientConnection;
import org.keycloak.RSATokenVerifier;
import org.keycloak.VerificationException;
import org.keycloak.jose.jws.JWSBuilder;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RealmModel;
import org.keycloak.models.RequiredCredentialModel;
import org.keycloak.models.UserCredentialModel;
import org.keycloak.models.UserModel;
import org.keycloak.models.UserSessionModel;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.representations.AccessToken;
import org.keycloak.services.managers.BruteForceProtector;
import org.keycloak.services.managers.ResourceAdminManager;
import org.keycloak.services.resources.RealmsResource;
import org.keycloak.services.util.CookieHelper;
import org.keycloak.util.Time;

public class AuthenticationManager {
    protected static Logger logger = Logger.getLogger(AuthenticationManager.class);
    public static final String FORM_USERNAME = "username";
    public static final String KEYCLOAK_IDENTITY_COOKIE = "KEYCLOAK_IDENTITY";
    public static final String KEYCLOAK_SESSION_COOKIE = "KEYCLOAK_SESSION";
    public static final String KEYCLOAK_REMEMBER_ME = "KEYCLOAK_REMEMBER_ME";
    protected BruteForceProtector protector;

    public AuthenticationManager() {
    }

    public AuthenticationManager(BruteForceProtector protector) {
        this.protector = protector;
    }

    public static boolean isSessionValid(RealmModel realm, UserSessionModel userSession) {
        boolean valid;
        if (userSession == null) {
            logger.info((Object)"userSession was null");
            return false;
        }
        int currentTime = Time.currentTime();
        int max = userSession.getStarted() + realm.getSsoSessionMaxLifespan();
        boolean bl = valid = userSession != null && userSession.getLastSessionRefresh() + realm.getSsoSessionIdleTimeout() > currentTime && max > currentTime;
        if (!valid) {
            logger.info((Object)("userSession.getLastSessionRefresh(): " + userSession.getLastSessionRefresh()));
            logger.info((Object)("realm.getSsoSessionIdleTimeout(): " + realm.getSsoSessionIdleTimeout()));
            logger.info((Object)("currentTime: " + currentTime));
            logger.info((Object)("max: " + max));
        }
        return valid;
    }

    public static void logout(KeycloakSession session, RealmModel realm, UserSessionModel userSession, UriInfo uriInfo, ClientConnection connection) {
        if (userSession == null) {
            return;
        }
        UserModel user = userSession.getUser();
        logger.infov("Logging out: {0} ({1})", (Object)user.getUsername(), (Object)userSession.getId());
        session.sessions().removeUserSession(realm, userSession);
        AuthenticationManager.expireIdentityCookie(realm, uriInfo, connection);
        AuthenticationManager.expireRememberMeCookie(realm, uriInfo, connection);
        new ResourceAdminManager().logoutUser(uriInfo.getRequestUri(), realm, user.getId(), userSession.getId());
    }

    public AccessToken createIdentityToken(RealmModel realm, UserModel user, UserSessionModel session) {
        logger.info((Object)"createIdentityToken");
        AccessToken token = new AccessToken();
        token.id(KeycloakModelUtils.generateId());
        token.issuedNow();
        token.subject(user.getId());
        token.audience(realm.getName());
        if (session != null) {
            token.setSessionState(session.getId());
        }
        if (realm.getSsoSessionIdleTimeout() > 0) {
            token.expiration(Time.currentTime() + realm.getSsoSessionIdleTimeout());
        }
        return token;
    }

    public void createLoginCookie(RealmModel realm, UserModel user, UserSessionModel session, UriInfo uriInfo, ClientConnection connection) {
        logger.info((Object)"createLoginCookie");
        String cookiePath = AuthenticationManager.getIdentityCookiePath(realm, uriInfo);
        AccessToken identityToken = this.createIdentityToken(realm, user, session);
        String encoded = this.encodeToken(realm, identityToken);
        boolean secureOnly = realm.getSslRequired().isRequired(connection);
        logger.debugv("creatingLoginCookie - name: {0} path: {1}", (Object)KEYCLOAK_IDENTITY_COOKIE, (Object)cookiePath);
        int maxAge = -1;
        if (session.isRememberMe()) {
            maxAge = realm.getSsoSessionIdleTimeout();
            logger.info((Object)("createLoginCookie maxAge: " + maxAge));
        }
        CookieHelper.addCookie(KEYCLOAK_IDENTITY_COOKIE, encoded, cookiePath, null, null, maxAge, secureOnly, true);
        String sessionCookieValue = realm.getName() + "/" + user.getId();
        if (session != null) {
            sessionCookieValue = sessionCookieValue + "/" + session.getId();
        }
        CookieHelper.addCookie(KEYCLOAK_SESSION_COOKIE, sessionCookieValue, cookiePath, null, null, realm.getSsoSessionMaxLifespan(), secureOnly, false);
    }

    public void createRememberMeCookie(RealmModel realm, UriInfo uriInfo, ClientConnection connection) {
        String path = AuthenticationManager.getIdentityCookiePath(realm, uriInfo);
        boolean secureOnly = realm.getSslRequired().isRequired(connection);
        CookieHelper.addCookie(KEYCLOAK_REMEMBER_ME, "true", path, null, null, realm.getSsoSessionIdleTimeout(), secureOnly, true);
    }

    protected String encodeToken(RealmModel realm, Object token) {
        String encodedToken = new JWSBuilder().jsonContent(token).rsa256(realm.getPrivateKey());
        return encodedToken;
    }

    public static void expireIdentityCookie(RealmModel realm, UriInfo uriInfo, ClientConnection connection) {
        logger.debug((Object)"Expiring identity cookie");
        String path = AuthenticationManager.getIdentityCookiePath(realm, uriInfo);
        AuthenticationManager.expireCookie(realm, KEYCLOAK_IDENTITY_COOKIE, path, true, connection);
        AuthenticationManager.expireCookie(realm, KEYCLOAK_SESSION_COOKIE, path, false, connection);
        AuthenticationManager.expireRememberMeCookie(realm, uriInfo, connection);
    }

    public static void expireRememberMeCookie(RealmModel realm, UriInfo uriInfo, ClientConnection connection) {
        logger.debug((Object)"Expiring remember me cookie");
        String path = AuthenticationManager.getIdentityCookiePath(realm, uriInfo);
        String cookieName = KEYCLOAK_REMEMBER_ME;
        AuthenticationManager.expireCookie(realm, cookieName, path, true, connection);
    }

    protected static String getIdentityCookiePath(RealmModel realm, UriInfo uriInfo) {
        URI uri = RealmsResource.realmBaseUrl(uriInfo).build(new Object[]{realm.getName()});
        return uri.getRawPath();
    }

    public static void expireCookie(RealmModel realm, String cookieName, String path, boolean httpOnly, ClientConnection connection) {
        logger.debugv("Expiring cookie: {0} path: {1}", (Object)cookieName, (Object)path);
        boolean secureOnly = realm.getSslRequired().isRequired(connection);
        CookieHelper.addCookie(cookieName, "", path, null, "Expiring cookie", 0, secureOnly, httpOnly);
    }

    public AuthResult authenticateIdentityCookie(KeycloakSession session, RealmModel realm, UriInfo uriInfo, ClientConnection connection, HttpHeaders headers) {
        return this.authenticateIdentityCookie(session, realm, uriInfo, connection, headers, true);
    }

    public AuthResult authenticateIdentityCookie(KeycloakSession session, RealmModel realm, UriInfo uriInfo, ClientConnection connection, HttpHeaders headers, boolean checkActive) {
        logger.info((Object)"authenticateIdentityCookie");
        Cookie cookie = (Cookie)headers.getCookies().get(KEYCLOAK_IDENTITY_COOKIE);
        if (cookie == null) {
            logger.infov("authenticateCookie could not find cookie: {0}", (Object)KEYCLOAK_IDENTITY_COOKIE);
            return null;
        }
        String tokenString = cookie.getValue();
        AuthResult authResult = this.verifyIdentityToken(session, realm, uriInfo, connection, checkActive, tokenString);
        if (authResult == null) {
            AuthenticationManager.expireIdentityCookie(realm, uriInfo, connection);
            return null;
        }
        authResult.getSession().setLastSessionRefresh(Time.currentTime());
        return authResult;
    }

    protected AuthResult verifyIdentityToken(KeycloakSession session, RealmModel realm, UriInfo uriInfo, ClientConnection connection, boolean checkActive, String tokenString) {
        try {
            UserModel user;
            AccessToken token = RSATokenVerifier.verifyToken((String)tokenString, (PublicKey)realm.getPublicKey(), (String)realm.getName(), (boolean)checkActive);
            logger.info((Object)"identity token verified");
            if (checkActive) {
                logger.info((Object)"Checking if identity token is active");
                if (!token.isActive() || token.getIssuedAt() < realm.getNotBefore()) {
                    logger.info((Object)"identity cookie expired");
                    return null;
                }
                logger.info((Object)("token.isActive() : " + token.isActive()));
                logger.info((Object)("token.issuedAt: " + token.getIssuedAt()));
                logger.info((Object)("real.notbefore: " + realm.getNotBefore()));
            }
            if ((user = session.users().getUserById(token.getSubject(), realm)) == null || !user.isEnabled()) {
                logger.info((Object)"Unknown user in identity token");
                return null;
            }
            UserSessionModel userSession = session.sessions().getUserSession(realm, token.getSessionState());
            if (!AuthenticationManager.isSessionValid(realm, userSession)) {
                if (userSession != null) {
                    AuthenticationManager.logout(session, realm, userSession, uriInfo, connection);
                }
                logger.info((Object)"User session not active");
                return null;
            }
            return new AuthResult(user, userSession, token);
        }
        catch (VerificationException e) {
            logger.info((Object)"Failed to verify identity token", (Throwable)e);
            return null;
        }
    }

    public AuthenticationStatus authenticateForm(KeycloakSession session, ClientConnection clientConnection, RealmModel realm, MultivaluedMap<String, String> formData) {
        String username = (String)formData.getFirst((Object)FORM_USERNAME);
        if (username == null) {
            logger.warn((Object)"Username not provided");
            return AuthenticationStatus.INVALID_USER;
        }
        if (realm.isBruteForceProtected() && this.protector.isTemporarilyDisabled(session, realm, username)) {
            return AuthenticationStatus.ACCOUNT_TEMPORARILY_DISABLED;
        }
        AuthenticationStatus status = this.authenticateInternal(session, realm, formData, username);
        if (realm.isBruteForceProtected()) {
            switch (status) {
                case SUCCESS: {
                    this.protector.successfulLogin(realm, username, clientConnection);
                    break;
                }
                case FAILED: 
                case MISSING_TOTP: 
                case MISSING_PASSWORD: 
                case INVALID_CREDENTIALS: {
                    this.protector.failedLogin(realm, username, clientConnection);
                    break;
                }
                case INVALID_USER: {
                    this.protector.invalidUser(realm, username, clientConnection);
                    break;
                }
            }
        }
        return status;
    }

    protected AuthenticationStatus authenticateInternal(KeycloakSession session, RealmModel realm, MultivaluedMap<String, String> formData, String username) {
        UserModel user = KeycloakModelUtils.findUserByNameOrEmail((KeycloakSession)session, (RealmModel)realm, (String)username);
        if (user == null) {
            logger.warn((Object)("User " + username + " not found"));
            return AuthenticationStatus.INVALID_USER;
        }
        if (!this.checkEnabled(user)) {
            return AuthenticationStatus.ACCOUNT_DISABLED;
        }
        HashSet<String> types = new HashSet<String>();
        for (RequiredCredentialModel credential : realm.getRequiredCredentials()) {
            types.add(credential.getType());
        }
        if (types.contains("password")) {
            LinkedList<UserCredentialModel> credentials = new LinkedList<UserCredentialModel>();
            String password = (String)formData.getFirst((Object)"password");
            if (password == null) {
                logger.warn((Object)"Password not provided");
                return AuthenticationStatus.MISSING_PASSWORD;
            }
            credentials.add(UserCredentialModel.password((String)password));
            if (user.isTotp()) {
                String token = (String)formData.getFirst((Object)"totp");
                if (token == null) {
                    logger.warn((Object)"TOTP token not provided");
                    return AuthenticationStatus.MISSING_TOTP;
                }
                credentials.add(UserCredentialModel.totp((String)token));
            }
            logger.debug((Object)("validating password for user: " + username));
            if (!session.users().validCredentials(realm, user, credentials)) {
                return AuthenticationStatus.INVALID_CREDENTIALS;
            }
            if (!user.getRequiredActions().isEmpty()) {
                return AuthenticationStatus.ACTIONS_REQUIRED;
            }
            return AuthenticationStatus.SUCCESS;
        }
        if (types.contains("secret")) {
            String secret = (String)formData.getFirst((Object)"secret");
            if (secret == null) {
                logger.warn((Object)"Secret not provided");
                return AuthenticationStatus.MISSING_PASSWORD;
            }
            if (!session.users().validCredentials(realm, user, new UserCredentialModel[]{UserCredentialModel.secret((String)secret)})) {
                return AuthenticationStatus.INVALID_CREDENTIALS;
            }
            if (!user.getRequiredActions().isEmpty()) {
                return AuthenticationStatus.ACTIONS_REQUIRED;
            }
            return AuthenticationStatus.SUCCESS;
        }
        logger.warn((Object)"Do not know how to authenticate user");
        return AuthenticationStatus.FAILED;
    }

    private boolean checkEnabled(UserModel user) {
        if (!user.isEnabled()) {
            logger.warn((Object)("AccountProvider is disabled, contact admin. " + user.getUsername()));
            return false;
        }
        return true;
    }

    public class AuthResult {
        private final UserModel user;
        private final UserSessionModel session;
        private final AccessToken token;

        public AuthResult(UserModel user, UserSessionModel session, AccessToken token) {
            this.user = user;
            this.session = session;
            this.token = token;
        }

        public UserSessionModel getSession() {
            return this.session;
        }

        public UserModel getUser() {
            return this.user;
        }

        public AccessToken getToken() {
            return this.token;
        }
    }

    public static enum AuthenticationStatus {
        SUCCESS,
        ACCOUNT_TEMPORARILY_DISABLED,
        ACCOUNT_DISABLED,
        ACTIONS_REQUIRED,
        INVALID_USER,
        INVALID_CREDENTIALS,
        MISSING_PASSWORD,
        MISSING_TOTP,
        FAILED;

    }
}

