/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.managers;

import java.util.List;
import java.util.concurrent.Callable;
import org.jboss.logging.Logger;
import org.keycloak.cluster.ClusterEvent;
import org.keycloak.cluster.ClusterListener;
import org.keycloak.cluster.ClusterProvider;
import org.keycloak.cluster.ExecutionResult;
import org.keycloak.common.util.Time;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.KeycloakSessionTask;
import org.keycloak.models.RealmModel;
import org.keycloak.models.UserFederationProvider;
import org.keycloak.models.UserFederationProviderFactory;
import org.keycloak.models.UserFederationProviderModel;
import org.keycloak.models.UserFederationSyncResult;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.services.ServicesLogger;
import org.keycloak.timer.TimerProvider;

public class UsersSyncManager {
    private static final String FEDERATION_TASK_KEY = "federation";
    private static final Logger logger = Logger.getLogger(UsersSyncManager.class);

    public void bootstrapPeriodic(final KeycloakSessionFactory sessionFactory, final TimerProvider timer) {
        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

            public void run(KeycloakSession session) {
                List realms = session.realms().getRealms();
                for (RealmModel realm : realms) {
                    List federationProviders = realm.getUserFederationProviders();
                    for (UserFederationProviderModel fedProvider : federationProviders) {
                        UsersSyncManager.this.refreshPeriodicSyncForProvider(sessionFactory, timer, fedProvider, realm.getId());
                    }
                }
                ClusterProvider clusterProvider = (ClusterProvider)session.getProvider(ClusterProvider.class);
                clusterProvider.registerListener(UsersSyncManager.FEDERATION_TASK_KEY, (ClusterListener)new UserFederationClusterListener(sessionFactory));
            }
        });
    }

    public UserFederationSyncResult syncAllUsers(final KeycloakSessionFactory sessionFactory, final String realmId, final UserFederationProviderModel fedProvider) {
        final Holder holder = new Holder();
        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

            public void run(KeycloakSession session) {
                ClusterProvider clusterProvider = (ClusterProvider)session.getProvider(ClusterProvider.class);
                String taskKey = fedProvider.getId() + "::sync";
                int timeout = Math.max(30, fedProvider.getFullSyncPeriod());
                holder.result = clusterProvider.executeIfNotExecuted(taskKey, timeout, (Callable)new Callable<UserFederationSyncResult>(){

                    @Override
                    public UserFederationSyncResult call() throws Exception {
                        UserFederationProviderFactory fedProviderFactory = (UserFederationProviderFactory)sessionFactory.getProviderFactory(UserFederationProvider.class, fedProvider.getProviderName());
                        UsersSyncManager.this.updateLastSyncInterval(sessionFactory, fedProvider, realmId);
                        return fedProviderFactory.syncAllUsers(sessionFactory, realmId, fedProvider);
                    }
                });
            }
        });
        if (holder.result == null || !holder.result.isExecuted()) {
            logger.debugf("syncAllUsers for federation provider %s was ignored as it's already in progress", (Object)fedProvider.getDisplayName());
            return UserFederationSyncResult.ignored();
        }
        return (UserFederationSyncResult)holder.result.getResult();
    }

    public UserFederationSyncResult syncChangedUsers(final KeycloakSessionFactory sessionFactory, final String realmId, final UserFederationProviderModel fedProvider) {
        final Holder holder = new Holder();
        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

            public void run(KeycloakSession session) {
                ClusterProvider clusterProvider = (ClusterProvider)session.getProvider(ClusterProvider.class);
                String taskKey = fedProvider.getId() + "::sync";
                int timeout = Math.max(30, fedProvider.getChangedSyncPeriod());
                holder.result = clusterProvider.executeIfNotExecuted(taskKey, timeout, (Callable)new Callable<UserFederationSyncResult>(){

                    @Override
                    public UserFederationSyncResult call() throws Exception {
                        UserFederationProviderFactory fedProviderFactory = (UserFederationProviderFactory)sessionFactory.getProviderFactory(UserFederationProvider.class, fedProvider.getProviderName());
                        int oldLastSync = fedProvider.getLastSync();
                        UsersSyncManager.this.updateLastSyncInterval(sessionFactory, fedProvider, realmId);
                        return fedProviderFactory.syncChangedUsers(sessionFactory, realmId, fedProvider, Time.toDate((int)oldLastSync));
                    }
                });
            }
        });
        if (holder.result == null || !holder.result.isExecuted()) {
            logger.debugf("syncChangedUsers for federation provider %s was ignored as it's already in progress", (Object)fedProvider.getDisplayName());
            return UserFederationSyncResult.ignored();
        }
        return (UserFederationSyncResult)holder.result.getResult();
    }

    public void notifyToRefreshPeriodicSync(KeycloakSession session, RealmModel realm, UserFederationProviderModel federationProvider, boolean removed) {
        FederationProviderClusterEvent event = FederationProviderClusterEvent.createEvent(removed, realm.getId(), federationProvider);
        ((ClusterProvider)session.getProvider(ClusterProvider.class)).notify(FEDERATION_TASK_KEY, (ClusterEvent)event, false);
    }

    protected void refreshPeriodicSyncForProvider(final KeycloakSessionFactory sessionFactory, TimerProvider timer, final UserFederationProviderModel fedProvider, final String realmId) {
        logger.debugf("Going to refresh periodic sync for provider '%s' . Full sync period: %d , changed users sync period: %d", (Object)fedProvider.getDisplayName(), (Object)fedProvider.getFullSyncPeriod(), (Object)fedProvider.getChangedSyncPeriod());
        if (fedProvider.getFullSyncPeriod() > 0) {
            timer.schedule(new Runnable(){

                @Override
                public void run() {
                    try {
                        boolean shouldPerformSync = UsersSyncManager.this.shouldPerformNewPeriodicSync(fedProvider.getLastSync(), fedProvider.getChangedSyncPeriod());
                        if (shouldPerformSync) {
                            UsersSyncManager.this.syncAllUsers(sessionFactory, realmId, fedProvider);
                        } else {
                            logger.debugf("Ignored periodic full sync with federation provider %s due small time since last sync", (Object)fedProvider.getDisplayName());
                        }
                    }
                    catch (Throwable t) {
                        ServicesLogger.LOGGER.errorDuringFullUserSync(t);
                    }
                }
            }, (long)(fedProvider.getFullSyncPeriod() * 1000), fedProvider.getId() + "-FULL");
        } else {
            timer.cancelTask(fedProvider.getId() + "-FULL");
        }
        if (fedProvider.getChangedSyncPeriod() > 0) {
            timer.schedule(new Runnable(){

                @Override
                public void run() {
                    try {
                        boolean shouldPerformSync = UsersSyncManager.this.shouldPerformNewPeriodicSync(fedProvider.getLastSync(), fedProvider.getChangedSyncPeriod());
                        if (shouldPerformSync) {
                            UsersSyncManager.this.syncChangedUsers(sessionFactory, realmId, fedProvider);
                        } else {
                            logger.debugf("Ignored periodic changed-users sync with federation provider %s due small time since last sync", (Object)fedProvider.getDisplayName());
                        }
                    }
                    catch (Throwable t) {
                        ServicesLogger.LOGGER.errorDuringChangedUserSync(t);
                    }
                }
            }, (long)(fedProvider.getChangedSyncPeriod() * 1000), fedProvider.getId() + "-CHANGED");
        } else {
            timer.cancelTask(fedProvider.getId() + "-CHANGED");
        }
    }

    private boolean shouldPerformNewPeriodicSync(int lastSyncTime, int period) {
        if (lastSyncTime <= 0) {
            return true;
        }
        int currentTime = Time.currentTime();
        int timeSinceLastSync = currentTime - lastSyncTime;
        return timeSinceLastSync * 2 > period;
    }

    protected void removePeriodicSyncForProvider(TimerProvider timer, UserFederationProviderModel fedProvider) {
        logger.debugf("Removing periodic sync for provider %s", (Object)fedProvider.getDisplayName());
        timer.cancelTask(fedProvider.getId() + "-FULL");
        timer.cancelTask(fedProvider.getId() + "-CHANGED");
    }

    private void updateLastSyncInterval(KeycloakSessionFactory sessionFactory, final UserFederationProviderModel fedProvider, final String realmId) {
        KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

            public void run(KeycloakSession session) {
                RealmModel persistentRealm = session.realms().getRealm(realmId);
                List persistentFedProviders = persistentRealm.getUserFederationProviders();
                for (UserFederationProviderModel persistentFedProvider : persistentFedProviders) {
                    if (!fedProvider.getId().equals(persistentFedProvider.getId())) continue;
                    int lastSync = Time.currentTime();
                    persistentFedProvider.setLastSync(lastSync);
                    persistentRealm.updateUserFederationProvider(persistentFedProvider);
                    fedProvider.setLastSync(lastSync);
                }
            }
        });
    }

    public static class FederationProviderClusterEvent
    implements ClusterEvent {
        private boolean removed;
        private String realmId;
        private UserFederationProviderModel federationProvider;

        public boolean isRemoved() {
            return this.removed;
        }

        public void setRemoved(boolean removed) {
            this.removed = removed;
        }

        public String getRealmId() {
            return this.realmId;
        }

        public void setRealmId(String realmId) {
            this.realmId = realmId;
        }

        public UserFederationProviderModel getFederationProvider() {
            return this.federationProvider;
        }

        public void setFederationProvider(UserFederationProviderModel federationProvider) {
            this.federationProvider = federationProvider;
        }

        public static FederationProviderClusterEvent createEvent(boolean removed, String realmId, UserFederationProviderModel fedProvider) {
            FederationProviderClusterEvent notification = new FederationProviderClusterEvent();
            notification.setRemoved(removed);
            notification.setRealmId(realmId);
            notification.setFederationProvider(fedProvider);
            return notification;
        }
    }

    private class UserFederationClusterListener
    implements ClusterListener {
        private final KeycloakSessionFactory sessionFactory;

        public UserFederationClusterListener(KeycloakSessionFactory sessionFactory) {
            this.sessionFactory = sessionFactory;
        }

        public void eventReceived(ClusterEvent event) {
            final FederationProviderClusterEvent fedEvent = (FederationProviderClusterEvent)event;
            KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)this.sessionFactory, (KeycloakSessionTask)new KeycloakSessionTask(){

                public void run(KeycloakSession session) {
                    TimerProvider timer = (TimerProvider)session.getProvider(TimerProvider.class);
                    if (fedEvent.isRemoved()) {
                        UsersSyncManager.this.removePeriodicSyncForProvider(timer, fedEvent.getFederationProvider());
                    } else {
                        UsersSyncManager.this.refreshPeriodicSyncForProvider(UserFederationClusterListener.this.sessionFactory, timer, fedEvent.getFederationProvider(), fedEvent.getRealmId());
                    }
                }
            });
        }
    }

    private class Holder {
        ExecutionResult<UserFederationSyncResult> result;

        private Holder() {
        }
    }
}

