/*
 * Copyright 2021 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kie.kogito.examples;

import java.net.URI;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import com.fasterxml.jackson.databind.jsonschema.JsonSchema;
import org.jbpm.util.JsonSchemaUtil;
import org.kie.kogito.Application;
import org.kie.kogito.process.Process;
import org.kie.kogito.process.ProcessInstance;
import org.kie.kogito.process.ProcessInstanceReadMode;
import org.kie.kogito.process.impl.Sig;
import org.kie.kogito.process.ProcessInstanceExecutionException;
import org.kie.kogito.process.WorkItem;
import org.kie.kogito.process.workitem.Attachment;
import org.kie.kogito.process.workitem.AttachmentInfo;
import org.kie.kogito.process.workitem.Comment;
import org.kie.kogito.process.workitem.Policies;
import org.kie.kogito.process.workitem.TaskModel;
import org.kie.kogito.services.uow.UnitOfWorkExecutor;
import org.jbpm.process.instance.impl.humantask.HumanTaskHelper;
import org.jbpm.process.instance.impl.humantask.HumanTaskTransition;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.http.ResponseEntity;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;
import org.kie.kogito.examples.FlightsModelOutput;

@RestController
@RequestMapping("/flights")
@org.springframework.stereotype.Component()
public class FlightsResource {

    @org.springframework.beans.factory.annotation.Autowired()
    @org.springframework.beans.factory.annotation.Qualifier("flights")
    Process<FlightsModel> process;

    @org.springframework.beans.factory.annotation.Autowired()
    Application application;

    @PostMapping(produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> createResource_flights(@RequestHeader HttpHeaders httpHeaders, @RequestParam(value = "businessKey", required = false) String businessKey, @RequestBody(required = false) @javax.validation.Valid() @javax.validation.constraints.NotNull() FlightsModelInput resource, UriComponentsBuilder uriComponentsBuilder) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> {
            FlightsModelInput inputModel = resource != null ? resource : new FlightsModelInput();
            ProcessInstance<FlightsModel> pi = process.createInstance(businessKey, inputModel.toModel());
            List<String> startFromNode = httpHeaders.get("X-KOGITO-StartFromNode");
            if (startFromNode != null && !startFromNode.isEmpty()) {
                pi.startFrom(startFromNode.get(0));
            } else {
                pi.start();
            }
            UriComponents uriComponents = uriComponentsBuilder.path("/flights/{id}").buildAndExpand(pi.id());
            URI location = uriComponents.toUri();
            return ResponseEntity.created(location).body(pi.checkError().variables().toOutput());
        });
    }

    @GetMapping(produces = MediaType.APPLICATION_JSON_VALUE)
    public List<FlightsModelOutput> getResources_flights() {
        return process.instances().values().stream().map(pi -> pi.variables().toOutput()).collect(Collectors.toList());
    }

    @GetMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> getResource_flights(@PathVariable("id") String id) {
        return process.instances().findById(id, ProcessInstanceReadMode.READ_ONLY).map(m -> ResponseEntity.ok(m.variables().toOutput())).orElseGet(() -> ResponseEntity.notFound().build());
    }

    @DeleteMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> deleteResource_flights(@PathVariable("id") final String id) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.abort();
            return pi.checkError().variables().toOutput();
        }).map(m -> ResponseEntity.ok(m))).orElseGet(() -> ResponseEntity.notFound().build());
    }

    @PutMapping(value = "/{id}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> updateModel_flights(@PathVariable("id") String id, @RequestBody(required = false) FlightsModel resource) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> pi.updateVariables(resource).toOutput()).map(m -> ResponseEntity.ok(m))).orElseGet(() -> ResponseEntity.notFound().build());
    }

    @GetMapping(value = "/{id}/tasks", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<List<TaskModel>> getTasks_flights(@PathVariable("id") String id, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return process.instances().findById(id, ProcessInstanceReadMode.READ_ONLY).map(pi -> pi.workItems(Policies.of(user, groups)).stream().map(org.kie.kogito.examples.Flights_TaskModelFactory::from).collect(Collectors.toList())).map(m -> ResponseEntity.ok(m)).orElseGet(() -> ResponseEntity.notFound().build());
    }

    @PostMapping(value = "/{id}/solvingTerminated", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.kie.kogito.examples.FlightsModelOutput> signal_0(@PathVariable("id") final String id, @RequestBody(required = false) final org.kie.kogito.examples.domain.Flight data) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> {
            return process.instances().findById(id).map(pi -> {
                pi.send(Sig.of("solvingTerminated", data));
                return ResponseEntity.ok(pi.checkError().variables().toOutput());
            }).orElseGet(() -> ResponseEntity.notFound().build());
        });
    }

    @PostMapping(value = "/{id}/newPassengerRequest", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.kie.kogito.examples.FlightsModelOutput> signal_1(@PathVariable("id") final String id, @RequestBody(required = false) final org.kie.kogito.examples.domain.PassengerDTO data) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> {
            return process.instances().findById(id).map(pi -> {
                pi.send(Sig.of("newPassengerRequest", data));
                return ResponseEntity.ok(pi.checkError().variables().toOutput());
            }).orElseGet(() -> ResponseEntity.notFound().build());
        });
    }

    @PostMapping(value = "/{id}/New_Best_Solution", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.kie.kogito.examples.FlightsModelOutput> signal_2(@PathVariable("id") final String id) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> {
            return process.instances().findById(id).map(pi -> {
                pi.send(Sig.of("New Best Solution", null));
                return ResponseEntity.ok(pi.checkError().variables().toOutput());
            }).orElseGet(() -> ResponseEntity.notFound().build());
        });
    }

    @PostMapping(value = "/{id}/newSolution", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.kie.kogito.examples.FlightsModelOutput> signal_3(@PathVariable("id") final String id, @RequestBody(required = false) final org.kie.kogito.examples.domain.Flight data) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> {
            return process.instances().findById(id).map(pi -> {
                pi.send(Sig.of("newSolution", data));
                return ResponseEntity.ok(pi.checkError().variables().toOutput());
            }).orElseGet(() -> ResponseEntity.notFound().build());
        });
    }

    @PostMapping(value = "/{id}/Passenger_Boarding_Request", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.kie.kogito.examples.FlightsModelOutput> signal_4(@PathVariable("id") final String id) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> {
            return process.instances().findById(id).map(pi -> {
                pi.send(Sig.of("Passenger Boarding Request", null));
                return ResponseEntity.ok(pi.checkError().variables().toOutput());
            }).orElseGet(() -> ResponseEntity.notFound().build());
        });
    }

    @PostMapping(value = "/{id}/finalizeSeatAssignment/{taskId}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> taskTransition_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "phase", required = false, defaultValue = "complete") final String phase, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody(required = false) final org.kie.kogito.examples.Flights_6_TaskOutput model) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.transitionWorkItem(taskId, HumanTaskTransition.withModel(phase, model, Policies.of(user, groups)));
            return ResponseEntity.ok(pi.checkError().variables().toOutput());
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PutMapping(value = "/{id}/finalizeSeatAssignment/{taskId}", consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.kie.kogito.examples.Flights_6_TaskOutput> saveTask_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody(required = false) final org.kie.kogito.examples.Flights_6_TaskOutput model) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> ResponseEntity.ok(org.kie.kogito.examples.Flights_6_TaskOutput.fromMap(pi.updateWorkItem(taskId, wi -> HumanTaskHelper.updateContent(wi, model), Policies.of(user, groups))))).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/finalizeSeatAssignment/{taskId}/phases/{phase}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> completeTask_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("phase") final String phase, @RequestParam("user") final String user, @RequestParam("group") final List<String> groups, @RequestBody(required = false) final org.kie.kogito.examples.Flights_6_TaskOutput model) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.transitionWorkItem(taskId, HumanTaskTransition.withModel(phase, model, Policies.of(user, groups)));
            return ResponseEntity.ok(pi.checkError().variables().toOutput());
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/finalizeSeatAssignment/{taskId}/comments", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.TEXT_PLAIN_VALUE)
    public ResponseEntity<Comment> addComment_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody String commentInfo, UriComponentsBuilder uriComponentsBuilder) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            Comment comment = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.addComment(wi, commentInfo, user), Policies.of(user, groups));
            return ResponseEntity.created(uriComponentsBuilder.path("/flights/{id}/finalizeSeatAssignment/{taskId}/comments/{commentId}").buildAndExpand(id, taskId, comment.getId()).toUri()).body(comment);
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PutMapping(value = "/{id}/finalizeSeatAssignment/{taskId}/comments/{commentId}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.TEXT_PLAIN_VALUE)
    public ResponseEntity<Comment> updateComment_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("commentId") final String commentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody String comment) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> ResponseEntity.ok(pi.updateWorkItem(taskId, wi -> HumanTaskHelper.updateComment(wi, commentId, comment, user), Policies.of(user, groups)))).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @DeleteMapping(value = "/{id}/finalizeSeatAssignment/{taskId}/comments/{commentId}")
    public ResponseEntity deleteComment_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("commentId") final String commentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            boolean removed = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.deleteComment(wi, commentId, user), Policies.of(user, groups));
            return removed ? ResponseEntity.ok().build() : ResponseEntity.notFound().build();
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/finalizeSeatAssignment/{taskId}/attachments", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Attachment> addAttachment_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody AttachmentInfo attachmentInfo, UriComponentsBuilder uriComponentsBuilder) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            Attachment attachment = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.addAttachment(wi, attachmentInfo, user), Policies.of(user, groups));
            return ResponseEntity.created(uriComponentsBuilder.path("/flights/{id}/finalizeSeatAssignment/{taskId}/attachments/{attachmentId}").buildAndExpand(id, taskId, attachment.getId()).toUri()).body(attachment);
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PutMapping(value = "/{id}/finalizeSeatAssignment/{taskId}/attachments/{attachmentId}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Attachment> updateAttachment_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("attachmentId") final String attachmentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody AttachmentInfo attachment) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> ResponseEntity.ok(pi.updateWorkItem(taskId, wi -> HumanTaskHelper.updateAttachment(wi, attachmentId, attachment, user), Policies.of(user, groups)))).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @DeleteMapping(value = "/{id}/finalizeSeatAssignment/{taskId}/attachments/{attachmentId}")
    public ResponseEntity deleteAttachment_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("attachmentId") final String attachmentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            boolean removed = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.deleteAttachment(wi, attachmentId, user), Policies.of(user, groups));
            return (removed ? ResponseEntity.ok() : ResponseEntity.notFound()).build();
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @GetMapping(value = "/{id}/finalizeSeatAssignment/{taskId}/attachments/{attachmentId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Attachment> getAttachment_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("attachmentId") final String attachmentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        Optional<ProcessInstance<FlightsModel>> pi = process.instances().findById(id);
        if (pi.isPresent()) {
            Attachment attachment = HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getAttachments().get(attachmentId);
            if (attachment != null) {
                return ResponseEntity.ok(attachment);
            }
        }
        return ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/finalizeSeatAssignment/{taskId}/attachments", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Collection<Attachment>> getAttachments_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user") final String user, @RequestParam(value = "group") final List<String> groups) {
        Optional<ProcessInstance<FlightsModel>> pi = process.instances().findById(id);
        return pi.isPresent() ? ResponseEntity.ok(HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getAttachments().values()) : ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/finalizeSeatAssignment/{taskId}/comments/{commentId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Comment> getComment_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("commentId") final String commentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        Optional<ProcessInstance<FlightsModel>> pi = process.instances().findById(id);
        if (pi.isPresent()) {
            Comment comment = HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getComments().get(commentId);
            if (comment != null) {
                return ResponseEntity.ok(comment);
            }
        }
        return ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/finalizeSeatAssignment/{taskId}/comments", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Collection<Comment>> getComments_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        Optional<ProcessInstance<FlightsModel>> pi = process.instances().findById(id);
        return pi.isPresent() ? ResponseEntity.ok(HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getComments().values()) : ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/finalizeSeatAssignment/{taskId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.kie.kogito.examples.Flights_6_TaskModel> getTask_finalizeSeatAssignment_5(@PathVariable("id") String id, @PathVariable("taskId") String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return process.instances().findById(id).map(pi -> org.kie.kogito.examples.Flights_6_TaskModel.from(pi.workItem(taskId, Policies.of(user, groups)))).map(m -> ResponseEntity.ok(m)).orElseGet(() -> ResponseEntity.notFound().build());
    }

    @GetMapping(value = "finalizeSeatAssignment/schema", produces = MediaType.APPLICATION_JSON_VALUE)
    public Map<String, Object> getSchema_finalizeSeatAssignment_5() {
        return JsonSchemaUtil.load(this.getClass().getClassLoader(), process.id(), "finalizeSeatAssignment");
    }

    @GetMapping(value = "/{id}/finalizeSeatAssignment/{taskId}/schema", produces = MediaType.APPLICATION_JSON_VALUE)
    public Map<String, Object> getSchemaAndPhases_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return JsonSchemaUtil.addPhases(process, application, id, taskId, Policies.of(user, groups), JsonSchemaUtil.load(this.getClass().getClassLoader(), process.id(), "finalizeSeatAssignment"));
    }

    @DeleteMapping(value = "/{id}/finalizeSeatAssignment/{taskId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> abortTask_finalizeSeatAssignment_5(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "phase", required = false, defaultValue = "abort") final String phase, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.transitionWorkItem(taskId, HumanTaskTransition.withoutModel(phase, Policies.of(user, groups)));
            return ResponseEntity.ok(pi.checkError().variables().toOutput());
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/finalizePassengerList/{taskId}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> taskTransition_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "phase", required = false, defaultValue = "complete") final String phase, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody(required = false) final org.kie.kogito.examples.Flights_7_TaskOutput model) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.transitionWorkItem(taskId, HumanTaskTransition.withModel(phase, model, Policies.of(user, groups)));
            return ResponseEntity.ok(pi.checkError().variables().toOutput());
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PutMapping(value = "/{id}/finalizePassengerList/{taskId}", consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.kie.kogito.examples.Flights_7_TaskOutput> saveTask_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody(required = false) final org.kie.kogito.examples.Flights_7_TaskOutput model) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> ResponseEntity.ok(org.kie.kogito.examples.Flights_7_TaskOutput.fromMap(pi.updateWorkItem(taskId, wi -> HumanTaskHelper.updateContent(wi, model), Policies.of(user, groups))))).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/finalizePassengerList/{taskId}/phases/{phase}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> completeTask_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("phase") final String phase, @RequestParam("user") final String user, @RequestParam("group") final List<String> groups, @RequestBody(required = false) final org.kie.kogito.examples.Flights_7_TaskOutput model) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.transitionWorkItem(taskId, HumanTaskTransition.withModel(phase, model, Policies.of(user, groups)));
            return ResponseEntity.ok(pi.checkError().variables().toOutput());
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/finalizePassengerList/{taskId}/comments", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.TEXT_PLAIN_VALUE)
    public ResponseEntity<Comment> addComment_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody String commentInfo, UriComponentsBuilder uriComponentsBuilder) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            Comment comment = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.addComment(wi, commentInfo, user), Policies.of(user, groups));
            return ResponseEntity.created(uriComponentsBuilder.path("/flights/{id}/finalizePassengerList/{taskId}/comments/{commentId}").buildAndExpand(id, taskId, comment.getId()).toUri()).body(comment);
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PutMapping(value = "/{id}/finalizePassengerList/{taskId}/comments/{commentId}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.TEXT_PLAIN_VALUE)
    public ResponseEntity<Comment> updateComment_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("commentId") final String commentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody String comment) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> ResponseEntity.ok(pi.updateWorkItem(taskId, wi -> HumanTaskHelper.updateComment(wi, commentId, comment, user), Policies.of(user, groups)))).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @DeleteMapping(value = "/{id}/finalizePassengerList/{taskId}/comments/{commentId}")
    public ResponseEntity deleteComment_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("commentId") final String commentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            boolean removed = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.deleteComment(wi, commentId, user), Policies.of(user, groups));
            return removed ? ResponseEntity.ok().build() : ResponseEntity.notFound().build();
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/finalizePassengerList/{taskId}/attachments", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Attachment> addAttachment_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody AttachmentInfo attachmentInfo, UriComponentsBuilder uriComponentsBuilder) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            Attachment attachment = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.addAttachment(wi, attachmentInfo, user), Policies.of(user, groups));
            return ResponseEntity.created(uriComponentsBuilder.path("/flights/{id}/finalizePassengerList/{taskId}/attachments/{attachmentId}").buildAndExpand(id, taskId, attachment.getId()).toUri()).body(attachment);
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PutMapping(value = "/{id}/finalizePassengerList/{taskId}/attachments/{attachmentId}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Attachment> updateAttachment_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("attachmentId") final String attachmentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody AttachmentInfo attachment) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> ResponseEntity.ok(pi.updateWorkItem(taskId, wi -> HumanTaskHelper.updateAttachment(wi, attachmentId, attachment, user), Policies.of(user, groups)))).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @DeleteMapping(value = "/{id}/finalizePassengerList/{taskId}/attachments/{attachmentId}")
    public ResponseEntity deleteAttachment_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("attachmentId") final String attachmentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            boolean removed = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.deleteAttachment(wi, attachmentId, user), Policies.of(user, groups));
            return (removed ? ResponseEntity.ok() : ResponseEntity.notFound()).build();
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @GetMapping(value = "/{id}/finalizePassengerList/{taskId}/attachments/{attachmentId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Attachment> getAttachment_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("attachmentId") final String attachmentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        Optional<ProcessInstance<FlightsModel>> pi = process.instances().findById(id);
        if (pi.isPresent()) {
            Attachment attachment = HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getAttachments().get(attachmentId);
            if (attachment != null) {
                return ResponseEntity.ok(attachment);
            }
        }
        return ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/finalizePassengerList/{taskId}/attachments", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Collection<Attachment>> getAttachments_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user") final String user, @RequestParam(value = "group") final List<String> groups) {
        Optional<ProcessInstance<FlightsModel>> pi = process.instances().findById(id);
        return pi.isPresent() ? ResponseEntity.ok(HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getAttachments().values()) : ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/finalizePassengerList/{taskId}/comments/{commentId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Comment> getComment_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("commentId") final String commentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        Optional<ProcessInstance<FlightsModel>> pi = process.instances().findById(id);
        if (pi.isPresent()) {
            Comment comment = HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getComments().get(commentId);
            if (comment != null) {
                return ResponseEntity.ok(comment);
            }
        }
        return ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/finalizePassengerList/{taskId}/comments", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Collection<Comment>> getComments_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        Optional<ProcessInstance<FlightsModel>> pi = process.instances().findById(id);
        return pi.isPresent() ? ResponseEntity.ok(HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getComments().values()) : ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/finalizePassengerList/{taskId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.kie.kogito.examples.Flights_7_TaskModel> getTask_finalizePassengerList_6(@PathVariable("id") String id, @PathVariable("taskId") String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return process.instances().findById(id).map(pi -> org.kie.kogito.examples.Flights_7_TaskModel.from(pi.workItem(taskId, Policies.of(user, groups)))).map(m -> ResponseEntity.ok(m)).orElseGet(() -> ResponseEntity.notFound().build());
    }

    @GetMapping(value = "finalizePassengerList/schema", produces = MediaType.APPLICATION_JSON_VALUE)
    public Map<String, Object> getSchema_finalizePassengerList_6() {
        return JsonSchemaUtil.load(this.getClass().getClassLoader(), process.id(), "finalizePassengerList");
    }

    @GetMapping(value = "/{id}/finalizePassengerList/{taskId}/schema", produces = MediaType.APPLICATION_JSON_VALUE)
    public Map<String, Object> getSchemaAndPhases_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return JsonSchemaUtil.addPhases(process, application, id, taskId, Policies.of(user, groups), JsonSchemaUtil.load(this.getClass().getClassLoader(), process.id(), "finalizePassengerList"));
    }

    @DeleteMapping(value = "/{id}/finalizePassengerList/{taskId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> abortTask_finalizePassengerList_6(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "phase", required = false, defaultValue = "abort") final String phase, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.transitionWorkItem(taskId, HumanTaskTransition.withoutModel(phase, Policies.of(user, groups)));
            return ResponseEntity.ok(pi.checkError().variables().toOutput());
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/approveDenyPassenger/{taskId}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> taskTransition_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "phase", required = false, defaultValue = "complete") final String phase, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody(required = false) final org.kie.kogito.examples.Flights_18_TaskOutput model) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.transitionWorkItem(taskId, HumanTaskTransition.withModel(phase, model, Policies.of(user, groups)));
            return ResponseEntity.ok(pi.checkError().variables().toOutput());
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PutMapping(value = "/{id}/approveDenyPassenger/{taskId}", consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.kie.kogito.examples.Flights_18_TaskOutput> saveTask_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody(required = false) final org.kie.kogito.examples.Flights_18_TaskOutput model) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> ResponseEntity.ok(org.kie.kogito.examples.Flights_18_TaskOutput.fromMap(pi.updateWorkItem(taskId, wi -> HumanTaskHelper.updateContent(wi, model), Policies.of(user, groups))))).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/approveDenyPassenger/{taskId}/phases/{phase}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> completeTask_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("phase") final String phase, @RequestParam("user") final String user, @RequestParam("group") final List<String> groups, @RequestBody(required = false) final org.kie.kogito.examples.Flights_18_TaskOutput model) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.transitionWorkItem(taskId, HumanTaskTransition.withModel(phase, model, Policies.of(user, groups)));
            return ResponseEntity.ok(pi.checkError().variables().toOutput());
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/approveDenyPassenger/{taskId}/comments", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.TEXT_PLAIN_VALUE)
    public ResponseEntity<Comment> addComment_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody String commentInfo, UriComponentsBuilder uriComponentsBuilder) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            Comment comment = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.addComment(wi, commentInfo, user), Policies.of(user, groups));
            return ResponseEntity.created(uriComponentsBuilder.path("/flights/{id}/approveDenyPassenger/{taskId}/comments/{commentId}").buildAndExpand(id, taskId, comment.getId()).toUri()).body(comment);
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PutMapping(value = "/{id}/approveDenyPassenger/{taskId}/comments/{commentId}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.TEXT_PLAIN_VALUE)
    public ResponseEntity<Comment> updateComment_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("commentId") final String commentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody String comment) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> ResponseEntity.ok(pi.updateWorkItem(taskId, wi -> HumanTaskHelper.updateComment(wi, commentId, comment, user), Policies.of(user, groups)))).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @DeleteMapping(value = "/{id}/approveDenyPassenger/{taskId}/comments/{commentId}")
    public ResponseEntity deleteComment_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("commentId") final String commentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            boolean removed = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.deleteComment(wi, commentId, user), Policies.of(user, groups));
            return removed ? ResponseEntity.ok().build() : ResponseEntity.notFound().build();
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PostMapping(value = "/{id}/approveDenyPassenger/{taskId}/attachments", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Attachment> addAttachment_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody AttachmentInfo attachmentInfo, UriComponentsBuilder uriComponentsBuilder) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            Attachment attachment = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.addAttachment(wi, attachmentInfo, user), Policies.of(user, groups));
            return ResponseEntity.created(uriComponentsBuilder.path("/flights/{id}/approveDenyPassenger/{taskId}/attachments/{attachmentId}").buildAndExpand(id, taskId, attachment.getId()).toUri()).body(attachment);
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @PutMapping(value = "/{id}/approveDenyPassenger/{taskId}/attachments/{attachmentId}", produces = MediaType.APPLICATION_JSON_VALUE, consumes = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Attachment> updateAttachment_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("attachmentId") final String attachmentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups, @RequestBody AttachmentInfo attachment) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> ResponseEntity.ok(pi.updateWorkItem(taskId, wi -> HumanTaskHelper.updateAttachment(wi, attachmentId, attachment, user), Policies.of(user, groups)))).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @DeleteMapping(value = "/{id}/approveDenyPassenger/{taskId}/attachments/{attachmentId}")
    public ResponseEntity deleteAttachment_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("attachmentId") final String attachmentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            boolean removed = pi.updateWorkItem(taskId, wi -> HumanTaskHelper.deleteAttachment(wi, attachmentId, user), Policies.of(user, groups));
            return (removed ? ResponseEntity.ok() : ResponseEntity.notFound()).build();
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }

    @GetMapping(value = "/{id}/approveDenyPassenger/{taskId}/attachments/{attachmentId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Attachment> getAttachment_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("attachmentId") final String attachmentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        Optional<ProcessInstance<FlightsModel>> pi = process.instances().findById(id);
        if (pi.isPresent()) {
            Attachment attachment = HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getAttachments().get(attachmentId);
            if (attachment != null) {
                return ResponseEntity.ok(attachment);
            }
        }
        return ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/approveDenyPassenger/{taskId}/attachments", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Collection<Attachment>> getAttachments_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user") final String user, @RequestParam(value = "group") final List<String> groups) {
        Optional<ProcessInstance<FlightsModel>> pi = process.instances().findById(id);
        return pi.isPresent() ? ResponseEntity.ok(HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getAttachments().values()) : ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/approveDenyPassenger/{taskId}/comments/{commentId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Comment> getComment_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @PathVariable("commentId") final String commentId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        Optional<ProcessInstance<FlightsModel>> pi = process.instances().findById(id);
        if (pi.isPresent()) {
            Comment comment = HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getComments().get(commentId);
            if (comment != null) {
                return ResponseEntity.ok(comment);
            }
        }
        return ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/approveDenyPassenger/{taskId}/comments", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Collection<Comment>> getComments_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        Optional<ProcessInstance<FlightsModel>> pi = process.instances().findById(id);
        return pi.isPresent() ? ResponseEntity.ok(HumanTaskHelper.findTask(pi.get(), taskId, Policies.of(user, groups)).getComments().values()) : ResponseEntity.notFound().build();
    }

    @GetMapping(value = "/{id}/approveDenyPassenger/{taskId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<org.kie.kogito.examples.Flights_18_TaskModel> getTask_approveDenyPassenger_7(@PathVariable("id") String id, @PathVariable("taskId") String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return process.instances().findById(id).map(pi -> org.kie.kogito.examples.Flights_18_TaskModel.from(pi.workItem(taskId, Policies.of(user, groups)))).map(m -> ResponseEntity.ok(m)).orElseGet(() -> ResponseEntity.notFound().build());
    }

    @GetMapping(value = "approveDenyPassenger/schema", produces = MediaType.APPLICATION_JSON_VALUE)
    public Map<String, Object> getSchema_approveDenyPassenger_7() {
        return JsonSchemaUtil.load(this.getClass().getClassLoader(), process.id(), "approveDenyPassenger");
    }

    @GetMapping(value = "/{id}/approveDenyPassenger/{taskId}/schema", produces = MediaType.APPLICATION_JSON_VALUE)
    public Map<String, Object> getSchemaAndPhases_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return JsonSchemaUtil.addPhases(process, application, id, taskId, Policies.of(user, groups), JsonSchemaUtil.load(this.getClass().getClassLoader(), process.id(), "approveDenyPassenger"));
    }

    @DeleteMapping(value = "/{id}/approveDenyPassenger/{taskId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<FlightsModelOutput> abortTask_approveDenyPassenger_7(@PathVariable("id") final String id, @PathVariable("taskId") final String taskId, @RequestParam(value = "phase", required = false, defaultValue = "abort") final String phase, @RequestParam(value = "user", required = false) final String user, @RequestParam(value = "group", required = false) final List<String> groups) {
        return UnitOfWorkExecutor.executeInUnitOfWork(application.unitOfWorkManager(), () -> process.instances().findById(id).map(pi -> {
            pi.transitionWorkItem(taskId, HumanTaskTransition.withoutModel(phase, Policies.of(user, groups)));
            return ResponseEntity.ok(pi.checkError().variables().toOutput());
        }).orElseGet(() -> ResponseEntity.notFound().build()));
    }
}
