/*
 * Copyright 2017 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jbpm.workflow.instance.node;

import org.drools.core.spi.ProcessContext;
import org.jbpm.workflow.core.node.MilestoneNode;
import org.kie.api.event.process.ContextAwareEventListener;
import org.kie.api.runtime.process.NodeInstance;

/**
 * Runtime counterpart of a milestone node.
 */
public class MilestoneNodeInstance extends StateBasedNodeInstance {

    private static final long serialVersionUID = 510L;

    protected MilestoneNode getMilestoneNode() {
        return (MilestoneNode) getNode();
    }

    @Override
    public void internalTrigger(final NodeInstance from, String type) {
        super.internalTrigger(from, type);
        // if node instance was cancelled, abort
        if (getNodeInstanceContainer().getNodeInstance(getId()) == null) {
            return;
        }
        if (!org.jbpm.workflow.core.Node.CONNECTION_DEFAULT_TYPE.equals(type)) {
            throw new IllegalArgumentException(
                    "A MilestoneNode only accepts default incoming connections!");
        }
        if (isCompleted()) {
            triggerCompleted();
        } else {
            addCompletionEventListener();
        }
    }

    private boolean isCompleted() {
        ProcessContext context = new ProcessContext(getProcessInstance().getKnowledgeRuntime())
                .setNodeInstance(this);
        return getMilestoneNode().canComplete(context);
    }

    @Override
    public void addEventListeners() {
        super.addEventListeners();
        addCompletionEventListener();
    }

    private void addCompletionEventListener() {
        getProcessInstance().getKnowledgeRuntime().getProcessRuntime().addEventListener(ContextAwareEventListener.using(listener -> {
            if(isCompleted()) {
                triggerCompleted();
                getProcessInstance().getKnowledgeRuntime().getProcessRuntime().removeEventListener(listener);
            }
        }));
    }

    @Override
    public void removeEventListeners() {
        super.removeEventListeners();
        getProcessInstance().removeEventListener(getActivationEventType(), this, true);
    }

    private String getActivationEventType() {
        return "RuleFlow-Milestone-" + getProcessInstance().getProcessId()
                + "-" + getMilestoneNode().getUniqueId();
    }
}
