/*
 * Copyright 2020 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.kie.kogito.codegen.process.persistence.proto;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.kie.kogito.codegen.metadata.PersistenceProtoFilesLabeler;

public abstract class AbstractProtoGenerator<T> implements ProtoGenerator<T> {

    public static final String GENERATED_PROTO_RES_PATH = "/classes/META-INF/resources/persistence/protobuf/";
    public static final String GENERATED_PROTO_PERSISTENCE_PATH = "/classes/persistence/";
    public static final String LISTING_FILE = "list.json";

    protected ObjectMapper mapper;

    public AbstractProtoGenerator() {
        mapper = new ObjectMapper();
    }

    /**
     * Flushes the generated proto files into the FileSystem
     */
    public final void writeFilesToFS(final String processId, final String targetDirectory, final Proto modelProto) throws IOException {
        String protoFileName = processId + ".proto";
        List<Path> protoFilePaths = new ArrayList<>();
        protoFilePaths.add(Paths.get(targetDirectory, GENERATED_PROTO_PERSISTENCE_PATH + protoFileName));
        protoFilePaths.add(Paths.get(targetDirectory, GENERATED_PROTO_RES_PATH + protoFileName));
        for (Path protoFilePath : protoFilePaths) {
            Files.createDirectories(protoFilePath.getParent());
            Files.write(protoFilePath, modelProto.toString().getBytes(StandardCharsets.UTF_8));
        }
    }

    /**
     * Reads from the persistence directory and generates a listing file ({@link #LISTING_FILE}) from its content
     *
     * @param targetDirectory target directory generated by Maven
     * @throws IOException if something wrong occurs during I/O
     */
    public final void generateProtoListing(final String targetDirectory) throws IOException {
        final Path persistencePath = Paths.get(targetDirectory, GENERATED_PROTO_RES_PATH);
        final List<String> fileList = new ArrayList<>();
        if (Files.isDirectory(persistencePath)) {
            for (final File protoFile : persistencePath.toFile().listFiles((dir, name) ->
                                                                                   name.toLowerCase().endsWith(PersistenceProtoFilesLabeler.PROTO_FILE_EXT))) {
                fileList.add(protoFile.getName());
            }
        }
        Files.write(Paths.get(persistencePath.toString(), LISTING_FILE), mapper.writeValueAsString(fileList).getBytes(StandardCharsets.UTF_8));
    }
}
