/*
 * Copyright 2022 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kie.kogito.event;

import java.io.IOException;
import java.util.function.Function;

import io.cloudevents.CloudEvent;
import io.cloudevents.CloudEventData;

/**
 * This interface is one of the extension point for users to incorporate more event formats.
 * It is responsible for converting Cloud events generated by Kogito into the format expected by the external service.
 * Default implementation uses Jackson.
 * 
 * @param <R> The expected output type that will be consumed by the external service
 */
public interface CloudEventMarshaller<R> {
    /**
     * Convert cloud event into the type expected by the external service
     * 
     * @param event Cloud event to be converted
     * @return object to be sent to the external service
     * @throws IOException if there is a conversion problem. This method must NOT report event formatting issues through a runtime exception, it must use IOException. This way the caller
     *         can differentiate between unexpected issues and event formatting ones.
     */
    R marshall(CloudEvent event) throws IOException;

    /**
     * Convert Kogito business object into a CloudEventData for marshaling
     * 
     * @param <T> the Kogito business object type
     * @return A CloudEventData that will be marshaled.
     */
    <T> Function<T, CloudEventData> cloudEventDataFactory();
}
