/*
 * Decompiled with CFR 0.152.
 */
package org.killbill.commons.jdbi.guice;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.health.HealthCheckRegistry;
import com.google.common.annotations.VisibleForTesting;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.mchange.v2.c3p0.ComboPooledDataSource;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import com.zaxxer.hikari.pool.PoolInitializationException;
import java.io.IOException;
import java.net.URI;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.concurrent.TimeUnit;
import javax.sql.DataSource;
import org.killbill.commons.embeddeddb.EmbeddedDB;
import org.killbill.commons.jdbi.guice.DaoConfig;
import org.skife.config.TimeSpan;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataSourceProvider
implements Provider<DataSource> {
    protected final DaoConfig config;
    protected final String poolName;
    protected final boolean useMariaDB;
    protected final EmbeddedDB embeddedDB;
    private DatabaseType databaseType;
    private String dataSourceClassName;
    private String driverClassName;
    private Object metricRegistry;
    private Object healthCheckRegistry;

    @Inject
    public DataSourceProvider(DaoConfig config) {
        this(config, null);
    }

    public DataSourceProvider(DaoConfig config, String poolName) {
        this(config, poolName, true);
    }

    public DataSourceProvider(DaoConfig config, EmbeddedDB embeddedDB, String poolName) {
        this(config, embeddedDB, poolName, true);
    }

    public DataSourceProvider(DaoConfig config, String poolName, boolean useMariaDB) {
        this(config, null, poolName, useMariaDB);
    }

    public DataSourceProvider(DaoConfig config, EmbeddedDB embeddedDB, String poolName, boolean useMariaDB) {
        this.config = config;
        this.poolName = poolName;
        this.useMariaDB = useMariaDB;
        this.embeddedDB = embeddedDB;
        this.parseJDBCUrl();
    }

    @Inject(optional=true)
    public void setMetricRegistry(MetricRegistry metricRegistry) {
        this.metricRegistry = metricRegistry;
    }

    @Inject(optional=true)
    public void setHealthCheckRegistry(HealthCheckRegistry healthCheckRegistry) {
        this.healthCheckRegistry = healthCheckRegistry;
    }

    public DataSource get() {
        DataSource dataSource = this.buildDataSource();
        if (this.embeddedDB != null) {
            this.embeddedDB.setDataSource(dataSource);
        }
        return dataSource;
    }

    private DataSource buildDataSource() {
        switch (this.config.getConnectionPoolingType()) {
            case C3P0: {
                this.loadDriver();
                return new C3P0DataSourceBuilder().buildDataSource();
            }
            case HIKARICP: {
                if (this.dataSourceClassName != null) {
                    this.loadDriver();
                }
                return new HikariDataSourceBuilder().buildDataSource();
            }
            case NONE: {
                if (this.embeddedDB == null) break;
                try {
                    this.embeddedDB.initialize();
                    this.embeddedDB.start();
                    return this.embeddedDB.getDataSource();
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
                catch (SQLException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        throw new IllegalArgumentException("DataSource " + (Object)((Object)this.config.getConnectionPoolingType()) + " unsupported");
    }

    static int toSeconds(TimeSpan timeSpan) {
        return DataSourceProvider.toSeconds(timeSpan.getPeriod(), timeSpan.getUnit());
    }

    static int toSeconds(long period, TimeUnit timeUnit) {
        return (int)TimeUnit.SECONDS.convert(period, timeUnit);
    }

    static int toMilliSeconds(TimeSpan timeSpan) {
        return DataSourceProvider.toMilliSeconds(timeSpan.getPeriod(), timeSpan.getUnit());
    }

    static int toMilliSeconds(long period, TimeUnit timeUnit) {
        return (int)TimeUnit.MILLISECONDS.convert(period, timeUnit);
    }

    private void parseJDBCUrl() {
        String schemeLocation;
        URI uri = URI.create(this.config.getJdbcUrl().substring(5));
        if (uri.getPath() != null) {
            schemeLocation = null;
        } else if (uri.getSchemeSpecificPart() != null) {
            String[] schemeParts = uri.getSchemeSpecificPart().split(":");
            schemeLocation = schemeParts[0];
        } else {
            schemeLocation = null;
        }
        this.dataSourceClassName = this.config.getDataSourceClassName();
        this.driverClassName = this.config.getDriverClassName();
        if ("mysql".equals(uri.getScheme())) {
            this.databaseType = DatabaseType.MYSQL;
            if (this.dataSourceClassName == null) {
                this.dataSourceClassName = this.useMariaDB ? "org.killbill.commons.embeddeddb.mysql.KillBillMariaDbDataSource" : "com.mysql.jdbc.jdbc2.optional.MysqlDataSource";
            }
            if (this.driverClassName == null) {
                this.driverClassName = this.useMariaDB ? "org.mariadb.jdbc.Driver" : "com.mysql.jdbc.Driver";
            }
        } else if ("h2".equals(uri.getScheme()) && ("mem".equals(schemeLocation) || "file".equals(schemeLocation))) {
            this.databaseType = DatabaseType.H2;
            if (this.dataSourceClassName == null) {
                this.dataSourceClassName = "org.h2.jdbcx.JdbcDataSource";
            }
            if (this.driverClassName == null) {
                this.driverClassName = "org.h2.Driver";
            }
        } else if ("postgresql".equals(uri.getScheme())) {
            this.databaseType = DatabaseType.POSTGRESQL;
            if (this.dataSourceClassName == null) {
                this.dataSourceClassName = "org.postgresql.ds.PGSimpleDataSource";
            }
            if (this.driverClassName == null) {
                this.driverClassName = "org.postgresql.Driver";
            }
        } else {
            this.databaseType = DatabaseType.GENERIC;
        }
    }

    private void loadDriver() {
        if (this.driverClassName != null) {
            try {
                Class.forName(this.driverClassName).newInstance();
            }
            catch (Exception e) {
                throw new IllegalStateException(e);
            }
        }
    }

    private class C3P0DataSourceBuilder {
        private C3P0DataSourceBuilder() {
        }

        DataSource buildDataSource() {
            ComboPooledDataSource cpds = new ComboPooledDataSource();
            cpds.setJdbcUrl(DataSourceProvider.this.config.getJdbcUrl());
            cpds.setUser(DataSourceProvider.this.config.getUsername());
            cpds.setPassword(DataSourceProvider.this.config.getPassword());
            cpds.setMinPoolSize(DataSourceProvider.this.config.getMinIdle());
            cpds.setMaxPoolSize(DataSourceProvider.this.config.getMaxActive());
            cpds.setCheckoutTimeout(DataSourceProvider.toMilliSeconds(DataSourceProvider.this.config.getConnectionTimeout()));
            cpds.setMaxIdleTime(DataSourceProvider.toSeconds(DataSourceProvider.this.config.getIdleMaxAge()));
            cpds.setMaxConnectionAge(DataSourceProvider.toSeconds(DataSourceProvider.this.config.getMaxConnectionAge()));
            cpds.setIdleConnectionTestPeriod(DataSourceProvider.toSeconds(DataSourceProvider.this.config.getIdleConnectionTestPeriod()));
            cpds.setMaxStatementsPerConnection(DataSourceProvider.this.config.getPreparedStatementsCacheSize());
            cpds.setDataSourceName(DataSourceProvider.this.poolName);
            String initSQL = DataSourceProvider.this.config.getConnectionInitSql();
            if (initSQL != null && !initSQL.isEmpty()) {
                HashMap<String, String> extensions = new HashMap<String, String>(4);
                extensions.put("initSql", initSQL);
                cpds.setExtensions(extensions);
                cpds.setConnectionCustomizerClassName("com.mchange.v2.c3p0.example.InitSqlConnectionCustomizer");
            }
            return cpds;
        }
    }

    private class HikariDataSourceBuilder {
        private final Logger logger = LoggerFactory.getLogger(HikariDataSourceBuilder.class);

        private HikariDataSourceBuilder() {
        }

        DataSource buildDataSource() {
            HikariConfig hikariConfig = new HikariConfig();
            hikariConfig.setUsername(DataSourceProvider.this.config.getUsername());
            hikariConfig.setPassword(DataSourceProvider.this.config.getPassword());
            hikariConfig.setMaximumPoolSize(DataSourceProvider.this.config.getMaxActive());
            hikariConfig.setLeakDetectionThreshold(DataSourceProvider.this.config.getLeakDetectionThreshold().getMillis());
            hikariConfig.setMinimumIdle(DataSourceProvider.this.config.getMinIdle());
            hikariConfig.setConnectionTimeout((long)DataSourceProvider.toMilliSeconds(DataSourceProvider.this.config.getConnectionTimeout()));
            hikariConfig.setIdleTimeout((long)DataSourceProvider.toMilliSeconds(DataSourceProvider.this.config.getIdleMaxAge()));
            hikariConfig.setMaxLifetime((long)DataSourceProvider.toMilliSeconds(DataSourceProvider.this.config.getMaxConnectionAge()));
            String initSQL = DataSourceProvider.this.config.getConnectionInitSql();
            if (initSQL != null && !initSQL.isEmpty()) {
                hikariConfig.setConnectionInitSql(initSQL);
            }
            hikariConfig.setInitializationFailFast(DataSourceProvider.this.config.isInitializationFailFast());
            hikariConfig.setTransactionIsolation(DataSourceProvider.this.config.getTransactionIsolationLevel());
            hikariConfig.setReadOnly(DataSourceProvider.this.config.isReadOnly());
            hikariConfig.setRegisterMbeans(true);
            if (DataSourceProvider.this.metricRegistry != null) {
                hikariConfig.setMetricRegistry(DataSourceProvider.this.metricRegistry);
            }
            if (DataSourceProvider.this.healthCheckRegistry != null) {
                hikariConfig.addHealthCheckProperty("connectivityCheckTimeoutMs", String.valueOf(DataSourceProvider.toMilliSeconds(DataSourceProvider.this.config.getHealthCheckConnectionTimeout())));
                hikariConfig.addHealthCheckProperty("expected99thPercentileMs", String.valueOf(DataSourceProvider.toMilliSeconds(DataSourceProvider.this.config.getHealthCheckExpected99thPercentile())));
                hikariConfig.setHealthCheckRegistry(DataSourceProvider.this.healthCheckRegistry);
            }
            if (DataSourceProvider.this.poolName != null) {
                hikariConfig.setPoolName(DataSourceProvider.this.poolName);
            }
            hikariConfig.addDataSourceProperty("url", (Object)DataSourceProvider.this.config.getJdbcUrl());
            hikariConfig.addDataSourceProperty("user", (Object)DataSourceProvider.this.config.getUsername());
            hikariConfig.addDataSourceProperty("password", (Object)DataSourceProvider.this.config.getPassword());
            if (DatabaseType.MYSQL.equals((Object)DataSourceProvider.this.databaseType)) {
                hikariConfig.addDataSourceProperty("cachePrepStmts", (Object)DataSourceProvider.this.config.isPreparedStatementsCacheEnabled());
                hikariConfig.addDataSourceProperty("prepStmtCacheSize", (Object)DataSourceProvider.this.config.getPreparedStatementsCacheSize());
                hikariConfig.addDataSourceProperty("prepStmtCacheSqlLimit", (Object)DataSourceProvider.this.config.getPreparedStatementsCacheSqlLimit());
                if (Float.valueOf(DataSourceProvider.this.config.getMySQLServerVersion()).compareTo(Float.valueOf("5.1")) >= 0) {
                    hikariConfig.addDataSourceProperty("useServerPrepStmts", (Object)DataSourceProvider.this.config.isServerSidePreparedStatementsEnabled());
                }
            }
            if (DataSourceProvider.this.dataSourceClassName != null) {
                hikariConfig.setDataSourceClassName(DataSourceProvider.this.dataSourceClassName);
            } else {
                hikariConfig.setJdbcUrl(DataSourceProvider.this.config.getJdbcUrl());
                if (DataSourceProvider.this.driverClassName != null) {
                    hikariConfig.setDriverClassName(DataSourceProvider.this.driverClassName);
                }
            }
            try {
                return new HikariDataSource(hikariConfig);
            }
            catch (PoolInitializationException e) {
                this.logger.error("Unable to initialize the database pool", (Throwable)e);
                throw e;
            }
        }
    }

    @VisibleForTesting
    static enum DatabaseType {
        GENERIC,
        MYSQL,
        H2,
        POSTGRESQL;

    }
}

