/*
 * Decompiled with CFR 0.152.
 */
package org.killbill.commons.jdbi.guice;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import com.zaxxer.hikari.metrics.MetricsTrackerFactory;
import com.zaxxer.hikari.pool.HikariPool;
import java.io.IOException;
import java.net.URI;
import java.sql.SQLException;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Provider;
import javax.sql.DataSource;
import org.killbill.commons.embeddeddb.EmbeddedDB;
import org.killbill.commons.health.api.HealthCheckRegistry;
import org.killbill.commons.jdbi.guice.DaoConfig;
import org.killbill.commons.jdbi.hikari.KillBillHealthChecker;
import org.killbill.commons.jdbi.hikari.KillBillMetricsTrackerFactory;
import org.killbill.commons.metrics.api.MetricRegistry;
import org.killbill.commons.utils.annotation.VisibleForTesting;
import org.skife.config.TimeSpan;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataSourceProvider
implements Provider<DataSource> {
    protected final DaoConfig config;
    protected final String poolName;
    protected final boolean useMariaDB;
    protected final EmbeddedDB embeddedDB;
    private DatabaseType databaseType;
    private String dataSourceClassName;
    private String driverClassName;
    private MetricRegistry metricRegistry;
    private HealthCheckRegistry healthCheckRegistry;

    @Inject
    public DataSourceProvider(DaoConfig config) {
        this(config, null);
    }

    public DataSourceProvider(DaoConfig config, String poolName) {
        this(config, poolName, true);
    }

    public DataSourceProvider(DaoConfig config, EmbeddedDB embeddedDB, String poolName) {
        this(config, embeddedDB, poolName, true);
    }

    public DataSourceProvider(DaoConfig config, String poolName, boolean useMariaDB) {
        this(config, null, poolName, useMariaDB);
    }

    public DataSourceProvider(DaoConfig config, EmbeddedDB embeddedDB, String poolName, boolean useMariaDB) {
        this.config = config;
        this.poolName = poolName;
        this.useMariaDB = useMariaDB;
        this.embeddedDB = embeddedDB;
        this.parseJDBCUrl();
    }

    @Inject
    public void setMetricsRegistry(@Nullable MetricRegistry metricRegistry) {
        this.metricRegistry = metricRegistry;
    }

    @Inject
    public void setHealthCheckRegistry(@Nullable HealthCheckRegistry healthCheckRegistry) {
        this.healthCheckRegistry = healthCheckRegistry;
    }

    public DataSource get() {
        DataSource dataSource = this.buildDataSource();
        if (this.embeddedDB != null) {
            this.embeddedDB.setDataSource(dataSource);
        }
        return dataSource;
    }

    private DataSource buildDataSource() {
        switch (this.config.getConnectionPoolingType()) {
            case HIKARICP: {
                if (this.dataSourceClassName != null) {
                    this.loadDriver();
                }
                return new HikariDataSourceBuilder().buildDataSource();
            }
            case NONE: {
                if (this.embeddedDB == null) break;
                try {
                    this.embeddedDB.initialize();
                    this.embeddedDB.start();
                    return this.embeddedDB.getDataSource();
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
                catch (SQLException e) {
                    throw new RuntimeException(e);
                }
            }
        }
        throw new IllegalArgumentException("DataSource " + this.config.getConnectionPoolingType() + " unsupported");
    }

    static int toSeconds(TimeSpan timeSpan) {
        return DataSourceProvider.toSeconds(timeSpan.getPeriod(), timeSpan.getUnit());
    }

    static int toSeconds(long period, TimeUnit timeUnit) {
        return (int)TimeUnit.SECONDS.convert(period, timeUnit);
    }

    static int toMilliSeconds(TimeSpan timeSpan) {
        return DataSourceProvider.toMilliSeconds(timeSpan.getPeriod(), timeSpan.getUnit());
    }

    static int toMilliSeconds(long period, TimeUnit timeUnit) {
        return (int)TimeUnit.MILLISECONDS.convert(period, timeUnit);
    }

    private void parseJDBCUrl() {
        String schemeLocation;
        URI uri = URI.create(this.config.getJdbcUrl().substring(5));
        if (uri.getPath() != null) {
            schemeLocation = null;
        } else if (uri.getSchemeSpecificPart() != null) {
            String[] schemeParts = uri.getSchemeSpecificPart().split(":");
            schemeLocation = schemeParts[0];
        } else {
            schemeLocation = null;
        }
        this.dataSourceClassName = this.config.getDataSourceClassName();
        this.driverClassName = this.config.getDriverClassName();
        if ("mysql".equals(uri.getScheme())) {
            this.databaseType = DatabaseType.MYSQL;
            if (this.dataSourceClassName == null) {
                this.dataSourceClassName = this.useMariaDB ? "org.killbill.commons.embeddeddb.mysql.KillBillMariaDbDataSource" : "com.mysql.jdbc.jdbc2.optional.MysqlDataSource";
            }
            if (this.driverClassName == null) {
                this.driverClassName = this.useMariaDB ? "org.mariadb.jdbc.Driver" : "com.mysql.cj.jdbc.Driver";
            }
        } else if ("h2".equals(uri.getScheme()) && ("mem".equals(schemeLocation) || "file".equals(schemeLocation))) {
            this.databaseType = DatabaseType.H2;
            if (this.dataSourceClassName == null) {
                this.dataSourceClassName = "org.h2.jdbcx.JdbcDataSource";
            }
            if (this.driverClassName == null) {
                this.driverClassName = "org.h2.Driver";
            }
        } else if ("postgresql".equals(uri.getScheme())) {
            this.databaseType = DatabaseType.POSTGRESQL;
            if (this.dataSourceClassName == null) {
                this.dataSourceClassName = "org.postgresql.ds.PGSimpleDataSource";
            }
            if (this.driverClassName == null) {
                this.driverClassName = "org.postgresql.Driver";
            }
        } else {
            this.databaseType = DatabaseType.GENERIC;
        }
    }

    private void loadDriver() {
        if (this.driverClassName != null) {
            try {
                Class.forName(this.driverClassName).newInstance();
            }
            catch (Exception e) {
                throw new IllegalStateException(e);
            }
        }
    }

    private class HikariDataSourceBuilder {
        private final Logger logger = LoggerFactory.getLogger(HikariDataSourceBuilder.class);

        private HikariDataSourceBuilder() {
        }

        DataSource buildDataSource() {
            HikariConfig hikariConfig = new HikariConfig();
            hikariConfig.setUsername(DataSourceProvider.this.config.getUsername());
            hikariConfig.setPassword(DataSourceProvider.this.config.getPassword());
            hikariConfig.setMaximumPoolSize(DataSourceProvider.this.config.getMaxActive());
            hikariConfig.setLeakDetectionThreshold(DataSourceProvider.this.config.getLeakDetectionThreshold().getMillis());
            hikariConfig.setMinimumIdle(DataSourceProvider.this.config.getMinIdle());
            hikariConfig.setConnectionTimeout((long)DataSourceProvider.toMilliSeconds(DataSourceProvider.this.config.getConnectionTimeout()));
            hikariConfig.setIdleTimeout((long)DataSourceProvider.toMilliSeconds(DataSourceProvider.this.config.getIdleMaxAge()));
            hikariConfig.setMaxLifetime((long)DataSourceProvider.toMilliSeconds(DataSourceProvider.this.config.getMaxConnectionAge()));
            String initSQL = DataSourceProvider.this.config.getConnectionInitSql();
            if (initSQL != null && !initSQL.isEmpty()) {
                hikariConfig.setConnectionInitSql(initSQL);
            }
            hikariConfig.setInitializationFailTimeout(DataSourceProvider.this.config.isInitializationFailFast() ? 1L : -1L);
            hikariConfig.setTransactionIsolation(DataSourceProvider.this.config.getTransactionIsolationLevel());
            hikariConfig.setReadOnly(DataSourceProvider.this.config.isReadOnly());
            hikariConfig.setRegisterMbeans(true);
            if (DataSourceProvider.this.metricRegistry != null) {
                hikariConfig.setMetricsTrackerFactory((MetricsTrackerFactory)new KillBillMetricsTrackerFactory(DataSourceProvider.this.metricRegistry));
            }
            if (DataSourceProvider.this.poolName != null) {
                hikariConfig.setPoolName(DataSourceProvider.this.poolName);
            }
            hikariConfig.addDataSourceProperty("url", (Object)DataSourceProvider.this.config.getJdbcUrl());
            hikariConfig.addDataSourceProperty("user", (Object)DataSourceProvider.this.config.getUsername());
            hikariConfig.addDataSourceProperty("password", (Object)DataSourceProvider.this.config.getPassword());
            if (DatabaseType.MYSQL.equals((Object)DataSourceProvider.this.databaseType)) {
                hikariConfig.addDataSourceProperty("cachePrepStmts", (Object)DataSourceProvider.this.config.isPreparedStatementsCacheEnabled());
                hikariConfig.addDataSourceProperty("prepStmtCacheSize", (Object)DataSourceProvider.this.config.getPreparedStatementsCacheSize());
                hikariConfig.addDataSourceProperty("prepStmtCacheSqlLimit", (Object)DataSourceProvider.this.config.getPreparedStatementsCacheSqlLimit());
                if (Float.valueOf(DataSourceProvider.this.config.getMySQLServerVersion()).compareTo(Float.valueOf("5.1")) >= 0) {
                    hikariConfig.addDataSourceProperty("useServerPrepStmts", (Object)DataSourceProvider.this.config.isServerSidePreparedStatementsEnabled());
                }
            }
            if (DataSourceProvider.this.dataSourceClassName != null) {
                hikariConfig.setDataSourceClassName(DataSourceProvider.this.dataSourceClassName);
            } else {
                hikariConfig.setJdbcUrl(DataSourceProvider.this.config.getJdbcUrl());
                if (DataSourceProvider.this.driverClassName != null) {
                    hikariConfig.setDriverClassName(DataSourceProvider.this.driverClassName);
                }
            }
            try {
                HikariDataSource hikariDataSource = new HikariDataSource(hikariConfig);
                if (DataSourceProvider.this.healthCheckRegistry != null) {
                    KillBillHealthChecker.registerHealthChecks((DataSource)hikariDataSource, hikariConfig, DataSourceProvider.this.healthCheckRegistry);
                }
                return hikariDataSource;
            }
            catch (HikariPool.PoolInitializationException e) {
                this.logger.error("Unable to initialize the database pool", (Throwable)e);
                throw e;
            }
        }
    }

    @VisibleForTesting
    static enum DatabaseType {
        GENERIC,
        MYSQL,
        H2,
        POSTGRESQL;

    }
}

