/*
 * Copyright (c) 2012, Francis Galiegue <fgaliegue@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the Lesser GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Lesser GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.eel.kitchen.jsonschema.validator;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.ImmutableList;
import org.eel.kitchen.jsonschema.main.JsonSchemaFactory;
import org.eel.kitchen.jsonschema.main.SchemaContainer;
import org.eel.kitchen.jsonschema.main.ValidationContext;
import org.eel.kitchen.jsonschema.main.ValidationReport;
import org.eel.kitchen.jsonschema.ref.JsonPointer;

import java.util.Collections;
import java.util.List;

/**
 * Validator called for array instance children
 *
 * <p>Array children must obey one schema, depending on the values of the
 * {@code items} and {@code additionalItems} schema keywords. No {@code items}
 * is equivalent to an empty array.</p>
 *
 * <p>Array indices start at 0. For a given index:</p>
 *
 * <ul>
 *     <li>if there is a schema at this index in {@code items}, there is a
 *     match;</li>
 *     <li>otherwise, {@code additionalItems} is considered (if {@code true} or
 *     nonexistent, then an empty schema).</li>
 * </ul>
 *
 * <p>Its {@link #next()} method should <b>never</b> be called (it throws an
 * {@link IllegalStateException} if it is).</p>
 */
public final class ArrayJsonValidator
    extends JsonValidator
{
    private final JsonNode additionalItems;

    private final List<JsonNode> items;

    ArrayJsonValidator(final JsonSchemaFactory factory,
        final SchemaNode schemaNode)
    {
        super(factory, schemaNode);

        final JsonNode schema = schemaNode.getNode();

        JsonNode node;

        node = schema.path("items");

        if (node.isObject()) {
            additionalItems = node;
            items = Collections.emptyList();
            return;
        }

        // We know that if "items" is not an object, it is an array
        items = ImmutableList.copyOf(node);

        node = schema.path("additionalItems");

        additionalItems = node.isObject() ? node : EMPTY_SCHEMA;
    }

    @Override
    public boolean validate(final ValidationContext context,
        final ValidationReport report, final JsonNode instance)
    {
        final SchemaContainer container = context.getContainer();
        final JsonPointer pwd = report.getPath();

        SchemaNode subSchema;
        JsonNode element;
        JsonValidator validator;

        for (int i = 0; i < instance.size(); i++) {
            subSchema = getSchemaNode(container, i);
            element = instance.get(i);
            report.setPath(pwd.append(i));
            validator = new RefResolverJsonValidator(factory, subSchema);
            while (validator.validate(context, report, element))
                validator = validator.next();
            context.setContainer(container);
        }

        report.setPath(pwd);
        return false;
    }

    @Override
    public JsonValidator next()
    {
        throw new IllegalStateException("I should not have been called");
    }

    private SchemaNode getSchemaNode(final SchemaContainer container,
        final int index)
    {
        final JsonNode node = index >= items.size() ? additionalItems
            : items.get(index);

        return new SchemaNode(container, node);
    }
}
