/**
 * Copyright 2010-2014 The Kuali Foundation
 *
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.opensource.org/licenses/ecl2.php
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kuali.common.util.base;

import static java.lang.String.format;

/**
 * <p>
 * Utility methods for creating {@code IllegalStateException's} and {@code IllegaArgumentException's} with richly formatted error messages.
 * </p>
 * 
 * Example usage:
 * 
 * <pre>
 * throw Exceptions.illegalArgument(&quot;port must be &gt;= %s and &lt;= %s&quot;, 0, 65535);
 * </pre>
 */
public class Exceptions {

	public static IllegalStateException illegalState(Throwable cause) {
		return new IllegalStateException(cause);
	}

	public static IllegalStateException illegalState(String msg) {
		return new IllegalStateException(msg);
	}

	public static IllegalStateException illegalState(String msg, Object... args) {
		return new IllegalStateException(formatted(msg, args));
	}

	public static IllegalStateException illegalState(Throwable cause, String msg, Object... args) {
		return new IllegalStateException(formatted(msg, args), cause);
	}

	public static IllegalArgumentException illegalArgument(Throwable cause) {
		return new IllegalArgumentException(cause);
	}

	public static IllegalArgumentException illegalArgument(String msg) {
		return new IllegalArgumentException(msg);
	}

	public static IllegalArgumentException illegalArgument(String msg, Object... args) {
		return new IllegalArgumentException(formatted(msg, args));
	}

	public static IllegalArgumentException illegalArgument(Throwable cause, String msg, Object... args) {
		return new IllegalArgumentException(formatted(msg, args), cause);
	}

	protected static String formatted(String msg, Object... args) {
		return (args == null || args.length == 0) ? msg : format(msg, args);
	}

}
