/**
 * Copyright 2010-2014 The Kuali Foundation
 *
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.opensource.org/licenses/ecl2.php
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kuali.common.util.resolver;

import java.util.Properties;

import org.kuali.common.util.Assert;
import org.kuali.common.util.PropertyUtils;
import org.springframework.util.PropertyPlaceholderHelper;

public class PropertiesValueResolver implements ValueResolver {

	private static final String PREFIX = "${";
	private static final String SUFFIX = "}";
	private static final String SEPARATOR = ":";

	public static final boolean DEFAULT_IGNORE_UNRESOLVABLE = false;

	private static final PropertyPlaceholderHelper DEFAULT_HELPER = new PropertyPlaceholderHelper(PREFIX, SUFFIX, SEPARATOR, DEFAULT_IGNORE_UNRESOLVABLE);

	private final Properties properties;
	private final PropertyPlaceholderHelper helper;

	public PropertiesValueResolver() {
		this(PropertyUtils.EMPTY);
	}

	public PropertiesValueResolver(Properties properties) {
		this(properties, DEFAULT_HELPER);
	}

	public PropertiesValueResolver(Properties properties, boolean ignoreUnresolvable) {
		this(properties, new PropertyPlaceholderHelper(PREFIX, SUFFIX, SEPARATOR, ignoreUnresolvable));
	}

	public PropertiesValueResolver(Properties properties, PropertyPlaceholderHelper helper) {
		Assert.noNulls(properties, helper);
		this.properties = PropertyUtils.toImmutable(properties);
		this.helper = helper;
	}

	@Override
	public String resolve(String value) {
		return helper.replacePlaceholders(value, properties);
	}

	public Properties getProperties() {
		return properties;
	}

	public PropertyPlaceholderHelper getHelper() {
		return helper;
	}

}
