/**
 * Copyright 2010-2014 The Kuali Foundation
 *
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.opensource.org/licenses/ecl2.php
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kuali.common.util.wait;

import static com.google.common.base.Preconditions.checkState;
import static java.lang.System.currentTimeMillis;
import static org.apache.commons.lang3.StringUtils.leftPad;
import static org.kuali.common.util.base.Threads.sleep;
import static org.kuali.common.util.log.Loggers.newLogger;

import org.kuali.common.util.FormatUtils;
import org.kuali.common.util.condition.Condition;
import org.slf4j.Logger;

public class DefaultWaitService implements WaitService {

	private static final Logger logger = newLogger();

	@Override
	public WaitResult wait(WaitContext context, Condition condition) {
		long start = currentTimeMillis();
		long timeout = start + context.getTimeoutMillis();
		sleep(context.getInitialPauseMillis());
		while (!condition.isTrue()) {
			long now = currentTimeMillis();
			checkState(now <= timeout, "Timed out waiting");
			String elapsed = leftPad(FormatUtils.getTime(now - start), 7, " ");
			String timeoutString = leftPad(FormatUtils.getTime(timeout - now), 7, " ");
			logger.info("[elapsed: {}  timeout: {}]", elapsed, timeoutString);
			sleep(context.getSleepMillis());
		}
		return WaitResult.create(start, currentTimeMillis());
	}

}
