/**
 * Copyright 2010-2014 The Kuali Foundation
 *
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.opensource.org/licenses/ecl2.php
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kuali.common.util.execute.impl;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.kuali.common.util.base.Precondition.checkNotBlank;
import static org.kuali.common.util.base.Precondition.checkNotNull;
import static org.kuali.common.util.log.Loggers.newLogger;

import java.nio.charset.Charset;
import java.util.List;
import java.util.Locale;

import org.kuali.common.util.execute.Executable;
import org.kuali.common.util.file.CanonicalFile;
import org.slf4j.Logger;

import com.google.common.collect.ImmutableList;

public final class ShowEnvExec implements Executable {

	private static final Logger logger = newLogger();
	private static final Object[] EMPTY_OBJECT_ARRAY = {};

	public ShowEnvExec() {
		this(false);
	}

	public ShowEnvExec(boolean skip) {
		this.skip = skip;
	}

	private final boolean skip;

	@Override
	public void execute() {
		if (skip) {
			return;
		}
		List<Object> java = copyOf(System.getProperty("java.runtime.version"), System.getProperty("java.vm.name"), System.getProperty("java.vm.vendor"));
		List<Object> javaHome = copyOf(new CanonicalFile(System.getProperty("java.home")));
		List<Object> JAVA_HOME = getJavaHomeEnvironmentVariable();
		List<Object> other = copyOf(Locale.getDefault().toString(), Charset.defaultCharset().displayName());
		List<Object> os = copyOf(System.getProperty("os.name"), System.getProperty("os.version"), System.getProperty("os.arch"));
		info("Java version: %s, name: %s, vendor: %s", java);
		info("java.home: %s", javaHome);
		info("JAVA_HOME: %s", JAVA_HOME);
		info("Default locale: %s, platform encoding: %s", other);
		info("OS name: %s, version: %s, arch: %s", os);
	}

	private static List<Object> getJavaHomeEnvironmentVariable() {
		String javaHome = System.getenv("JAVA_HOME");
		if (isBlank(javaHome)) {
			return copyOf("-- Not set --");
		} else {
			return copyOf(new CanonicalFile(javaHome));
		}
	}

	private static List<Object> copyOf(Object... args) {
		return ImmutableList.copyOf(args);
	}

	private static void info(String msg, List<Object> args) {
		logger.info(format(checkNotBlank(msg, "msg"), checkNotNull(args, "args").toArray(EMPTY_OBJECT_ARRAY)));
	}

	public boolean isSkip() {
		return skip;
	}

}
